// Copyright (c) 2003 Raghavan Komondoor and Susan Horwitz

// This file is part of DUP.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software DUP and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include <clones.h>

Clone::Clone(SCM clonePair, WhichComponent which) {
    switch(which) {
      case first: {
	fileName = CHARS(STk_car(clonePair));
	funcName = CHARS(STk_cadr(clonePair));

	SCM currNodePairPtr = STk_car(STk_cddddr(clonePair));
	int currChild, currParent;
	while(currNodePairPtr != STk_nil) {
	    currParent = STk_integer_value_no_overflow
	      (STk_caaar(currNodePairPtr));
	    currChild =  STk_integer_value_no_overflow
	      (STk_cadaar(currNodePairPtr));
	    const char *kindString =
	      CHARS(STk_symbol2string(STk_car(STk_cddaar(currNodePairPtr))));
	    PDGEdge::EdgeKind kind =
	      (!strcmp(kindString, "control")) ? PDGEdge::Control :
	      ((!strcmp(kindString, "data")) ? PDGEdge::Data :
	       PDGEdge::Ignore);
	      
	    PDGEdge currEdge(currParent, currChild, kind);
	    edges.insert(currEdge);
	    givenEdgeList.push_back(currEdge);
	    nodes.insert(currParent);
	    
	    currNodePairPtr = STk_cdr(currNodePairPtr);
	}
	root_id = currParent;
	break;
      }
      case second: {
	fileName = CHARS(STk_caddr(clonePair));
	funcName = CHARS(STk_cadddr(clonePair));

	SCM currNodePairPtr = STk_car(STk_cddddr(clonePair));
	int currChild, currParent;
	while(currNodePairPtr != STk_nil) {
	    currParent = STk_integer_value_no_overflow
	      (STk_cadar(currNodePairPtr));
	    currChild =  STk_integer_value_no_overflow
	      (STk_caddar(currNodePairPtr));
	    const char *kindString =
	      CHARS(STk_symbol2string(STk_car(STk_cdddar(currNodePairPtr))));
	    PDGEdge::EdgeKind kind =
	      (!strcmp(kindString, "control")) ? PDGEdge::Control :
	      ((!strcmp(kindString, "data")) ? PDGEdge::Data :
	       PDGEdge::Ignore);
	      
	    PDGEdge currEdge(currParent, currChild, kind);
	    edges.insert(currEdge);
	    givenEdgeList.push_back(currEdge);
	    nodes.insert(currParent);

	    currNodePairPtr = STk_cdr(currNodePairPtr);
	}
	root_id = currParent;
      }
    }
}

void
Clone::WriteSTkMapping(ostream &out) const {
    if (finalEdgeList.empty()) return;
    
    // write a mapping with just one clone
    out << "(\"" << fileName << "\" \"" << funcName <<
      "\" " << -1 << ' ' << -1 << " (";

    // Iterate through the finalEdgeList of the clone and write out pairs of
    // edges.
    vector<PDGEdge>::const_iterator pPDGEdge = finalEdgeList.begin();
    while(pPDGEdge != finalEdgeList.end()) {
	out << "((" <<
	  pPDGEdge->parent << ' ' <<
	  pPDGEdge->child << ' ' <<
	  ((pPDGEdge->kind == PDGEdge::Control) ? "'control" :
	  ((pPDGEdge->kind == PDGEdge::Data) ? "'data" :
	   "'ignore")) << ") () () 'ignore) ";

	++pPDGEdge;
    }
    out << "))";
}

bool
Clone::Subsumes(const Clone &other) const {
    // Does *this subsume "other"?
    
    vector<int> difference(other.nodes.size());
    vector<int>::const_iterator endDiff =
      set_difference(other.nodes.begin(), other.nodes.end(),
		     nodes.begin(), nodes.end(),
		     difference.begin());
    return (endDiff <= difference.begin());
}

Clone::WhichClone
Clone::MoreControlPreds(const Clone &other) const {
    // Assumptions:
    //
    // "*this" and "other" are from the same function, and have the
    // same root. "edges" is sorted by the key (child, parent).
    //
    // Output: compare the control parents of the root of "*this" with the
    // control parents of the root of "other". If the first is a superset of
    // the second then return This, else if the second is a superset of the
    // first return Other, else return Neither.

    // Find the control parents of root_id
    vector<int> thisControlPreds;
    pair<set<PDGEdge>::const_iterator, set<PDGEdge>::const_iterator>
      pRange = equal_range(edges.begin(), edges.end(),
			   PDGEdge(0, root_id, PDGEdge::Ignore),
			   PDGEdge::SameChild());

    for(;pRange.first != pRange.second; ++pRange.first)
	if((*pRange.first).kind == PDGEdge::Control)
	    thisControlPreds.push_back((*pRange.first).parent);

    // Find the control parents of other.root_id
    vector<int> otherControlPreds;
    pRange = equal_range(other.edges.begin(), other.edges.end(),
			 PDGEdge(0, other.root_id, PDGEdge::Ignore),
			 PDGEdge::SameChild());
    
    for(;pRange.first != pRange.second; ++pRange.first)
	if((*pRange.first).kind == PDGEdge::Control)
	    otherControlPreds.push_back((*pRange.first).parent);

    if(includes(thisControlPreds.begin(), thisControlPreds.end(),
		otherControlPreds.begin(), otherControlPreds.end()))
	return This;
    else if (includes(otherControlPreds.begin(), otherControlPreds.end(),
		      thisControlPreds.begin(), thisControlPreds.end()))
	return Other;
    else
	return Neither;
}


int
Clone::IntersectionSize(const NodeSet &other) const {
    vector<int> intersection(other.size());
    vector<int>::const_iterator lastInt =
      set_intersection(nodes.begin(), nodes.end(),
		       other.begin(), other.end(),
		       intersection.begin());

    return lastInt - intersection.begin();
}

void
Clone::Intersect(NodeSet &other) const {
    set<int>::iterator currOther = other.begin();
    set<int>::const_iterator lastOther = other.end();
    set<int>::const_iterator currThis = nodes.begin();
    set<int>::const_iterator lastThis = nodes.end();

    // erase those elements from "other" that are not present in
    // "*this"
    while (currThis != lastThis && currOther != lastOther) 
	if (*currOther < *currThis) {
	    set<int>::iterator toErase = currOther;
	    ++currOther;
	    other.erase(toErase);
	}
	else if (*currThis < *currOther)
	    ++currThis;
	else {
	    ++currOther;
	    ++currThis;
	}
    while(currOther != lastOther) {
	set<int>::iterator toErase = currOther;
	++currOther;
	other.erase(toErase);
    }

    return;
}

void
Clone::reorderEdgeList(const Clone &otherClone, const Clone &canonicalOther) {
    // see comment in clones.h

    // make a copy of givenEdgeList
    for(int i = 0; i < canonicalOther.finalEdgeList.size(); ++i) {
	// now, find the PDGEdge in otherClone.givenEdgeList that is
	// == to the ith PDGEdge in canonicalOther.finalEdgeList.
	bool found = false;
	for(int j = 0; j < otherClone.givenEdgeList.size(); ++j) {
	    if (otherClone.givenEdgeList[j] ==
		canonicalOther.finalEdgeList[i]) {

		// fill in ith element of finalEdgeList
		finalEdgeList.push_back(givenEdgeList[j]);
		found = true;
		break;
	    }
	}
	assert(found);
    }
}


void
WriteSTkMapping(const Clone &clone1, const Clone &clone2, ostream &out) {

    out << "(\"" << clone1.fileName << "\" \"" << clone1.funcName <<
      "\" \"" << clone2.fileName << "\" \"" << clone2.funcName <<
      "\" (";

    // Iterate through the edges of both clones and write out pairs of
    // edges.
    vector<PDGEdge>::const_iterator p1Edge =  clone1.finalEdgeList.begin();
    vector<PDGEdge>::const_iterator p2Edge =  clone2.finalEdgeList.begin();
    
    while((p1Edge != clone1.finalEdgeList.end()) &&
	  (p2Edge != clone2.finalEdgeList.end())) {
	out << "((" <<
	  p1Edge->parent << ' ' <<
	  p1Edge->child << ' ' <<
	  ((p1Edge->kind == PDGEdge::Control) ?
	  "'control" :
	  ((p1Edge->kind == PDGEdge::Data) ? "'data" :
	   "'ignore")) <<   ") " <<
	  p2Edge->parent << ' ' <<
	  p2Edge->child << ' ' <<
	  ((p2Edge->kind == PDGEdge::Control) ?
	  "'control" :
	  ((p2Edge->kind == PDGEdge::Data) ?"'data" :
	   "'ignore")) <<  ") ";

	++p1Edge;
	++p2Edge;
    }
    out << "))";
}

bool
ClonePair::SharesClone(const ClonePair &otherPair,
		       WhichComponentPair &how) const {
    if (*firstClone == *otherPair.firstClone) {
	how.from = first;
	how.to = first;
	return true;
    }
    else if(*firstClone == *otherPair.secondClone) {
	how.from = first;
	how.to = second;
	return true;
    }
    else if(*secondClone == *otherPair.firstClone) {
	how.from = second;
	how.to = first;
	return true;
    }
    else if(*secondClone == *otherPair.secondClone) {
	how.from = second;
	how.to = second;
	return true;
    }
    else
	return false;

    // the clone of *this specified by how.from is identical to the clone of
    // otherPair specified by how.to.
}

OverlapOrientation
ClonePair::IsOverlapping(const Clone::NodeSet &otherFirst,
			 const Clone::NodeSet &otherSecond,
			 int minIntersectionSize) const {
    
    int firstCompIntSize =
      FirstClone()->IntersectionSize(otherFirst);
    int secondCompIntSize =
      SecondClone()->IntersectionSize(otherSecond);
    
    if(firstCompIntSize >= minIntersectionSize &&
       secondCompIntSize >= minIntersectionSize)
	// cp overlaps the current group with Straight orientation
	
	return Straight;
    else if (!strcmp(firstClone->FileName(), secondClone->FileName()) &&
	     !strcmp(firstClone->FuncName(), secondClone->FuncName())) {
	// both components are from the same function. Therefore we should
	// check for "crossed" overlap.
	
	int firstCompIntSize =
	  SecondClone()->IntersectionSize(otherFirst);
	int secondCompIntSize =
	  FirstClone()->IntersectionSize(otherSecond);

	if(firstCompIntSize >= minIntersectionSize &&
	   secondCompIntSize >= minIntersectionSize)
	    // cp overlaps the current group with Crossed orientation

	    return Crossed;
	else
	    return NoOverlap;
    }
    else
	return NoOverlap;
}
				 
CloneGroup::CloneGroup(SCM groupsAsList) {
    // Each element of groupsAsList is an sexpr representing a clone pair.

    SCM currGroupsPtr = groupsAsList;
    while(currGroupsPtr != STk_nil) {
	push_back(new Clone(STk_car(currGroupsPtr), ::first));
	if (STRINGP(STk_caddr(STk_car(currGroupsPtr))))
	    push_back(new Clone(STk_car(currGroupsPtr), ::second));

	currGroupsPtr = STk_cdr(currGroupsPtr);
    }
}

pair<CloneGroup::PosInGroup, CloneGroup::PosInGroup>
CloneGroup::addBothClones(ClonePair *p) {
    p->FirstClone()->initializeFinalEdgeList();
    push_back(p->FirstClone());
    p->SecondClone()->initializeFinalEdgeList();
    push_back(p->SecondClone());
    return pair<CloneGroup::PosInGroup, CloneGroup::PosInGroup>
	(size() - 2, size() - 1);
}

CloneGroup::PosInGroup
CloneGroup::addOneClone(ClonePair *p, WhichComponent whichCloneToAdd,
			PosInGroup otherClonePos) {
    // see comment in clones.h

    Clone *cloneToAdd;
    const Clone *otherClone;
    switch(whichCloneToAdd) {
    case first:
	cloneToAdd = p->FirstClone();
	otherClone = p->SecondClone();
	break;
    case second:
	cloneToAdd = p->SecondClone();
	otherClone = p->FirstClone();
    }
    const Clone *canonicalOther = (*this)[otherClonePos];
    cloneToAdd->reorderEdgeList(*otherClone, *canonicalOther);
    push_back(cloneToAdd);
    return size()-1;
}

void
CloneGroup::WriteSTkSexpr(ofstream &out) {
    out << '('; // start current group
    int numClonesInGroup = size();
    int numClonesWritten = 0;

    vector<const Clone *>::const_iterator pClone;
    for(pClone = begin();
	numClonesWritten < (numClonesInGroup/2)*2;
	numClonesWritten += 2)

	WriteSTkMapping(**pClone++, **pClone++, out);

    if(numClonesWritten < numClonesInGroup) {
	// write out last remaining clone
	(*pClone)->WriteSTkMapping(out);
	out << ')' << endl; // end of current group
    }
    else
	out << ")" << endl; // end of current group
}

bool
FunctionPair::operator==(const FunctionPair &otherFuncPair) const {
    return (!strcmp(firstFile, otherFuncPair.firstFile) &&
	    !strcmp(firstFunc, otherFuncPair.firstFunc) &&
	    !strcmp(secondFile, otherFuncPair.secondFile) &&
	    !strcmp(secondFunc, otherFuncPair.secondFunc));
}

size_t
FunctionPair::HashFunctionPair::operator()
    (const FunctionPair &functionPair) const {

    size_t retVal = 0;
    for(const char *p = functionPair.firstFile; *p; ++p) {
	retVal += *p;
	retVal %= INT_MAX;
    }
    for(const char *p = functionPair.firstFunc; *p; ++p) {
	retVal += *p;
	retVal %= INT_MAX;
    }
    for(const char *p = functionPair.secondFile; *p; ++p) {
	retVal += *p;
	retVal %= INT_MAX;
    }
    for(const char *p = functionPair.secondFunc; *p; ++p) {
	retVal += *p;
	retVal %= INT_MAX;
    }
    
    return retVal;
}

