// Copyright (c) 2003 Raghavan Komondoor and Susan Horwitz

// This file is part of DUP.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software DUP and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include <iostream.h>
#include <string.h>
#include <limits.h>
#include <stk.h>
#include <clones.h>
#include <map.h>
#include <hash_map.h>
#include <list.h>
#include <vector.h>
#include <algo.h>

/* The sole externally visible function in this file is FilterSubsumed. It is
   exported to Scheme, where it is known as filter-subsumed. */

ofstream binFile("bins.txt");

typedef vector<const ClonePair *> ClonePairsList;

template <class InputIterator, class OrientIterator, class InsertIterator>
void
RemoveWorseClonePairs_NoOvChk(InputIterator first,
			      InputIterator last,
			      OrientIterator oFirst,
			      OrientIterator oLast,
			      InsertIterator result) {
    // Assumptions:
    //
    // 1. The given clone pairs are in the range [first, last)
    //
    // 2. The first components of all given clone pairs are from the same
    // function f1, and the second components are all from the same function
    // f2. f1 may or may not be equal to f2.
    //
    // 3. All given clone pairs are assumed to already overlap each
    // other. This function therefore does not need to check for overlap
    // (hence the suffix "_NoOvChk" in the name). The overlapping orientation
    // information is provided in the range [oFirst, oLast).
    //
    // In case f1 \neq f2, then every entry in the range [oFirst, oLast)
    // should be equal to Straight. I.e., the first (second) component of each
    // clone pair overlaps the first (second) component of every other pair.
    //
    // In case f1 == f2, then *oFirst should be equal to Straight. If oCurr is
    // the ith iterator in the range [oFirst, oLast) then *oCurr is the
    // orientation of the ith clone pair in [first, last) (wrt to the first
    // clone pair).

    // Function output:
    //
    // Let (A1, B1), (A2, B2) be any two clone pairs among the given ones. If
    // the root of the jth component of the first pair is equal to the root of
    // the kth component of the second pair is equal to m, and the jth
    // component of the first pair overlaps the kth component of the second
    // pair, and the control parents of m in one pair are a subset of the
    // control parents of m in the other pair, then the pair that has the
    // subset is "worse than" the other pair. This function eliminates all
    // clone pairs that are worse than some clone pair, and returns the
    // remaining ones in "result". j, k range over {1, 2}.

    hash_map<const ClonePair *, OverlapOrientation, HashClonePairPtr>
      orientations; // store the orientation of each given
    // clone pair here.
    InputIterator pClonePair;
    OrientIterator pOrient;
    for(pClonePair = first, pOrient = oFirst;
	pClonePair != last;
	++pClonePair, ++pOrient)

	orientations[*pClonePair] = *pOrient;

    // For any clone pair cp let cp.f be the component of cp that overlaps
    // with the first component of *first. (recall that every clone pair has
    // two components). We are now going to construct the map
    // pairsCommonRootOvFirstComp. For any clone pair cpi,
    // pairsCommonRootOvFirstComp[cpi] will contain every clone pair cpj such
    // that the roots of cpi.f and cpj.f are identical
    // (pairsCommonRootOvFirstComp[cpj] is equal to
    // pairsCommonRootOvFirstComp[cpi]). 
    
    CompareFirstOvRoots compareObjFirst(orientations);
    map<const ClonePair *, list<const ClonePair *>, CompareFirstOvRoots>
      pairsCommonRootOvFirstComp(compareObjFirst);

    for(InputIterator pClonePair = first;
	pClonePair != last;
	++pClonePair) {

	const ClonePair *currClonePair = *pClonePair;

	pairsCommonRootOvFirstComp[currClonePair].
	  push_back(currClonePair);
    }

    CompareSecondOvRoots compareObjSecond(orientations);
    map<const ClonePair *, list<const ClonePair *>, CompareSecondOvRoots>
      pairsCommonRootOvSecondComp(compareObjSecond);

    // For each partition cp in pairsCommonRootOvFirstComp, decide which clone
    // pairs in pairsCommonRootOvFirstComp[cp] are worse than other clone
    // pairs in pairsCommonRootOvFirstComp[cp]. If a clone pair j is worse
    // than a clone pair i, then throw away j. Place every other clone pair in
    // pairsCommonRootOvSecondComp.
    
    for(map<const ClonePair *, list<const ClonePair *> >
	  ::iterator pPairsCommonRoot = pairsCommonRootOvFirstComp.begin();
	pPairsCommonRoot != pairsCommonRootOvFirstComp.end();
	++pPairsCommonRoot) {
	  
	list<const ClonePair *> &pairsWithCommonRoot =
	  (*pPairsCommonRoot).second;
	// compare each clone pair in pairsWithCommonRoot with every other
	// clone pair in it
	for(list<const ClonePair *>::iterator i =
	      pairsWithCommonRoot.begin();
	    i != pairsWithCommonRoot.end();
	    ++i) {

	    list<const ClonePair *>::iterator j;
	    bool iIsWorse = false;
	    for(j = i, ++j;
		(j != pairsWithCommonRoot.end()) && !iIsWorse;) {

		const Clone *firstDotf =
		  orientations[*i] == Straight ?
		  (*i)->FirstClone() :
		  (*i)->SecondClone();

		const Clone *secondDotf =
		  orientations[*j] == Straight ?
		  (*j)->FirstClone() :
		  (*j)->SecondClone();
		
		switch(firstDotf->
		       MoreControlPreds(*secondDotf)) {
		
		  case Clone::This: {
		      // j is worse than i, and can therefore be thrown away
		      list<const ClonePair *>::iterator t = j;
		      ++t;
		      pairsWithCommonRoot.erase(j);
		      j = t;
		      break;
		  }
		  case Clone::Other:
		    // i is worse
		    iIsWorse = true;
		    ++j;
		    break;
		  default: ++j; // neither is better than the other
		}
	    }

	    if(!iIsWorse)
		// i is not worse than anything in pairsWithCommonRoot
		pairsCommonRootOvSecondComp[*i].push_back(*i);
	}
    }

    // For each partition cp in pairsCommonRootOvSecondComp, decide which
    // clone pairs in pairsCommonRootOvSecondComp[cp] are worse than other
    // clone pairs in pairsCommonRootOvSecondComp[cp]. If a clone pair j is
    // worse than a clone pair i, then throw away j. Place every other clone
    // pair in "result".
    for(map<const ClonePair *, list<const ClonePair *> >
	  ::iterator pPairsCommonRoot = pairsCommonRootOvSecondComp.begin();
	pPairsCommonRoot != pairsCommonRootOvSecondComp.end();
	++pPairsCommonRoot) {
	  
	list<const ClonePair *> &pairsWithCommonRoot =
	  (*pPairsCommonRoot).second;
	// compare each clone pair in pairsWithCommonRoot with every other
	// clone pair in it
	for(list<const ClonePair *>::iterator i =
	      pairsWithCommonRoot.begin();
	    i != pairsWithCommonRoot.end();
	    ++i) {

	    list<const ClonePair *>::iterator j;
	    bool iIsWorse = false;
	    for(j = i, ++j;
		(j != pairsWithCommonRoot.end()) && !iIsWorse;) {

		const Clone *firstDots =
		  orientations[*i] == Straight ?
		  (*i)->SecondClone() :
		  (*i)->FirstClone();

		const Clone *secondDots =
		  orientations[*j] == Straight ?
		  (*j)->SecondClone() :
		  (*j)->FirstClone();
		
		switch(firstDots->
		       MoreControlPreds(*secondDots)) {

		  case Clone::This: {
		      // j is worse than i, and can therefore be thrown away
		      list<const ClonePair *>::iterator t = j;
		      ++t;
		      pairsWithCommonRoot.erase(j);
		      j = t;
		      break;
		  }
		  case Clone::Other:
		    // i is worse
		    iIsWorse = true;
		    ++j;
		    break;
		  default: ++j; // neither is better than the other
		}
	    }

	    if(!iIsWorse)
		// i is not worse than anything in pairsWithCommonRoot
		*result++ = *i;
	}
    }
}

class OverlappingClonePairs {
    // An object of this class represents a group of overlapping clone
    // pairs. The first (second) components of all clone pairs are assumed to
    // be from the same function.

    vector<const ClonePair *> clonePairs; // all clone pairs in this group
    vector<OverlapOrientation> orientation;
    // the size of this vector will at all times be equal to the size of the
    // vector clonePairs. orientation[0] will always be
    // Straight. orientation[i] will be Straight if the first (second)
    // component of clonePairs[i] overlaps the first (second) component of
    // clonePairs[0], and will be Crossed if the overlap is the other way
    // around. (If the two components of each clone pair are from two
    // different functions, then clearly all the orientation entries will be
    // equal to Straight.)
    
    Clone::NodeSet *firstCompNodes; // nodes that belong to the first
    // component of the first clone pair added to this object, and that belong
    // to *one of the two* components (the one that overlaps with the  first
    // component of the first clone pair) in each subsequent clone pair added.
    
    Clone::NodeSet *secondCompNodes; // likewise, for the second component of
    // the first clone pair.
    int largestCloneSize; // size of the largest clone in this group.
    int overlapPercentage; // see comment in class ClonePairsSameFuncPair

    // above fields are used to quickly determine if a new clone pair overlaps
    // *all* clone pairs already in the group.
    
    bool removedWorse;
    vector<const ClonePair *> betterClonePairs;
    
    void operator= (const OverlappingClonePairs &);
    OverlappingClonePairs(const OverlappingClonePairs &);

  public:
    OverlappingClonePairs
    (int overlapPercentage_a, const ClonePair *cp) :
	overlapPercentage(overlapPercentage_a), removedWorse(false) {

	clonePairs.push_back(cp);
	orientation.push_back(Straight);
	firstCompNodes = cp->FirstClone()->GetNodes();
	secondCompNodes = cp->SecondClone()->GetNodes();
	largestCloneSize = cp->FirstClone()->Size();
    }

    ~OverlappingClonePairs() {
	delete firstCompNodes;
	delete secondCompNodes;
    }
    
    bool InsertIfCompatible(const ClonePair *cp);
    // checks if cp overlaps *all* clone pairs in this group, and adds cp to
    // the group if overlap exists. 

    void RemoveWorseClonePairs() {
	RemoveWorseClonePairs_NoOvChk
	  (clonePairs.begin(),
	   clonePairs.end(),
	   orientation.begin(),
	   orientation.end(),
	   back_inserter(betterClonePairs));
	removedWorse = true;
    }
    
    typedef vector<const ClonePair *>::const_iterator const_iterator;

    const_iterator begin() {
	assert(removedWorse);
	return betterClonePairs.begin();
    }
    const_iterator end() { return betterClonePairs.end();}

    void PrintStats(ostream &out) {
	out << "Largest size = " << largestCloneSize << "\tInitial # = " <<
	  clonePairs.size() << "\tAfter removal # = " <<
	  betterClonePairs.size() << endl;
    }
};

class ClonePairsSameFuncPair {
    // This class implements a set of ClonePairs. The first components of all
    // clone pairs in the set must be from the same function, and the second
    // components of all clone pairs in the set must be from the same
    // function.
    
    // The set is stored in grouped form, with each group containing
    // overlapping clone pairs. Every clone pair belongs to a unique
    // group. Each group is represented by a OverlappingClonePairs object.

    // The ClonePairsSameFuncPair object is created by
    // RemoveSubsumedClonePairs (it places all non subsumed clone pairs in
    // this object).

  public:
    enum Kind {SameFunc, DiffFuncs};

  private:
    vector<OverlappingClonePairs *> clonePairGroups;
    // Each element in this vector is a group of overlapping clone pairs.

    // Assumption: every group in clonePairGroups must be non-empty. Otherwise
    // input_iterator does not work.
    
    Kind kind; // whether both components of each clone pair belong to the
    // same function/different functions.

    int overlapPercentage; // For any group of clone pairs, atleast
    // overlapPercentage of the nodes in each clone pair in the group should
    // be present in all other clone pairs in the same group. This is the
    // criteria that is used to decide which group a newly incoming clone pair
    // goes into.
    
  public:
    ClonePairsSameFuncPair(Kind kind_a, int overlapPercentage_a) :
	kind(kind_a), overlapPercentage(overlapPercentage_a) {}

    ~ClonePairsSameFuncPair() {
	for(vector<OverlappingClonePairs *>::const_iterator
	      pGroup = clonePairGroups.begin();
	    pGroup != clonePairGroups.end();
	    ++pGroup)

	    delete *pGroup;
    }
    
    friend class insert_iterator {
	// an object of this class can be used to insert a clone pair into
	// the (correct group of) clonePairGroups.
	ClonePairsSameFuncPair &cPairs;
      public:
	insert_iterator(ClonePairsSameFuncPair &cPairs_a)
	    : cPairs(cPairs_a) {}

	insert_iterator& operator*() { return *this; }
	insert_iterator& operator++() { return *this; }
	insert_iterator& operator++(int) { return *this; }

	insert_iterator& operator=(const ClonePair *cp);
	//Insert cp into the first group in clonePairGroups with which it is
	//compatible (i.e., with which it overlaps) If it is incompatible with
	//all existing groups, create a new group, insert cp into that group,
	//and add the group to clonePairGroups.
    };

    friend class input_iterator {
	// an object of this class can be used to scan *all* clone pairs in
	// clonePairGroups without worrying about the grouping inside.
	
	ClonePairsSameFuncPair *cPairs;

	vector<OverlappingClonePairs *>::const_iterator pGroup; // current
	// group
	OverlappingClonePairs::const_iterator pClonePair; // current clone
	// pair within current group.

	static const OverlappingClonePairs::const_iterator DummyIt;

	// The following invariants should always be maintained:
	//
	// 1. As mentioned earlier, every group in clonePairGroups must be
	// non-empty. However clonePairGroups is allowed to be empty -- i.e.,
	// have no groups.
	//
	// 2. If an input_iterator is advanced past the last valid ClonePair
	// *, then pGroup should become equal to clonePairGroups.end() and
	// pClonePair should become equal to DummyIt. The same should hold if
	// an input_iterator is constructed as an End iterator.
	//
	// If an input_iterator is constructed as a Begin iterator and if
	// clonePairGroups is empty, then pGroup should be equal to
	// clonePairGroups.begin() (= clonePairGroups.end()) and pClonePair
	// should be equal to DummyIt. Therefore in this case the Begin
	// iterator is equal to the End iterator.
	//
	// 3. If an input_iterator is not equal to the End iterator then
	// pClonePair should point to a valid ClonePair * (i.e., it should be
	// not equal to end()).

	void AdvanceGroup() {
	    ++pGroup;
	    if(pGroup != cPairs->clonePairGroups.end())
		pClonePair = (*pGroup)->begin();
	    else
		pClonePair = DummyIt;
	}
	
      public:
	enum ConstructionKind {Begin, End};
	
	input_iterator(ClonePairsSameFuncPair *cPairs_a,
		       ConstructionKind kind_a);
	const ClonePair *operator*() const {return *pClonePair;}
	input_iterator& operator++();
	input_iterator operator++(int);
	bool operator== (const input_iterator &other) {
	    return
	      cPairs == other.cPairs &&
	      pGroup == other.pGroup &&
	      pClonePair == other.pClonePair;
	}
	bool operator!= (const input_iterator &other) {
	    return !(*this == other);
	}
    };

    input_iterator begin() {
	return input_iterator(this, input_iterator::Begin);
    }

    input_iterator end() {
	return input_iterator(this, input_iterator::End);
    }
    
    insert_iterator output() {
	return insert_iterator(*this);
    }

    void RemoveWorseClonePairs() {
	for(vector<OverlappingClonePairs *>::iterator
	      pGroup = clonePairGroups.begin();
	    pGroup != clonePairGroups.end();
	    ++pGroup)

	    (*pGroup)->RemoveWorseClonePairs();
    }

    void PrintStats(ostream &out) {
	for(vector<OverlappingClonePairs *>::const_iterator
	      pGroup = clonePairGroups.begin();
	    pGroup != clonePairGroups.end();
	    ++pGroup)

	    (*pGroup)->PrintStats(out);
    }
};

const OverlappingClonePairs::const_iterator
ClonePairsSameFuncPair::input_iterator::DummyIt = OverlappingClonePairs::const_iterator();

bool
OverlappingClonePairs::InsertIfCompatible(const ClonePair *cp) {
    assert(clonePairs.size() > 0);
	   
    int minIntersectionSize =
      int(double(max(largestCloneSize, cp->FirstClone()->Size())) *
	  double(overlapPercentage) / 100.0);
    
    switch(cp->IsOverlapping(*firstCompNodes, *secondCompNodes,
			     minIntersectionSize)) {

      case Straight:
	clonePairs.push_back(cp);
	orientation.push_back(Straight);
	cp->FirstClone()->Intersect(*firstCompNodes);
	cp->SecondClone()->Intersect(*secondCompNodes);
	largestCloneSize =
	  max(largestCloneSize, cp->FirstClone()->Size());
	
	return true;

      case Crossed:
	clonePairs.push_back(cp);
	orientation.push_back(Crossed);
	cp->SecondClone()->Intersect(*firstCompNodes);
	cp->FirstClone()->Intersect(*secondCompNodes);
	largestCloneSize =
	  max(largestCloneSize, cp->FirstClone()->Size());
	
	return true;

      default:
	// no overlap, therefore do not insert
	return false;
    }
}

ClonePairsSameFuncPair::insert_iterator &
ClonePairsSameFuncPair::insert_iterator::
operator=(const ClonePair *clonePair) {

    // Insert clonePair into the first group with which it is compatible
    for(vector<OverlappingClonePairs *>::iterator
	  pGroup = cPairs.clonePairGroups.begin();
	pGroup != cPairs.clonePairGroups.end();
	++pGroup)

	if((*pGroup)->InsertIfCompatible(clonePair))
	    return *this;

    // clonePair is incompatible with all existing groups. Need to create a
    // new group now.
    OverlappingClonePairs *newGroup =
      new OverlappingClonePairs(cPairs.overlapPercentage, clonePair);
    cPairs.clonePairGroups.push_back(newGroup);
    
    return *this;
}

ClonePairsSameFuncPair::input_iterator::
input_iterator(ClonePairsSameFuncPair *cPairs_a,
	       ConstructionKind kind_a)
    : cPairs(cPairs_a) {
    
    switch(kind_a) {
      case ClonePairsSameFuncPair::input_iterator::Begin:
	pGroup = cPairs->clonePairGroups.begin();
	if(pGroup != cPairs->clonePairGroups.end())
	    pClonePair = (*pGroup)->begin();
	else
	    pClonePair = DummyIt;
	break;
      case ClonePairsSameFuncPair::input_iterator::End:
	pGroup = cPairs->clonePairGroups.end();
	pClonePair = DummyIt;
    }
}

ClonePairsSameFuncPair::input_iterator &
ClonePairsSameFuncPair::input_iterator::
operator++() {
    ++pClonePair;
    if(pClonePair == (*pGroup)->end())
	AdvanceGroup();
    return *this;
}

ClonePairsSameFuncPair::input_iterator
ClonePairsSameFuncPair::input_iterator::
operator++(int) {
    input_iterator t = *this;
    ++(*this);
    return t;
}

template <class InputIterator, class InsertIterator>
void
RemoveSubsumedClonePairs(InputIterator first, InputIterator last,
			 InsertIterator result, bool sameFunc) {
    // The input to this function, given via "first" and "last", is a set
    // of ClonePair *'s. This function determines which of the given
    // ClonePairs are subsumed by others in the set, and places the unsubsumed
    // ClonePairs alone into "result". The function assumes that the first
    // components of all given clone pairs are from the same function, and the
    // second components of all given clone pairs are from the same function
    // (both functions could be the same).

    if(first == last)
	return;
    
    // put each clone pair into the bucket corresponding to its size
    map<int, ClonePairsList> sortedPartition;
    do {
	const ClonePair *currClonePair = *first;
	int currSize = currClonePair->Size();

	sortedPartition[currSize].push_back(currClonePair);

	++first;
    } while(first != last);

    // iterate over all sizes 
    for(map<int, ClonePairsList>::const_iterator
	  pCurrSize = sortedPartition.begin();
	pCurrSize != sortedPartition.end();
	++pCurrSize) {
	
	const ClonePairsList &clonePairs = pCurrSize->second;
	// all clone pairs of the current size
	    
	for(ClonePairsList::const_iterator
	      pCurrClonePair = clonePairs.begin();
	    pCurrClonePair != clonePairs.end();
	    ++pCurrClonePair) { // iterate over all clone pairs of the current
	    // size
		
	    const ClonePair *currClonePair = *pCurrClonePair;
	    bool subsumed = false;

	    // Go through all clone pairs in "clonePairs" that come after
	    // currClonePair and see if any of them subsume currClonePair.
	    ClonePairsList::const_iterator pNextClonePair = pCurrClonePair;
	    for(++pNextClonePair;
		!subsumed && (pNextClonePair != clonePairs.end());
		++pNextClonePair) 
		    
		if((*pNextClonePair)->
		   Subsumes(*currClonePair, sameFunc))
		    subsumed = true;
		
	    // Go through all clone pairs that are bigger in size than
	    // currClonePair and see if any of them subsume currClonePair.
	    map<int, ClonePairsList>::const_iterator pNextSize = pCurrSize;
	    for(++pNextSize;
	        !subsumed && (pNextSize != sortedPartition.end());
		++pNextSize) {

		const ClonePairsList &biggerClonePairs = pNextSize->second;

		for(ClonePairsList::const_iterator
		      pBiggerClonePair = biggerClonePairs.begin();
		    !subsumed && (pBiggerClonePair != biggerClonePairs.end());
		    ++pBiggerClonePair)

		    if((*pBiggerClonePair)->
		       Subsumes(*currClonePair, sameFunc))
			subsumed = true;
	    }
		
	    if (!subsumed)
		*result++ = currClonePair;
	}
    }
}

static PRIMITIVE
FilterSubsumed(SCM listOfClonePairs, SCM commonPercentage_a) {
    // This function is the sole entry point for this library. Input is a list
    // of clone pairs. The function discards clone pairs that are "subsumed"
    // by other clone pairs, and clone pairs that are "worse" than others and
    // returns the remaining ones. The argument listOfClonePairs is not
    // destructively updated.
    //
    // The following assumption is made on the clone pairs in
    // listOfClonePairs: for any two different functions f1 and f2,
    // considering all clone pairs from this function pair, the first (second)
    // components of all these clone pairs are from the same function (f1 or
    // f2).
    //
    // A clone pair is said to subsume another clone pair if each clone in the
    // first pair subsumes a distinct clone in the second. A clone is said to
    // subsume another if the set of nodes in the first is a superset of the
    // set of nodes in the second. Therefore subsumption testing does not use
    // commonPercentage_a.
    //
    // In the beginning of the function RemoveWorseClonePairs_NoOvChk I
    // describe what it means for a clone pair to be "worse" than
    // another. Note that a clone pair can be worse than another only if the
    // two pairs overlap (as defined below).
    
    int commonPercentage = STk_integer_value_no_overflow(commonPercentage_a);
    // The parameter commonPercentage is used to determine which clone-pairs
    // are overlapping. A pair of clone pairs is said to be overlapping if
    // each component in the first pair overlaps a distinct component in the
    // second pair.  A component (i.e., a clone) is said to overlap another if
    // commonPercentage of the nodes in each of the two components is present
    // in the other.
    

    // The first step is to partition the given clone pairs into buckets. Each
    // partition should have all clone pairs from the same pair of
    // functions. Recall the assumption stated earlier.
    //
    // Clearly a clone pair can be subsumed by or can be worse than another
    // only if both belong to the same partition.
    typedef hash_map<FunctionPair, ClonePairsList,
      FunctionPair::HashFunctionPair> PartitionedClonePairs ;
    PartitionedClonePairs partitionedClonePairs; // this is the table
    // containing all partitions.

    PRIMITIVE currSTkClonePair = listOfClonePairs;
    while(currSTkClonePair != STk_nil) {
	ClonePair *currClonePair = new ClonePair(STk_car(currSTkClonePair));
	FunctionPair currFunctionPair = currClonePair->GetFunctionPair();
	partitionedClonePairs[currFunctionPair].push_back(currClonePair);
	currSTkClonePair = STk_cdr(currSTkClonePair);
    }

    // Iterate over all partitions, and remove subsumed and worse clone pairs
    // from each one.
    
    PRIMITIVE retVal = STk_nil;
    for(PartitionedClonePairs::const_iterator pIt =
	  partitionedClonePairs.begin();
	pIt != partitionedClonePairs.end();
	++pIt) {

	// pIt->second points to current partition.
	
	const ClonePairsList &currPartition = pIt->second;
	
	bool sameFunc = (*currPartition.begin())->SameFunc();
	
	ClonePairsSameFuncPair::Kind kind =
	  sameFunc ?
	  ClonePairsSameFuncPair::SameFunc : ClonePairsSameFuncPair::DiffFuncs;
	
	// Remove subsumed clone pairs, and place the rest in
	// "result". "result" stores clone pairs in groups, such that every
	// pair of clone pairs within a group are overlapping. Notice that the
	// overlapping criteria is not used to decide which clone pairs
	// subsumes which others; it is used only to group the result after
	// subsumed clone pairs have been removed.
	
	ClonePairsSameFuncPair result(kind, commonPercentage);
	RemoveSubsumedClonePairs(currPartition.begin(),
				 currPartition.end(),
				 result.output(), sameFunc);

	// Remove worse clone pairs
	result.RemoveWorseClonePairs();

	result.PrintStats(binFile);
	
	for(ClonePairsSameFuncPair::input_iterator
	      pClonePair = result.begin();
	    pClonePair != result.end();
	    ++pClonePair)
	    
	    retVal = STk_cons((*pClonePair)->GetSTkList(), retVal);
    }
	
    // free storage allocated for clone pairs
    for(PartitionedClonePairs::const_iterator pPart =
	  partitionedClonePairs.begin();
	pPart != partitionedClonePairs.end();
	++pPart) {
	
	const ClonePairsList &currPartition = pPart->second;
	for(ClonePairsList::const_iterator pClonePair =
	      currPartition.begin();
	    pClonePair != currPartition.end();
	    ++pClonePair)

	    delete (*pClonePair);
    }
    
    return retVal;
}


extern "C"
PRIMITIVE STk_init_libfilter_pairs() {

    STk_add_new_primitive("filter-subsumed", tc_subr_2,
			  ((PRIMITIVE (*)(...))FilterSubsumed));

    return UNDEFINED;
}
