#ifndef __DIGRAPH_HEADER__
#define __DIGRAPH_HEADER__

//============================================================================
// DiGraph
// =======
// a subclass of the generic templated Graph class
// Implements a directed graph together with class DiGraphNode and 
// DiGraphEdge.
//
// Constructors
// ------------
// DiGraph()
// creates an empty directed graph
// 
// Destructors
// -----------
// ~DiGraph()
// frees allocated memory by new_edge and new_node
//
// Functions
// ---------
// DiGraphNode * new_node(const NodeInfoType &info)
// creates a node associated with input ``info''
//
// DiGraphEdge * new_edge(DiGraphNode * target, DiGraphNode * source, 
//                        const EdgeInfoType& info)
// creates a new edge from target to source, associated with ``info''
//
// NodeIterator NodeBegin(), NodeEnd()
// returns the iterator referencing the begin/end of the list of nodes
// in the graph
//        
// EdgeIterator EdgeBegin(), EdgeEnd()
// returns the iterator referencing the begin/end of the list of edges
// in the graph
//
// DiGraph::NodeIterator
// =====================
// Iterator class, iterates through nodes in the graph
// 
// Constructors/Destructors
// ------------------------
// none
//
// Overloaded operators
// --------------------
// ++ () prefix
// increments the iterator, points to next node in graph
// order of the traverse is not specificied
// 
// = (const NodeIterator& node_ite)
// assigns node_ite to current iterator
//
// ==, /= (const NodeIterator& node_ite)
// test for equality/inequality of two NodeIterators
//
// *()
// dereference the iterator, returns the pointer to the referenced node
// (DiGraphNode *)
//
/// DiGraph::EdgeIterator
// =====================
// Iterator class, iterates through edges in the graph
// 
// Constructors/Destructors
// ------------------------
// none
//
// Overloaded operators
// --------------------
// ++ () prefix
// increments the iterator, points to next unhidden edge in graph
// 
// = (const EdgeIterator& edge_ite)
// assigns edge_ite to current iterator
//
// ==, /= (const EdgeIterator& edge_ite)
// test for equality/inequality of two EdgeIterators
//
// *()
// dereference the iterator, returns the pointer to the referenced node
// (DiGraphEdge *)/ 
//============================================================================

#include "graph.h"
#include "dinode.h"
#include "diedge.h"


template <class NodeInfoType, class EdgeInfoType>
class DiGraph : public Graph<NodeInfoType, EdgeInfoType> {
    
    typedef typename std::map<Node *, NodeInfoType *>::iterator  nmap_iterator;
    typedef typename std::map<Edge *, EdgeInfoType *>::iterator  emap_iterator;
    typedef typename std::pair<nmap_iterator, bool> insert_node_ret_val;
    typedef typename std::pair<emap_iterator, bool> insert_edge_ret_val;

    public:

    class NodeIterator {

            friend class DiGraph;

            private:
                nmap_iterator ite;
            public:

                NodeIterator& operator ++();
                NodeIterator& operator =(const NodeIterator& node_ite);
                bool operator ==(const NodeIterator& node_ite );
                bool operator !=(const NodeIterator& node_ite );
                DiGraphNode * operator *();
        };
        class EdgeIterator {

            friend class DiGraph;
            private:
                emap_iterator ite;
            public:
                EdgeIterator& operator ++();
                EdgeIterator& operator =(const EdgeIterator& edge_ite);
                bool operator ==(const EdgeIterator& edge_ite );
                bool operator !=(const EdgeIterator& edge_ite );
                DiGraphEdge * operator *();
        };

        typedef typename DiGraph::NodeIterator node_ite;
        typedef typename DiGraph::EdgeIterator edge_ite;


        DiGraph();
        ~DiGraph();

        DiGraphNode * new_node(const NodeInfoType &info);
        DiGraphEdge * new_edge(DiGraphNode * target, DiGraphNode * source, 
                               const EdgeInfoType& info);

        NodeIterator NodeBegin();
        NodeIterator NodeEnd();
        
        EdgeIterator EdgeBegin();
        EdgeIterator EdgeEnd();
};

#endif
