#ifndef __UGRAPH_HEADER__
#define __UGRAPH_HEADER__

#include "graph.h"
#include "unode.h"
#include "uedge.h"

//============================================================================
// UGraph
// =======
// a subclass of the generic templated Graph class
// Implements an undirected graph together with class UGraphNode and 
// UGraphEdge.
//
// Constructors
// ------------
// UGraph()
// creates an empty undirected graph
// 
// Destructors
// -----------
// ~UGraph()
// frees allocated memory by new_edge and new_node
//
// Functions
// ---------
// UGraphNode * new_node(const NodeInfoType &info)
// creates a node associated with input ``info''
//
// UGraphEdge * new_edge(UGraphNode * node1, UGraphNode * node2, 
//                        const EdgeInfoType& info)
// creates a new edge connecting node1 and node2 and
// associated with ``info''
//
// NodeIterator NodeBegin(), NodeEnd()
// returns the iterator referencing the begin/end of the list of nodes
// in the graph
//        
// EdgeIterator EdgeBegin(), EdgeEnd()
// returns the iterator referencing the begin/end of the list of edges
// in the graph
//
// UGraph::NodeIterator
// =====================
// Iterator class, iterates through nodes in the graph
// 
// Constructors/Destructors
// ------------------------
// none
//
// Overloaded operators
// --------------------
// ++ () prefix
// increments the iterator, points to next node in graph
// order of the traverse is not specificied
// 
// = (const NodeIterator& node_ite)
// assigns node_ite to current iterator
//
// ==, /= (const NodeIterator& node_ite)
// test for equality/inequality of two NodeIterators
//
// *()
// dereference the iterator, returns the pointer to the referenced node
// (UGraphNode *)
//
/// UGraph::EdgeIterator
// =====================
// Iterator class, iterates through edges in the graph
// 
// Constructors/Destructors
// ------------------------
// none
//
// Overloaded operators
// --------------------
// ++ () prefix
// increments the iterator, points to next unhidden edge in graph
// 
// = (const EdgeIterator& edge_ite)
// assigns edge_ite to current iterator
//
// ==, /= (const EdgeIterator& edge_ite)
// test for equality/inequality of two EdgeIterators
//
// *()
// dereference the iterator, returns the pointer to the referenced node
// (UGraphEdge *) 
//============================================================================


template <class NodeInfoType, class EdgeInfoType>
class UGraph : public Graph<NodeInfoType, EdgeInfoType> {
    
    typedef typename std::map<Node *, NodeInfoType *>::iterator  nmap_iterator;
    typedef typename std::map<Edge *, EdgeInfoType *>::iterator  emap_iterator;
    typedef typename std::pair<nmap_iterator, bool> insert_node_ret_val;
    typedef typename std::pair<emap_iterator, bool> insert_edge_ret_val;

    public:

    class NodeIterator {

            friend class UGraph;

            private:
                nmap_iterator ite;
            public:

                NodeIterator& operator ++();
                NodeIterator& operator =(const NodeIterator& node_ite);
                bool operator ==(const NodeIterator& node_ite );
                bool operator !=(const NodeIterator& node_ite );
                UGraphNode * operator *();
        };
        class EdgeIterator {

            friend class UGraph;
            private:
                emap_iterator ite;
            public:
                EdgeIterator& operator ++();
                EdgeIterator& operator =(const EdgeIterator& edge_ite);
                bool operator ==(const EdgeIterator& edge_ite );
                bool operator !=(const EdgeIterator& edge_ite );
                UGraphEdge * operator *();
        };

        typedef typename UGraph::NodeIterator node_ite;
        typedef typename UGraph::EdgeIterator edge_ite;


        UGraph();
        ~UGraph();

        UGraphNode * new_node(const NodeInfoType &info);
        UGraphEdge * new_edge(UGraphNode * node1, UGraphNode * node2, 
                               const EdgeInfoType& info);

        NodeIterator NodeBegin();
        NodeIterator NodeEnd();
        
        EdgeIterator EdgeBegin();
        EdgeIterator EdgeEnd();
};

#endif
