// Copyright (c) 2003 Raghavan Komondoor and Susan Horwitz

// This file is part of DUP.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software DUP and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include <iostream.h>
extern "C" {
#include <string.h>
}

#include <stk.h>
#include <hash_set.h>
#include <hash_map.h>
#include <list>
#include <map>
#include <stack>
#include <template.cpp>
/* This file has code that analyzes the CFGs in order to extract information
   that is required during clone-pair detection. There are three methods in
   this file that are exported (i.e., made visible) to the clone-pair
   detection code (which is written in Scheme): the constructor of the CFG
   class, CFG::ComputeReachability, and
   CFG::ComputeControlLabels. In the Scheme code these three methods are
   respectively known as make-cfg, compute-reachability, and
   compute-control-labels.

   make-cfg constructs an object of the CFG class (i.e., the cfg of a
   procedure), wraps this object to make it look like a Scheme object, and
   then returns it. make-cfg takes three arguments: the vertex id of the ENTRY
   node of the procedure, the list of edges in the procedure (each edge is a
   pair of vertex ids), and the name of the procedure.

   compute-reachability takes two arguments: the (wrapped) CFG object of a
   procedure, and the list of flow-dependence edges in that procedure (each
   edge is specified as a pair of vertex ids). compute-reachability returns
   one giant list, with each element of this list itself being a list that
   contains computed information about one of the given flow-dependence
   edges. For more information on this, see the comment at the beginning of
   the method CFG::ComputeReachability. Note that CFG::ComputeReachability
   takes one argument only -- the list of flow-dependence edges;
   compute-reachability is essentially a wrapper around
   CFG::ComputeReachability, with the first argument to compute-reachability
   (the CFG object) becoming the implicit "this" parameter to
   CFG::ComputeReachability.

   compute-control-labels takes two arguments: the (wrapped) CFG object of a
   procedure, and the list of control-dependence edges in that procedure (each
   edge is specified as a pair of vertex ids). This function also returns a
   giant list, with each element containing (a conservative guess
   of) the label of one of the given control-dependence edges. See the
   comment at the beginning of CFG::ComputeControlLabels for the exact format
   of this returned list. Note that labels cannot necessarily be guessed for
   every given control-dependence edge; if a label cannot be guessed for an
   edge, that edge is not present at all in the return value. As with
   compute-reachability, the first argument of compute-control-labels becomes
   the "this" argument to CFG::ComputeControlLabels.

*/

/* A note:

   As we know, a "back edge" in a CFG is an edge whose target dominates the
   source; and natural loops are defined in terms of back edges. On the other
   hand, a "backward edge" in a DFS spanning tree is something else. The code
   in this file attempts to find all natural loops in a CFG; it does so
   because part of the information returned by compute-reachability is, for
   each flow-dependence edge, whether that edge is loop carried or loop
   independent.

   However, the code in this file does not compute dominators, and does not
   compute true "back edges". It simply does a DFS, and pretends that all
   backward edges in the DFS spanning tree are back edges. It then finds
   natural loops on this basis. This is an incorrect thing to do, but
   hopefully in practice it will not result in too much inaccuracy. Isn't it
   true that every backward edge is a back edge unless the CFG is irreducible?

*/




typedef DiGraphNode * node;
typedef DiGraphEdge * edge;

#define nil NULL

static int tc_CFG;

struct EdgeInfo {
    enum EdgeKind {True, False, Ignore} kind;

    EdgeInfo(EdgeKind kind_a) : kind(kind_a) {}
    EdgeInfo() {} // only for LEDA

    friend ostream &operator << (ostream &, const EdgeInfo &);
    friend istream &operator >> (istream &, EdgeInfo &);
};



ostream &
operator << (ostream& out, const EdgeInfo &) {
    return out;
}

istream &
operator >> (istream& in, EdgeInfo &) {
    return in;
}

class CFG;

class Loop : private map<node, bool> {
    CFG &cfg;
    list<edge> backEdges;
    node header;


  public:
    Loop(node header_a, list<edge> backEdges_a,
	 CFG &cfg_a);
    // pre-condition: all edges in backEdges_a must be back
    // edges into header_a.  The constructor builds the loop body

    void *operator new(size_t size) {
	return STk_must_malloc(size);
    }

    void Print(ostream &);
    
    //REP_LEDA_MAP
    //bool Member(Leda_node v) {return (*this)[v];}

    bool Member(node v) {
        
        map<node, bool>::iterator ite = (*this).find(v);
        if(ite == (*this).end()) {
            (*this).insert(value_type(v, false));
            return false;
        }
        else {
            return (*ite).second;
        }
    
    }
    //REP_LEDA_MAP
    void MarkMembers(map<node, bool> &result) {
	//forall_defined(v, *((Leda_map<node, bool> *)this))
	//  result[v] = (*this)[v] || result[v];
    //
    
        for(map<node, bool>::iterator ite = (*this).begin();
                ite!=(*this).end();
                ite++) {
            result[(*ite).first] = (*ite).second || result[(*ite).first];
        }
    }

};
struct NodeInfo {
    int pdgVertexId;
    node dfsTreeParent;
    int dfsNum;
    list<edge> backEdges;  // backedges coming into this node.

    Loop *loop; // `loop' will be non NULL only for loop header nodes. 
    
    NodeInfo(int pdgVertexId_a) :
	pdgVertexId(pdgVertexId_a), dfsTreeParent(nil),	loop(NULL) {}

    NodeInfo() : loop(NULL) {}; // this constructor is only for LEDA

    ~NodeInfo() {delete loop;}
    
    /* NOTE: LEDA does not like this :-(, because it defines its own three
       argument operator new().

       void *operator new(size_t size) {
	return STk_must_malloc(size);
	}
    */
    friend ostream &operator << (ostream &, const NodeInfo &);
    friend istream &operator >> (istream &, NodeInfo &);
};


template class DiGraph<NodeInfo, EdgeInfo>;
template class Graph<NodeInfo, EdgeInfo>;



ostream &
operator << (ostream& out, const NodeInfo &) {
    return out;
}

istream &
operator >> (istream& in, NodeInfo &) {
    return in;
}

//REP_LEDA_GRAPH
typedef  DiGraph<NodeInfo, EdgeInfo>::NodeIterator node_iterator;
typedef  DiGraphNode::EdgeIterator edge_iterator;




class CFG : public DiGraph<NodeInfo, EdgeInfo> {
    char *name;
    node enterNode;
 

    
    //REP_LEDA_H_ARRAY
    //leda_h_array<int, Leda_node> nodeOf;
    map<int, node> nodeOf;
    typedef map<int, node>::iterator nodeOfIterator;
    typedef map<int, node>::value_type nodeOfType;
    // nodeOf is a mapping pdg-vertex-id -> node


    enum VisitState {unvisited, inProgress, finished};
    public:
    struct VertexIDPair {
        int from;
        int to;

        VertexIDPair() {}
        VertexIDPair(int from_a, int to_a)
            : from(from_a), to(to_a) {}
    };
    class VertexIDPairLess {
        public:
        bool operator() ( const VertexIDPair & p1, const VertexIDPair & p2) const
        {
            if(p2.from > p1.from)
                return true;
            else if(p2.from < p1.from)
                return false;
            else
                return p1.to < p2.to;
        }
    };


    bool NodeOfDefinedNotNil (int VertexID) {
    
       nodeOfIterator ite = nodeOf.find(VertexID);
     
       //undefined
       if(ite == nodeOf.end()) {
            //put in dom
            nodeOf.insert(nodeOfType(VertexID, nil));
            return false;
       }
       else if( (*ite).second == nil)
           return false;
       else
           return true;
    }
    


    struct HashVertexIDPair {
	size_t operator()(const VertexIDPair &x) const {
	    return Hash(x);
	}
    };

    void InitializingDFS(node startNode,
			 int &startNum,
			 map<node, VisitState> &visitState);
    void DFS(node startNode,
            map<node, VisitState> &visitState,
            list<node> &nodesVisited);

    //REP_LEDA_SLIST
    typedef list<node> ListOfContainingHeaders;
    
    //REP_LEDA_SLIST
    
    PRIMITIVE RelativeNestingLoops
    (ListOfContainingHeaders &headers1, ListOfContainingHeaders &headers2);
    
    friend int Hash(const VertexIDPair &);
    friend bool operator== (const VertexIDPair &, const VertexIDPair &);
    
  public:
    CFG(int enterVertexId, SCM listOfEdges, SCM name_a);
    // listOfEdges should be a list
    // of vertex-id pairs representing the CFG edges.
    ~CFG() {delete name;}
    void Mark() {};
    void *operator new(size_t size) {
	return STk_must_malloc(size);
    }
    PRIMITIVE ComputeReachability(SCM listOfDepEdges); // listOfDepEdges
    // should be a list of vertex-id pairs representing the PDG edges. For
    // each of these edges, this method returns forward-reachable and relative
    // loop nesting depth info.

    PRIMITIVE ComputeControlLabels(SCM listOfControlEdges);
    // listOfControlEdges should be a list of control dependence edges. For
    // each of these edges, this method tried to determine and return a T/F
    // label.

    const char *Name() {return name;}
};

CFG::CFG(int enterVertexId, SCM listOfEdges, SCM name_a):
      name(strcpy(new char[strlen(CHARS(name_a))+1], CHARS(name_a)))  {
    // add the edges to the CFG
    SCM currEdgePos = listOfEdges;
    while(currEdgePos != STk_nil) {

        int headVertexId =
            STk_integer_value_no_overflow(STk_caar(currEdgePos));

        //REP_LEDA_H_ARRAY
        //node headNode = nodeOf[headVertexId];
        //if(headNode == nil)
        node headNode;
        nodeOfIterator headnode_ite= nodeOf.find(headVertexId);
        if(headnode_ite == nodeOf.end()) {
            // nodeOf[headVertexId] = headNode =
            //  new_node(*(new NodeInfo(headVertexId)));
            headNode = new_node(*(new NodeInfo(headVertexId)));
            nodeOf.insert(nodeOfType(headVertexId, headNode));
        }
        else if ((*headnode_ite).second == nil) {
        
            (*headnode_ite).second = headNode = new_node(*(new NodeInfo(headVertexId)));

        }
        else headNode =  (*headnode_ite).second;
        
        int tailVertexId =
            STk_integer_value_no_overflow(STk_cadar(currEdgePos));

        node tailNode;
        nodeOfIterator tailnode_ite= nodeOf.find(tailVertexId);
        if(tailnode_ite == nodeOf.end()) {
            //nodeOf[tailVertexId] = tailNode =
            //  new_node(*(new NodeInfo(tailVertexId)));
            tailNode = new_node(*(new NodeInfo(tailVertexId))); 
            nodeOf.insert( nodeOfType( tailVertexId, tailNode));
        }
        else if ((*tailnode_ite).second == nil) {
        
            (*tailnode_ite).second = tailNode = new_node(*(new NodeInfo(tailVertexId)));

        }
        else tailNode =  (*tailnode_ite).second;
        
        
        EdgeInfo::EdgeKind kind;
        PRIMITIVE kind_a = STk_caddar(currEdgePos);
        if(kind_a == STk_truth)
            kind = EdgeInfo::True;
        else if(kind_a == STk_ntruth)
            kind = EdgeInfo::False;
        else
            kind = EdgeInfo::Ignore;

        //REP_LEDA_GRAH
        
        // :) in my graph data structure the order is target, source
        // leda is the other way round,... BUT .. I'm not changing my graph
        
        //new_edge(headNode, tailNode, *(new EdgeInfo(kind)));

        new_edge(tailNode, headNode,  *(new EdgeInfo(kind)));
        
        currEdgePos = STk_cdr(currEdgePos);
    }
    
    //REP_LEDA_H_ARRAY
    //assert((enterNode = nodeOf[enterVertexId]) != nil);
    
    nodeOfIterator enternode_ite = nodeOf.find(enterVertexId);
    assert(enternode_ite!=nodeOf.end());
    enterNode = (*enternode_ite).second;


    
    // do an initializing DFS
    
    //REP_LEDA_NODE_ARRAY
    //LEDA_NODE_ARRAY<VisitState> visitState(*this, unvisited);
    map<node, VisitState> visitState;
    
    //REP_LEDA_GRPAH
    node_iterator vi;
    //node vi;
    //forall_nodes(vi, *this) {
    for_all_nodes(vi, *this) {
        //visitState.insert(map<node, VisitState>::value_type(vi, unvisited));
        visitState.insert(map<node, VisitState>::value_type(*vi, unvisited));
    }
    
    int startNum = 1;
    InitializingDFS(enterNode, startNum, visitState);
    
    // Create all the loops
    //REP_LEDA_GRPAH
    //node v;
    //forall_nodes(v, *this) {
    node_iterator vi2;
      for_all_nodes(vi2, *this) {
        if(!inf(*vi2).backEdges.empty()) { // v is a loop header
            ((*this).inf(*vi2)).loop =
                new Loop(*vi2, inf(*vi2).backEdges, *this);
        }
      }

    //CFG DEBUG
    /*
       cerr<< "CFG construction finish" << endl;
       node_iterator debug_nodeite;
       for_all_nodes(debug_nodeite, *this) {
       cerr << (*this).inf(*debug_nodeite).pdgVertexId << ":";
       edge_iterator debug_edgeite;
       for_all_out_edges(debug_edgeite, *debug_nodeite) {

       cerr << inf(source(*debug_edgeite)).pdgVertexId << "=>" << inf(target(*debug_edgeite)).pdgVertexId << " ";
       }
       cerr << endl;

       }

       cerr << endl;

       cerr << endl;
    */    

}

inline PRIMITIVE
MakeList(SCM a, SCM b) {
    return STk_cons(a, STk_cons(b, STk_nil));
}

inline PRIMITIVE
MakeList(SCM a, SCM b, SCM c) {
    return STk_cons(a, STk_cons(b, STk_cons(c, STk_nil)));
}
bool getValueLoopIndependent(map<CFG::VertexIDPair, bool, CFG::VertexIDPairLess>& loop_indep , CFG::VertexIDPair& p) {
    
    map<CFG::VertexIDPair, bool, CFG::VertexIDPairLess>::iterator ite = loop_indep.find(p);
    if(ite == loop_indep.end())
        return false;
    else
        return (*ite).second;
}
PRIMITIVE
CFG::ComputeReachability(SCM listOfDepEdges) {
    // For every pair (p . q) in listOfDepEdges, return a list ((p . q)
    // <reach> (containing-headers-of-p
    // . containing-headers-for-q)). p and q are vertex IDs.
	
    // <reach> is true iff there is a CFG path from p to q after all backedges
    // that correspond to loops that enclose both p and q have been removed.

    // containing-headers-of-p is a list of vertex IDs of headers of loops
    // that contain p, ignoring loops that also contain q. Similarly,
    // containing-headers-of-q is a list of vertex IDs of headers of loops
    // that contain q, ignoring loops that also contain p. Both lists are
    // sorted from innermost loop to outermost loop.

    //REP_LEDA_H_ARRAY
    //leda_h_array<VertexIDPair, bool> loopIndependent(false);
    //leda_h_array<int, ListOfContainingHeaders *>
    //  containingLoopHeadersOf(NULL);
    map<int, ListOfContainingHeaders *> containingLoopHeadersOf;
    typedef map<int, ListOfContainingHeaders *>::iterator containingLoopHeadersOfIterator;
    typedef map<int, ListOfContainingHeaders *>::value_type containingLoopHeadersOfType;
    
    map<VertexIDPair, bool, VertexIDPairLess> loopIndependent ;
    typedef map<VertexIDPair, bool, VertexIDPairLess>::value_type loopIndepType;
    typedef map<VertexIDPair, bool, VertexIDPairLess>::iterator loopIndepIterator;
   
    //REP_LEDA_GRAPH
    node currNode;
    node_iterator curr_ite;
    //forall_nodes(currNode, *this) {
    for_all_nodes(curr_ite, *this) {
        currNode = *curr_ite;
        // Build a list of loop headers in whose loops currNode is
        // contained. The list should be ordered from innermost loop to
        // outermost loop.


        //REP_LEDA_H_ARRAY
        //ListOfContainingHeaders &containingLoopHeaders =
        //    *(containingLoopHeadersOf[inf(currNode).pdgVertexId] =
        //            new ListOfContainingHeaders);
        ListOfContainingHeaders &containingLoopHeaders = *(new ListOfContainingHeaders);
        containingLoopHeadersOf.insert(containingLoopHeadersOfType(inf(currNode).pdgVertexId, &containingLoopHeaders));

        
        for(node currDFSTreeParent = currNode;
                currDFSTreeParent != nil;
                currDFSTreeParent = inf(currDFSTreeParent).dfsTreeParent) {

            // check whether currDFSTreeParent is a loop header, and if yes
            // whether currNode is contained in its loop.
            if(!inf(currDFSTreeParent).backEdges.empty() &&
                    inf(currDFSTreeParent).loop->Member(currNode))

                //REP_LEDA_SLIST
                containingLoopHeaders.push_back(currDFSTreeParent);
        }

        // Hide all back-edges coming into the headers in
        // containingLoopHeaders.

        //REP_LEDA_SLIST
        /*
           for(slist_item headerItem = containingLoopHeaders.first();
           headerItem != nil;
           headerItem = containingLoopHeaders.succ(headerItem)) 
           */

        for(ListOfContainingHeaders::iterator headerItem = containingLoopHeaders.begin();
                headerItem != containingLoopHeaders.end();
                headerItem++ ) 
            //REP_LEDA_SLIST

        {

            const list<edge> &backEdges =
                inf(*headerItem).backEdges;

            //REP_LEDA_SLIST

            /*
               for(slist_item bEItem = backEdges.first();
               bEItem != nil;
               bEItem = backEdges.succ(bEItem))

               hide_edge(backEdges[bEItem]);
               */

            list<edge>::const_iterator ite;

            for(ite = backEdges.begin(); ite != backEdges.end(); ite++) {
                hide_edge(*ite);
            }
            //REP_LEDA_SLIST
        }

        //REP_LEDA_NODE_ARRAY
        map<node, bool> ignore; 

        //REP_LEDA_GRAPH
        //node vi;
        //forall_nodes(vi, *this) {
        node_iterator vi;
        for_all_nodes(vi, *this) {
            ignore.insert(map<node, bool>::value_type(*vi, false));
        }
                

        //REP_LEDA_SLIST
        
        //slist_item headerItem = containingLoopHeaders.first();
        
        ListOfContainingHeaders::iterator headerItem = containingLoopHeaders.begin();
        
        bool done = false;
        do {
            
            //REP_LEDA_NODE_ARRAY
            //LEDA_NODE_ARRAY<VisitState> visitState(*this, unvisited);
            map<node, VisitState> visitState;

            //REP_LEDA_GRPAPH
            //node vi;
            //forall_nodes(vi, *this) {
            node_iterator vi;
            for_all_nodes(vi, *this) {
                visitState.insert(map<node, VisitState>::value_type(*vi, unvisited));
            }
            
                
            
            list<node> nodesVisited;
            DFS(currNode, visitState, nodesVisited);

            // Everything in nodesVisited but not in `ignore' should be added
            // to loopIndependent.
            for(list<node>::const_iterator it = nodesVisited.begin();
                    it != nodesVisited.end();
                    ++it) {

                node nodeVisited = *it;
                if(!ignore[nodeVisited]) {
                    
                    //REP_LEDA_H_ARRAY
                    //loopIndependent
                    //    [VertexIDPair(inf(currNode).pdgVertexId,
                    //            inf(nodeVisited).pdgVertexId)] = true;
                    
                    VertexIDPair p;
                    p.from = inf(currNode).pdgVertexId;
                    p.to = inf(nodeVisited).pdgVertexId;
    
                    loopIndepIterator loopindep_ite = loopIndependent.find(p);
                    if(loopindep_ite != loopIndependent.end()) {
                        (*loopindep_ite).second = true;
                    }
                    else {
                        loopIndependent.insert(loopIndepType(p, true));
                    }
                
                
                }
            }

            if(headerItem != containingLoopHeaders.end() ) { // we do have another surrounding loop to
                // process.

                // restore back-edges corresponding to the loop
                //node headerNode = containingLoopHeaders[headerItem];
                
                node headerNode = *headerItem;
                
                const list<edge> &backEdges =
                    inf(headerNode).backEdges;

                //REP_LEDA_SLIST
                /*
                   for(slist_item bEItem = backEdges.first();
                   bEItem != nil;
                   bEItem = backEdges.succ(bEItem))

                   restore_edge(backEdges[bEItem]);
                   */
                list<edge>::const_iterator ite;

                for(ite = backEdges.begin(); ite != backEdges.end(); ite++) {
                    restore_edge(*ite);
                }

                //REP_LEDA_SLIST


                // mark all nodes belonging to the loop as ignorable.
                inf(headerNode).loop->MarkMembers(ignore);

                // go to next surrounding loop
                //headerItem = containingLoopHeaders.succ(headerItem);
                headerItem++;
            }
            else
                done = true;
        } while(!done);
    }

    // build the list to return
    PRIMITIVE retVal = STk_nil;
    SCM currEdgePos = listOfDepEdges;
    while(currEdgePos != STk_nil) {
	const int from = STk_integer_value_no_overflow(STk_caar(currEdgePos));
	const int to = STk_integer_value_no_overflow(STk_cdar(currEdgePos));
    
    //REP_LEDA_H_ARRAY
//	if(nodeOf.defined(from) && (nodeOf[from] != nil) &&
//	   nodeOf.defined(to) && (nodeOf[to] != nil)) {
        
    if(NodeOfDefinedNotNil(from) && NodeOfDefinedNotNil(to)) {
        
        VertexIDPair p;
        p.from = from;
        p.to = to;
	    retVal =
	      STk_cons(MakeList(STk_car(currEdgePos),
				//loopIndependent[VertexIDPair(from, to)] ?
				getValueLoopIndependent(loopIndependent, p)?
                STk_truth : STk_ntruth,
				RelativeNestingLoops
				(*(containingLoopHeadersOf[from]),
				 *(containingLoopHeadersOf[to])
				 )
				),
		       retVal
		       );
    }

	currEdgePos = STk_cdr(currEdgePos);
    }
    
    
    // free up storage
    //REP_LEDA_SLIST
    /*
       ListOfContainingHeaders *headers;
       forall(headers, containingLoopHeadersOf) {
       delete headers;
       }
       */
    //REP_LEDA_SLIST
    
    return retVal;
}



PRIMITIVE
CFG::ComputeControlLabels(SCM listOfControlEdges) {
    // For each pair (p . q) in listOfControlEdges for which it is possible,
    // compute and return a list ((p . q) #t) or a list ((p . q) #f). That is
    // the label on the control dependence edge. This method assumes that p
    // is a predicate, and that q is control dependent on p. 
	
    // This method uses a safe heuristic to compute the labels: If q is not
    // reachable from p when the true (false) CFG-edge out of p is removed,
    // then q is true (false) control dependent on p. Note that this heuristic
    // never computes more than one label for a control-dependence
    // edge. However, it may compute no label for some edges.

    // create the set of source nodes of the edges in listOfControlEdges.
    hash_set<int> predicateNodes;
    for(SCM currEdgePos = listOfControlEdges;
	currEdgePos != STk_nil;
	currEdgePos = STk_cdr(currEdgePos))

	predicateNodes.insert
	  (STk_integer_value_no_overflow(STk_caar(currEdgePos)));

    const int ReachedWithoutTrueEdge = 01;
    const int ReachedWithoutFalseEdge = 02;
    hash_map<VertexIDPair, int, HashVertexIDPair> vertexPairInfo; // for
	// each control-dependence edge p -> q, remember whether
	// q can be reached from p with the "true" edge out of p removed, and
	// whether q can be reached from p with the "false" edge out of p removed.
    
    for(hash_set<int>::const_iterator i = predicateNodes.begin();
	i != predicateNodes.end();
	++i) {
	// this loop iterates over all nodes (predicates) in predicateNodes.

    //REP_LEDA_H_ARRAY
    node currNode;
	//Leda_node currNode = nodeOf[*i]; // the current predicate
    
    nodeOfIterator findcurr_ite = nodeOf.find(*i);
    if(findcurr_ite == nodeOf.end()) {
        currNode = nil;
        nodeOf.insert(nodeOfType(*i, currNode));
    }
    else
        currNode = (*findcurr_ite).second;

	// check if there is a true/false CFG-edge out of currNode.
	edge trueEdge = nil, falseEdge = nil;
	
    edge currEdge;
    edge_iterator edge_ite;
    //REP_LEDA_GRAPH
	//forall_adj_edges(currEdge, currNode) {
	for_all_out_edges(edge_ite, currNode) {
        currEdge = *edge_ite;
        //switch((*this)[currEdge].kind) {
        switch( (*this).inf(currEdge).kind) {
	      case EdgeInfo::True:
		trueEdge = currEdge;
		break;

	      case EdgeInfo::False:
		falseEdge = currEdge;
	    }
	}
	// determine the nodes that can be reached from currNode with the
	// "true" edge out of currNode removed.
	if (trueEdge != nil) {
	    hide_edge(trueEdge);

        //REP_LEDA_NODE_ARRAY
	    //LEDA_NODE_ARRAY<VisitState> visitState(*this, unvisited);
        map<node, VisitState> visitState;
        
        //REP_LEDA_GRAPH
        node_iterator vi;
        //node vi;
        //forall_nodes(vi, *this) {
        for_all_nodes(vi, *this) {
            visitState.insert(map<node, VisitState>::value_type(*vi, unvisited));
        }
	    list<node> nodesVisited;
	    DFS(currNode, visitState, nodesVisited);

	    for(list<node>::const_iterator nodePtr
		  = nodesVisited.begin();
		nodePtr != nodesVisited.end();
		++nodePtr)

		vertexPairInfo[VertexIDPair((*this).inf(currNode).pdgVertexId,
					    (*this).inf(*nodePtr).pdgVertexId)]
		  |= ReachedWithoutTrueEdge;

	    restore_edge(trueEdge);
	}

	// determine the nodes that can be reached from currNode with the
	// "false" edge out of currNode removed.
	if (falseEdge != nil) {
	    hide_edge(falseEdge);

	    //LEDA_NODE_ARRAY<VisitState> visitState(*this, unvisited);
        map<node, VisitState> visitState;
        
        //REP_LEDA_GRAPH
        //node vi;
        node_iterator vi;
        //forall_nodes(vi, *this) {
        for_all_nodes(vi, *this) {
            visitState.insert(map<node, VisitState>::value_type(*vi, unvisited));
        }

	    list<node> nodesVisited;
	    DFS(currNode, visitState, nodesVisited);

	    for(list<node>::const_iterator nodePtr
		  = nodesVisited.begin();
		nodePtr != nodesVisited.end();
		++nodePtr) {

		vertexPairInfo[VertexIDPair((*this).inf(currNode).pdgVertexId,
					    (*this).inf(*nodePtr).pdgVertexId)]
		  |= ReachedWithoutFalseEdge;
	    }
	    
	    restore_edge(falseEdge);
	}
    }

    // build the list to return
    PRIMITIVE retVal = STk_nil;
    SCM currEdgePos = listOfControlEdges;
    while(currEdgePos != STk_nil) {
	const int from = STk_integer_value_no_overflow(STk_caar(currEdgePos));
	const int to = STk_integer_value_no_overflow(STk_cdar(currEdgePos));

	if(vertexPairInfo.find(VertexIDPair(from, to)) !=
	   vertexPairInfo.end()) {

	    if(!(vertexPairInfo[VertexIDPair(from, to)] &
		 ReachedWithoutTrueEdge))
		// "to" was not reachable from "from" with true-outgoing edge
		// from "from" removed.
		
		retVal = STk_cons(MakeList(STk_car(currEdgePos), STk_truth),
				  retVal);
	    
	    else if(!(vertexPairInfo[VertexIDPair(from, to)] &
		      ReachedWithoutFalseEdge))
		// "to" was not reachable from "from" with false-outgoing edge
		// from "from" removed.
		
		retVal = STk_cons(MakeList(STk_car(currEdgePos), STk_ntruth),
				  retVal);
	}
	
	currEdgePos = STk_cdr(currEdgePos);
    }
    
    return retVal;
}

void
CFG::InitializingDFS(node currNode, int &startNum,
		     map<node, VisitState>  &visitState) {
    
    // This routine sets visitState. It also sets the dfsTreeParent, dfsNum
    // and backEdges fields in the NodeInfo structures.

    //GRAPH INDEXING
    //REP_LEDA_GRAPH
    //(*this)[currNode].dfsNum = startNum++;
    (*this).inf(currNode).dfsNum = startNum++;
    visitState[currNode] = inProgress;

    //REP_LEDA_GRAPH
    edge adjEdge;
    edge_iterator adjEdge_ite;
    //forall_adj_edges(adjEdge, currNode) {
    
    for_all_out_edges(adjEdge_ite, currNode) {
    adjEdge = *adjEdge_ite;
	node succNode = target(adjEdge);
	
	switch(visitState[succNode]) {
	  case unvisited:
	    InitializingDFS(succNode, startNum, visitState);
	    (*this).inf(succNode).dfsTreeParent = currNode;
	    break;
	    
	  case inProgress:
        //REP_LEDA_SLIST
        //REP_LEDA_GRAPH
	    //(*this)[succNode].backEdges.push_front(adjEdge);
	    (*this).inf(succNode).backEdges.push_front(adjEdge);
        //REP_LEDA_SLIST
        break;
	}
    }
	    
    visitState[currNode] = finished;
}

void
CFG::DFS(node currNode,
	 map<node, VisitState> &visitState,
	 list<node> &nodesVisited) {

    // This routine sets visitState and nodesVisited. It does *not* affect the
    // dfsTreeParent, dfsNum and backEdges fields in the NodeInfo structures.
    
    visitState[currNode] = inProgress;
    
    //REP_LEDA_GRAPH
    //we have not forall_adj_nodes implemented, so just go with
    //for_all_out edges and use target to get the adj nodes
    node succNode;
    edge_iterator out_ite, in_ite;
    //forall_adj_nodes(succNode, currNode) {
    for_all_out_edges(out_ite, currNode) { 
        succNode = target(*out_ite);
        if(visitState[succNode] == unvisited) {
            DFS(succNode, visitState, nodesVisited);
        }
    }
    
    visitState[currNode] = finished;
    nodesVisited.push_back(currNode);
}

PRIMITIVE
CFG::RelativeNestingLoops(ListOfContainingHeaders &headers1,
        ListOfContainingHeaders &headers2) {
    // headers1 is the list of loop headers in whose loops a node p is
    // contained. Similarly, `headers2' is the list of loop headers in whose
    // loops a node q is contained. Both lists should be sorted from
    // innermost to outermost.
    // 
    // This function returns a (Scheme) pair (containing-headers-of-p,
    // containing-headers-of-q), where each component of the pair is itself a
    // (Scheme) list. See the comment in the beginning of
    // CFG:ComputeReachability to see what these lists are.

    //REP_LEDA_SLIST
    /*
    slist_item it1 = headers1.first();
    SCM list1 = STk_nil;

    slist_item it2 = headers2.first();
    SCM list2 = STk_nil;
    */
    ListOfContainingHeaders::iterator it1 = headers1.begin();
    SCM list1 = STk_nil;
    
    ListOfContainingHeaders::iterator it2 = headers2.begin();
    SCM list2 = STk_nil;


    /*
    while((it1 != nil) &&
            (it2 != nil) &&
            (headers1[it1] != headers2[it2])) {

        if(inf(headers1[it1]).dfsNum > inf(headers2[it2]).dfsNum) {
            list1 = STk_cons(STk_makeinteger(inf(headers1[it1]).pdgVertexId),
                    list1);
            it1 = headers1.succ(it1);
        }
        else {
            list2 = STk_cons(STk_makeinteger(inf(headers2[it2]).pdgVertexId),
                    list2);
            it2 = headers2.succ(it2);
        }
    }
    */
    while((it1 != headers1.end()) &&
            (it2 != headers2.end()) &&
            (*it1 != *it2)) {

        if(inf(*it1).dfsNum > inf(*it2).dfsNum) {
            list1 = STk_cons(STk_makeinteger(inf(*it1).pdgVertexId),
                    list1);
            it1++;
        }
        else {
            list2 = STk_cons(STk_makeinteger(inf(*it2).pdgVertexId),
                    list2);
            it2++ ;
        }
    }

    //REP_LEDA_SLIST
    /*
    if ((it1 != nil) && (it2 == nil)) {
        do {
            list1 = STk_cons(STk_makeinteger(inf(headers1[it1]).pdgVertexId),
                    list1);
            it1 = headers1.succ(it1);
        } while(it1 != nil);
    }
    else if ((it1 == nil) && (it2 != nil)) {
        do {
            list2 = STk_cons(STk_makeinteger(inf(headers2[it2]).pdgVertexId),
                    list2);
            it2 = headers1.succ(it2);
        } while(it2 != nil);
    }
    */
    if ((it1 != headers1.end()) && (it2 == headers2.end())) {
        do {
            list1 = STk_cons(STk_makeinteger(inf(*it1).pdgVertexId),
                    list1);
            it1++;
        } while(it1 != headers1.end());
    }
    else if ((it1 == headers1.end()) && (it2 != headers2.end())) {
        do {
            list2 = STk_cons(STk_makeinteger(inf(*it2).pdgVertexId),
                    list2);
            it2++ ;
        } while(it2 != headers2.end());
    }

    //REP_LEDA_SLIST
    return STk_cons(list1, list2);
}

int
Hash(const CFG::VertexIDPair &pair) {
    return pair.from<<(sizeof(int)*4) + pair.to;
}

bool
operator==(const CFG::VertexIDPair &pair1,
	   const CFG::VertexIDPair &pair2)
{
    return (pair1.from == pair2.from) && (pair1.to == pair2.to);
}


void
Loop::Print(ostream &out) {

    out << "<LOOP header=" << cfg.inf(header).pdgVertexId;

    out << " tails=(";
    
    //REP_LEDA_SLIST
    /*
    for(slist_item it = backEdges.first(); it != nil; it = backEdges.succ(it))
        out << cfg[cfg.source(backEdges[it])].pdgVertexId << ' ';
    */
    
    
    list<edge>::const_iterator ite;

    for(ite = backEdges.begin(); ite != backEdges.end(); ite++) {
        out << cfg.inf(source(*ite)).pdgVertexId << ' ';
    }
    
    
    //REP_LEDA_SLIST
    
    out << ") ";

    out << "body=(";
    node v;

    //PROBLEM !!!!!!!!!
    //forall_defined(v, *((Leda_map<node, bool> *)this)) {
	
    for(map<node, bool>::iterator ite = (*this).begin(); 
            ite!=(*this).end();
            ite++) {

        if((*this)[v])
            out << cfg.inf(v).pdgVertexId << ' ';
    }
    out << ")>" << endl;
}

Loop::Loop(node header_a, list<edge> backEdges_a,
	   CFG &cfg_a)
  : map<node, bool>(), cfg(cfg_a),
    header(header_a), backEdges(backEdges_a) {

    (*this)[header] = true;
    
    edge currBackEdge;

    //REP_LEDA_SLIST
    
  /*  forall(currBackEdge, backEdges)  */
    list<edge>::const_iterator ite;
    for(ite = backEdges.begin(); ite != backEdges.end(); ite++) {
        currBackEdge = *ite;
        node currTail = source(currBackEdge);
        stack<node> stack;
        stack.push(currTail);

        while(!stack.empty()) {
            node currNode = stack.top();
            stack.pop();
            
            //REP_LEDA_MAP
            //(*this)[currNode] = true;

            map<node, bool>::iterator ite = (*this).find(currNode);
            if(ite == (*this).end()) {
                (*this).insert(value_type(currNode, true));
            }
            else {
                (*ite).second = true;
            }


            edge inEdge;
            //REP_LEDA_GRAPH
            edge_iterator inEdge_ite;
            //forall_in_edges(inEdge, currNode) {
            for_all_in_edges(inEdge_ite, currNode) {
                inEdge = *inEdge_ite;
                node predNode = source(inEdge);
                //if(!(*this)[predNode])
                //REP_LEDA_MAP
                bool tt;
                map<node, bool>::iterator ite2 = (*this).find(predNode);
                if(ite2 == (*this).end())
                    tt = false;
                else
                    tt = (*ite2).second;
                if(!tt)
                    stack.push(predNode);
            }
        }
    }
}

static void
MarkCFG(SCM pCFG) {
    ((CFG *) EXTDATA(pCFG))->Mark();
}

static void
FreeCFG(SCM pCFG) {
    delete ((CFG *) (EXTDATA(pCFG)));
}

static STk_extended_scheme_type cfgType = {
    "CFG",
    0,
    MarkCFG,
    FreeCFG,
    NULL,
    NULL
};

static PRIMITIVE
MakeCFG(SCM enterVertexId_a, SCM listOfEdges, SCM name_a) {
    SCM z;

    NEWCELL(z, tc_CFG);
    EXTDATA(z) = new CFG(STk_integer_value_no_overflow(enterVertexId_a),
			 listOfEdges, name_a);
    return z;
}

static PRIMITIVE
ComputeReachability(SCM pCFG, SCM listOfDepEdges) {
    return ((CFG *) EXTDATA(pCFG))->ComputeReachability(listOfDepEdges);
}

static PRIMITIVE
ComputeControlLabels(SCM pCFG, SCM listOfControlEdges) {
    return ((CFG *) EXTDATA(pCFG))->ComputeControlLabels(listOfControlEdges);
}

extern "C"
PRIMITIVE STk_init_libreachability() {

    tc_CFG = STk_add_new_type(&cfgType);

    STk_add_new_primitive("make-cfg", tc_subr_3,
			  ((PRIMITIVE (*)(...))&MakeCFG));
    STk_add_new_primitive("compute-reachability", tc_subr_2,
			  ((PRIMITIVE (*)(...))ComputeReachability));
    STk_add_new_primitive("compute-control-labels", tc_subr_2,
			  ((PRIMITIVE (*)(...))ComputeControlLabels));

    return UNDEFINED;
}
