// Copyright (c) 2003 Raghavan Komondoor and Susan Horwitz

// This file is part of DUP.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software DUP and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include <clones.h>
#include <fstream.h>
#include <stk.h>
#include <list>
#include <map>
#include "graph/template_undirected.cpp"
/* The sole externally visible function in this file is GroupPairs. It is
   known in Scheme as group-pairs. 

*/

typedef UGraphNode * node;
typedef UGraphEdge * edge;

#define nil NULL

void DFS(UGraphNode *n, set<UGraphNode *>& S) {

    //visit n
    S.insert(set<UGraphNode *>::value_type(n));
    
    UGraphNode::EdgeIterator ite;
	UGraphNode * neighbor;
    for_all_adj_edges(ite, n) {
        
		if(NODE1(*ite) == n)
            neighbor = NODE2(*ite);
        else
            neighbor = NODE1(*ite);
        //if eighbor is unvisited
        if( S.find(neighbor) == S.end())
            DFS(neighbor, S);
	}
}

template<class NodeType, class EdgeType>
int
COMPONENT(UGraph<NodeType, EdgeType> & graph, map<UGraphNode *, int>& compnum) {

    typedef typename UGraph<NodeType, EdgeType>::NodeIterator ugraph_node_ite;
    set<UGraphNode *> unmarked;
    int c = -1;
    ugraph_node_ite ite;
    for_all_nodes(ite, graph) {
        unmarked.insert(set<UGraphNode *>::value_type(*ite));
        compnum.insert(map<UGraphNode *, int>::value_type(*ite, -1));
    }
    set<UGraphNode *> S;   
    while (!unmarked.empty()){
        c++;
        S.clear();
        //get the first one as n
        UGraphNode * n = *(unmarked.begin());
        DFS(n, S);
        for(set<UGraphNode *>::iterator m = S.begin(); m != S.end(); m++) {
            
            //mark m
            unmarked.erase(*m);
            compnum[*m]=c;
        }
    }
    return c+1;  //since LEDA's component returns the number of components
}

struct NodeOf {
    ClonePair *clonePair;
    node lNode;
    NodeOf(ClonePair *clonePair_a, node lNode_a) :
        clonePair(clonePair_a), lNode(lNode_a) {}
    NodeOf() {} // this constructor is only for leda
};

istream &
operator>>(istream &in, NodeOf &) {
    return in;
}

ostream &
operator<<(ostream &out, const NodeOf &) {
    return out;
}

struct CSGNodeInfo {
    ClonePair *clonePair;
    bool firstMark; // first clone in clonePair has already been added
		    // to a group.
    CloneGroup::PosInGroup firstPosInGroup;
    // firstPosInGroup holds meaningful data only when firstMark is
    // true. It is the position in the clone group.
    bool secondMark;
    CloneGroup::PosInGroup secondPosInGroup;
public:
    CSGNodeInfo(ClonePair *clonePair_a) :
	clonePair(clonePair_a), firstMark(false), secondMark(false) {}
    CSGNodeInfo() {} // only for leda
};

istream &
operator>>(istream &in, CSGNodeInfo &) {
    return in;
}

ostream &
operator<<(ostream &out, const CSGNodeInfo &) {
    return out;
}

class CSGEdgeInfo {
    node whichNodePerspective;
    WhichComponentPair how;

    public:
    CSGEdgeInfo(node whichNodePerspective_a,
            WhichComponentPair how_a) :
        whichNodePerspective(whichNodePerspective_a),
    how(how_a) {}
    CSGEdgeInfo() {} // just for Leda
    WhichComponentPair FromPerspective(node whichNodePerspective_a) {
        if(whichNodePerspective_a == whichNodePerspective)
            return how;
        else
            return WhichComponentPair(how.to, how.from);
    }
};

istream &
operator>>(istream &in, CSGEdgeInfo &) {
    return in;
}

ostream &
operator<<(ostream &out, const CSGEdgeInfo &) {
    return out;
}

template class UGraph<CSGNodeInfo, CSGEdgeInfo>;
template class Graph<CSGNodeInfo, CSGEdgeInfo>;

static PRIMITIVE
GroupPairs(SCM listOfClonePairs, SCM outfile) {
    /*
       listOfClonePairs is a list of clone pairs in Scheme syntax (obtained
       from the output of the clone-detection tool / filtering tool). Although
       this is not a requirement of this function, it makes sense for all given
       clones to be of the same size (both in terms of number of nodes, and in
       terms of number of PDG edges present in the partial backward slice).

       Say (A, B) and (C, D) are of two of the clone pairs in
       listOfClonePairs. A is identical to C iff:

       - A and C are from the same procedure

       - both have identical roots

       - both are identical as sets of nodes

       - for any node v in A / C, the set of dependence parents of v within A
       (considering only the PDG edges present in the partial slice
       corresponding to A) is equal to the set of dependence parents of v
       within C (considering only the PDG edges present in the partial
       slice corresponding to C).

       Similarly, A could be identical to D, B could be identical to C, or B
       could be identical to D. The pair (A, B) is said to "overlap" the pair
       (C, D) if a clone in one of the pairs is identical to a clone in the
       other pair. In that case the two pairs can be combined into a single
       group (of three clones).

       What this function does is:

       - partition the clone pairs in listOfClonePairs, such that within any
       partition each clone pair is related to every other clone pair by
       the transitive closure of the "overlap" relation.

       - collapse each partition into a group of clones by eliminating
       duplicate copies of identical clones.

       - write out the groups into "outfile", as Scheme lists.

       - return a list of integers that is as long as the final number of
       groups, with each element of the list being the number of (unique)
       clones in a group.

*/
    
    
    list<NodeOf> clonePairs;
    typedef list<NodeOf>::iterator clonePairsIterator;
    UGraph<CSGNodeInfo, CSGEdgeInfo> cloneShareGraph;
    ofstream out(CHARS(outfile));

    // create all the ClonePair objects
    SCM currClonePairPtr = listOfClonePairs;
    while(currClonePairPtr != STk_nil) {
        ClonePair *clonePair = new ClonePair(STk_car(currClonePairPtr));

        clonePairs.push_back
            (NodeOf(clonePair, cloneShareGraph.new_node
                        (CSGNodeInfo(clonePair))));

        currClonePairPtr = STk_cdr(currClonePairPtr);
    }

    // insert the edges into cloneShareGraph
    for(clonePairsIterator first_ite = clonePairs.begin(); first_ite != clonePairs.end(); first_ite++ ) {


        clonePairsIterator first_ite_cpy = first_ite;
        for(clonePairsIterator second = (++first_ite_cpy); second != clonePairs.end(); second++) {

            WhichComponentPair how;
            if((*first_ite).clonePair->
                    SharesClone(*((*second).clonePair), how)) {
                cloneShareGraph.new_edge
                    ((*first_ite).lNode,
                     (*second).lNode,
                     CSGEdgeInfo((*first_ite).lNode, how)
                    );
            }
        }
    }


    // find the connected components
    map<node, int> componentNumOf;
        

    int numComponents = COMPONENT(cloneShareGraph, componentNumOf);

    // build the list of nodes in each connected component
    list<node> *nodesInComponent = new list<node>[numComponents];
    
    int *sizeOfComponent = new int[numComponents];
    for(int compNum = 0; compNum < numComponents; ++compNum)
        sizeOfComponent[compNum] = 0;
    node lNode;
        
    UGraph<CSGNodeInfo, CSGEdgeInfo>::NodeIterator node_ite;
    for_all_nodes(node_ite, cloneShareGraph) {
        lNode = *node_ite;
        nodesInComponent[componentNumOf[lNode]].push_back(lNode);
        sizeOfComponent[componentNumOf[lNode]]++;
    }

    map<Node *, CSGNodeInfo*> &nodesInfo = cloneShareGraph.node_data();
    SCM retVal = STk_nil;
    // Iterate through all connected components. For each component, find the
    // number of unique clones in it.
    for(int compNum = 0; compNum < numComponents; ++compNum) {
        int numUniqueClones = 0;
        int numMarksTrued = 0; // We don't have to use numMarksTrued. We can
        // simply go through all nodes in nodesInComponent[compNum] but that
        // would be more inefficient.

        // Iterate through all nodes in the clone share graph
        // connected component (i.e., clone group) and determine the
        // set of all clones in it.
        CloneGroup clonesInGroup;
        list<node> &nodes = nodesInComponent[compNum];
        node currGraphNode;
        for(list<node>::iterator ite = nodes.begin();
	    (numMarksTrued < sizeOfComponent[compNum]*2)
		&& (ite != nodes.end());
	    ite++) {
            currGraphNode = *ite;
	    CSGNodeInfo &currGraphNodeInfo = *(nodesInfo[currGraphNode]);

	    // Of the two clones in the current graph node, add the
	    // ones that are not already in the current clone group
	    // to the group.
	    if (!currGraphNodeInfo.firstMark &&
		!currGraphNodeInfo.secondMark) {
		// neither clone is in the group. therefore, add both.
		pair<CloneGroup::PosInGroup, CloneGroup::PosInGroup>
		    positions = clonesInGroup.
		    addBothClones(nodesInfo[currGraphNode]->clonePair);
		currGraphNodeInfo.firstPosInGroup = positions.first;
		currGraphNodeInfo.secondPosInGroup = positions.second;
		numUniqueClones += 2;
	    }
	    else if (!currGraphNodeInfo.firstMark) {
		// second clone already in group. therefore, add first.
		currGraphNodeInfo.firstPosInGroup =
		    clonesInGroup.
		    addOneClone(currGraphNodeInfo.clonePair,
				first,
				currGraphNodeInfo.secondPosInGroup);
		numUniqueClones++;
	    }
	    else if (!currGraphNodeInfo.secondMark) {
		// first clone already in group. therefore, add second.
		currGraphNodeInfo.secondPosInGroup =
		    clonesInGroup.
		    addOneClone(currGraphNodeInfo.clonePair,
				second,
				currGraphNodeInfo.firstPosInGroup);
		numUniqueClones++;
	    }
	    currGraphNodeInfo.firstMark = currGraphNodeInfo.secondMark = true;
		
	    // Go through all graph nodes adjacent to
	    // currGraphNode. If the ith clone (i = 1 | 2) in the
	    // clone pair of an adjacent node is the same as the jth
	    // clone in currGraphNode (j = 1 | 2), then set the ith
	    // mark of the adjacent node to true, and set
	    // <i>PosInGroup of the adhjacent node to <j>PosInGroup of
	    // currGraphNode.
	    UGraphNode::EdgeIterator ite;
	    for_all_adj_edges(ite, currGraphNode) {

		edge adjEdge = *ite;   
		WhichComponentPair
		    how(cloneShareGraph.inf(adjEdge).
			FromPerspective(currGraphNode));
		// how.from is j (see above comment). how.to is i.
		node adjNode =
		    OPPOSITE(currGraphNode, adjEdge);
		switch(how.to) {
		case first:
		    if (!nodesInfo[adjNode]->firstMark) {
			nodesInfo[adjNode]->firstMark = true;
			numMarksTrued++;
			nodesInfo[adjNode]->firstPosInGroup =
			    (how.from == first) ?
			    currGraphNodeInfo.firstPosInGroup :
			    currGraphNodeInfo.secondPosInGroup;
		    }
		    break;
		case second:
		    if (!nodesInfo[adjNode]->secondMark) {
			nodesInfo[adjNode]->secondMark = true;
			numMarksTrued++;
			nodesInfo[adjNode]->secondPosInGroup =
			    (how.from == first) ?
			    currGraphNodeInfo.firstPosInGroup :
			    currGraphNodeInfo.secondPosInGroup;
		    }
		}
	    }
        }
	
        // Write out the clones in clonesInGroup
        clonesInGroup.WriteSTkSexpr(out);

        // numUniqueClones has the number of unique clones in the current
        // clone group.
        retVal = STk_cons(STk_makeinteger(numUniqueClones), retVal);
    }

    // free up memory

    for(clonePairsIterator ite = clonePairs.begin(); ite != clonePairs.end(); ite++) {
        delete (*ite).clonePair;
    }


    delete[] nodesInComponent;
    delete[] sizeOfComponent;

    out.close();
    return retVal;
}

extern "C"
PRIMITIVE
STk_init_libgroup_pairs() {

    STk_add_new_primitive("group-pairs", tc_subr_2,
            ((PRIMITIVE (*)(...))GroupPairs));

    return UNDEFINED;
}
