package org.jallinone.events.server;

import java.sql.Connection;
import org.jallinone.system.server.JAIOUserSessionParameters;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.ServletContext;

/**
 * <p>Title: JAllInOne ERP/CRM application</p>
 * <p>Description: Event fired by a server-side business object.</p>
 * <p>Copyright: Copyright (C) 2006 Mauro Carniel</p>
 *
 * <p> This file is part of JAllInOne ERP/CRM application.
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the (LGPL) Lesser General Public
 * License as published by the Free Software Foundation;
 *
 *                GNU LESSER GENERAL PUBLIC LICENSE
 *                 Version 2.1, February 1999
 *
 * This application is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *       The author may be contacted at:
 *           maurocarniel@tin.it</p>
 *
 * @author Mauro Carniel
 * @version 1.0
 */
public class GenericEvent {

  /** event fired by the business object as the first instruction, after creating the Connection */
  public static final int CONNECTION_CREATED = 0;

  /** event fired by the business object when finishing the processing, just before commit the transaction */
  public static final int BEFORE_COMMIT = 1;

  /** event fired by the business object when finishing the processing, just after commit the transaction */
  public static final int AFTER_COMMIT = 2;



  /** object that fires this event (an action class or helper bean) */
  private Object source;

  /** request name of an action class; it may be null for helper bean classes */
  private String requestName;

  /** event type; possible values: CONNECTION_CREATED, BEFORE_COMMIT, AFTER_COMMIT */
  private int eventType;

  /** user parameters, stored in the servlet session */
  private JAIOUserSessionParameters userSessionPars;

  /** HTTP request */
  private HttpServletRequest httpRequest;

  /** HTTP response */
  private HttpServletResponse httpResponse;

  /** user session */
  private HttpSession userSession;

  /** ServletContext */
  private ServletContext context;

  /** connection used by the business object */
  private Connection conn;

  /** input object passed to the business object from the client-side or from another server-side bean */
  private Object inputPar;

  /** response generated by the business object after commit the transaction; this is null for CONNECTION_CREATED event */
  private Object response;

  /** (optional) error message that rule engine can return */
  private String errorMessage;


  public GenericEvent(
      Object source,
      String requestName,
      int eventType,
      JAIOUserSessionParameters userSessionPars,
      HttpServletRequest httpRequest,
      HttpServletResponse httpResponse,
      HttpSession userSession,
      ServletContext context,
      Connection conn,
      Object inputPar,
      Object response) {
    this.source = source;
    this.requestName = requestName;
    this.eventType = eventType;
    this.userSessionPars = userSessionPars;
    this.conn = conn;
    this.httpRequest = httpRequest;
    this.httpResponse = httpResponse;
    this.userSession = userSession;
    this.context = context;
    this.inputPar = inputPar;
    this.response = response;
  }


  /**
   * @return object that fires this event (an action class or helper bean)
   */
  public final Object getSource() {
    return source;
  }


  /**
   * @return event type; possible values: CONNECTION_CREATED, BEFORE_COMMIT, AFTER_COMMIT
   */
  public final int getEventType() {
    return eventType;
  }


  /**
   * @return connection used by the business object
   */
  public final Connection getConn() {
    return conn;
  }


  /**
   * @return input object passed to the business object from the client-side or from another server-side bean
   */
  public final Object getInputPar() {
    return inputPar;
  }


  /**
   * @return response generated by the business object after commit the transaction; this is null for CONNECTION_CREATED event
   */
  public final Object getResponse() {
    return response;
  }


  /**
   * @return request name of an action class; it may be null for helper bean classes
   */
  public final String getRequestName() {
    return requestName;
  }


  /**
   * @return user parameters, stored in the servlet session
   */
  public final JAIOUserSessionParameters getUserSessionPars() {
    return userSessionPars;
  }


  /**
   * @return ServletContext
   */
  public ServletContext getContext() {
    return context;
  }


  /**
   * @return HTTP request
   */
  public final HttpServletRequest getHttpRequest() {
    return httpRequest;
  }


  /**
   * @return user session
   */
  public final HttpSession getUserSession() {
    return userSession;
  }


  /**
   * @return HTTP response
   */
  public final HttpServletResponse getHttpResponse() {
    return httpResponse;
  }


  /**
   * @return (optional) error message that rule engine can return
   */
  public final String getErrorMessage() {
    return errorMessage;
  }


  /**
   * Method invoked from the rule engine.
   * @param errorMessage (optional) error message that rule engine can return
   */
  public final void setErrorMessage(String errorMessage) {
    this.errorMessage = errorMessage;
  }

}
