package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.util.*;
import java.sql.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * This  class implements some administrative type
 * database routines used by the 'local' client on
 * first time startup or initialization
 */
public class Admin extends DBType {

	private static final String[] drops =
	{ 	"drop table delete_info",
		"drop table state_tax",
		"drop table role_permission",
		"drop table user_role",
		"drop table user_group_member",
		"drop table user_group",
		"drop table sellwin_order",
		"drop table quote_line",
		"drop table quote",
		"drop table attendee",
		"drop table activity",
		"drop table lead",
		"drop table campaign",
		"drop table customer_inventory",
		"drop table product",
		"drop table customer",
		"drop table sales_person",
		"drop table sales_person_role",
		"drop table forecast",
		"drop table contact",
		"drop table opportunity",
		"drop table address"
	};

	private static final String[] creates = {
		"CREATE TABLE delete_info (" +
    		"pk bigint(15) NOT NULL PRIMARY KEY, "+
    		"class_name VARCHAR(25) NOT NULL)",

		"CREATE TABLE state_tax (" +
    		"state_code          VARCHAR(2) NOT NULL PRIMARY KEY,"+
    		"state_full_name     VARCHAR(30),"+
    		"tax_pct             decimal(6,5) NOT NULL,"+
    		"modified_by         VARCHAR(25) NOT NULL,"+
    		"modified_date       DATE NOT NULL)",

		"CREATE TABLE address ("+
    		"pk          bigint(15) NOT NULL PRIMARY KEY,"+
    		"first_name          VARCHAR(25),"+
    		"last_name           VARCHAR(25),"+
    		"middle_initial      VARCHAR(20),"+
    		"address_line1       VARCHAR(80),"+
    		"address_line2       VARCHAR(80),"+
    		"title               VARCHAR(7),"+
    		"phone               VARCHAR(25),"+
    		"fax                 VARCHAR(25),"+
    		"pager               VARCHAR(25),"+
    		"cell                VARCHAR(25),"+
    		"email               VARCHAR(80),"+
    		"city                VARCHAR(25),"+
    		"state               VARCHAR(25),"+
    		"country             VARCHAR(25),"+
    		"zip                 VARCHAR(5),"+
    		"zip4                VARCHAR(4), "+
    		"modified_by         VARCHAR(25) NOT NULL,"+
    		"modified_date       DATE NOT NULL)",

		"CREATE TABLE sales_person ("+
			"pk                  bigint(15) NOT NULL PRIMARY KEY,"+
			"id                  VARCHAR(25) NOT NULL UNIQUE,"+
			"password            VARCHAR(25) NOT NULL,"+
			"address_pk          bigint(15) NOT NULL,"+
			"modified_by         VARCHAR(25) NOT NULL,"+
			"modified_date       DATE NOT NULL,"+
			"sync_date           DATE,"+
			"CONSTRAINT SPFK1 FOREIGN KEY (address_pk) REFERENCES address (pk))",

		"CREATE TABLE customer ( "+
			"pk                  bigint(15) NOT NULL PRIMARY KEY, "+
			"name                VARCHAR(25) NOT NULL UNIQUE, "+
			"ship_address_pk     bigint(15) NOT NULL, "+
			"bill_address_pk     bigint(15) NOT NULL, "+
			"annual_sales        INTEGER NOT NULL, "+
			"industry            VARCHAR(25), "+
			"channel             VARCHAR(25), "+
			"contact_pk          bigint(15) NOT NULL, "+
			"modified_by         VARCHAR(25) NOT NULL, "+
			"modified_date       DATE NOT NULL, "+
			"CONSTRAINT CUSTFK1 FOREIGN KEY (ship_address_pk) REFERENCES address (pk), "+
			"CONSTRAINT CUSTFK2 FOREIGN KEY (bill_address_pk) REFERENCES address (pk), "+
			"CONSTRAINT CUSTFK3 FOREIGN KEY (contact_pk) REFERENCES address (pk))",

		"CREATE TABLE product (" +
			"pk                  bigint(15) NOT NULL PRIMARY KEY," +
			"product_group       VARCHAR(25) NOT NULL," +
			"line                VARCHAR(25) NOT NULL," +
			"name                VARCHAR(25) NOT NULL," +
			"model_no            VARCHAR(25) NOT NULL," +
			"cost                DECIMAL(12,2) NOT NULL," +
			"price               DECIMAL(12,2) NOT NULL," +
			"description         VARCHAR(128) NOT NULL," +
			"modified_by         VARCHAR(25) NOT NULL," +
			"modified_date       DATE NOT NULL)",

		"CREATE TABLE customer_inventory ( "+
			"pk                  bigint(15) NOT NULL PRIMARY KEY, "+
			"customer_pk         bigint(15) NOT NULL, "+
			"product_pk          bigint(15) NOT NULL, "+
			"cnt                 INTEGER NOT NULL, "+
			"modified_by         VARCHAR(25) NOT NULL, "+
			"modified_date       DATE NOT NULL, "+
			"CONSTRAINT CIFK1 FOREIGN KEY (customer_pk) REFERENCES customer (pk), "+
			"CONSTRAINT CIFK2 FOREIGN KEY (product_pk) REFERENCES product (pk))",

		"CREATE TABLE campaign (" +
			"pk                  bigint(15) NOT NULL PRIMARY KEY," +
			"name                VARCHAR(25) NOT NULL," +
			"type                VARCHAR(25) NOT NULL," +
			"product_group       VARCHAR(25) NOT NULL," +
			"product_line        VARCHAR(25) NOT NULL," +
			"lead_count          INTEGER NOT NULL," +
			"modified_by         VARCHAR(25) NOT NULL," +
			"modified_date       DATE NOT NULL)",

		"CREATE TABLE lead (" +
			"pk                  bigint(15) NOT NULL PRIMARY KEY," +
			"campaign_pk         bigint(15) NOT NULL UNIQUE," +
			"address_pk          bigint(15) NOT NULL," +
			"company             VARCHAR(25) NOT NULL," +
			"interest_level      VARCHAR(25) NOT NULL," +
			"contacted_by_pk     bigint(15) NOT NULL," +
			"contact_date        DATE NOT NULL," +
			"converted_opp       CHAR NOT NULL," +
			"modified_by         VARCHAR(25) NOT NULL," +
			"modified_date       DATE NOT NULL," +
			"CONSTRAINT LEADFK1 FOREIGN KEY (campaign_pk) REFERENCES campaign (pk)," +
			"CONSTRAINT LEADFK2 FOREIGN KEY (address_pk) REFERENCES address (pk)," +
			"CONSTRAINT LEADFK3 FOREIGN KEY (contacted_by_pk) REFERENCES sales_person (pk))",

		"CREATE TABLE opportunity ("+
			"pk                  bigint(15) NOT NULL PRIMARY KEY,"+
			"name                VARCHAR(80) NOT NULL,"+
			"dollar_value        INTEGER NOT NULL,"+
			"probability         VARCHAR(25) NOT NULL,"+
			"stage               VARCHAR(25) NOT NULL,"+
			"lead_source         VARCHAR(25) NOT NULL,"+
			"lead_type           VARCHAR(25) NOT NULL,"+
			"description         VARCHAR(125) NOT NULL,"+
			"group_name          VARCHAR(25) NOT NULL,"+
			"prime_sp_pk         bigint(15),"+
			"cust_pk             bigint(15),"+
			"close_date          DATE NOT NULL,"+
			"lead_pk             bigint(15),"+
			"modified_by         VARCHAR(25) NOT NULL,"+
			"modified_date       DATE NOT NULL)",

		"CREATE TABLE forecast ("+
			"pk                  bigint(15) NOT NULL PRIMARY KEY,"+
			"opp_pk              bigint(15) NOT NULL,"+
			"name                VARCHAR(25) NOT NULL ,"+
			"close_date          DATE NOT NULL,"+
			"submitted           CHAR NOT NULL,"+
			"submitted_date      DATE NOT NULL,"+
			"submitted_by        VARCHAR(25) NOT NULL,"+
			"revenue             INTEGER NOT NULL,"+
			"profit              INTEGER NOT NULL,"+
			"margin              DECIMAL(3,2) NOT NULL,"+
			"scenario            VARCHAR(25) NOT NULL,"+
			"amount_jan          INTEGER NOT NULL,"+
			"amount_feb          INTEGER NOT NULL,"+
			"amount_mar          INTEGER NOT NULL,"+
			"q1                  INTEGER NOT NULL,"+
			"amount_apr          INTEGER NOT NULL,"+
			"amount_may          INTEGER NOT NULL,"+
			"amount_jun          INTEGER NOT NULL,"+
			"q2                  INTEGER NOT NULL,"+
			"amount_jul          INTEGER NOT NULL,"+
			"amount_aug          INTEGER NOT NULL,"+
			"amount_sep          INTEGER NOT NULL,"+
			"q3                  INTEGER NOT NULL,"+
			"amount_oct          INTEGER NOT NULL,"+
			"amount_nov          INTEGER NOT NULL,"+
			"amount_dec          INTEGER NOT NULL,"+
			"q4                  INTEGER NOT NULL,"+
			"modified_by         VARCHAR(25) NOT NULL,"+
			"modified_date       DATE NOT NULL,"+
			"CONSTRAINT FORFK1 FOREIGN KEY (opp_pk) REFERENCES opportunity (pk))",

		"CREATE TABLE contact ( "+
			"pk                  bigint(15) NOT NULL PRIMARY KEY, "+
			"opp_pk              bigint(15) NOT NULL, "+
			"address_pk          bigint(15) NOT NULL, "+
			"modified_by         VARCHAR(25) NOT NULL, "+
			"modified_date       DATE NOT NULL, "+
			"CONSTRAINT CONTFK1 FOREIGN KEY (opp_pk) REFERENCES opportunity (pk), "+
			"CONSTRAINT CONTFK2 FOREIGN KEY (address_pk) REFERENCES address (pk))",

		"CREATE TABLE activity ( "+
			"pk                  bigint(15) NOT NULL PRIMARY KEY, "+
			"opp_pk              bigint(15) NOT NULL, "+
			"subject             VARCHAR(45) NOT NULL, "+
			"type                VARCHAR(25) NOT NULL, "+
			"place               VARCHAR(120) NOT NULL, "+
			"message             VARCHAR(255) NOT NULL, "+
			"start_date          DATE NOT NULL,  "+
			"duration            int NOT NULL, "+
			"group_alarm         CHAR NOT NULL, "+
			"group_alarm_ack     CHAR NOT NULL, "+
			"alarm               CHAR NOT NULL, "+
			"alarm_ack           CHAR NOT NULL, "+
			"modified_by         VARCHAR(25) NOT NULL, "+
			"modified_date       DATE NOT NULL, "+
			"CONSTRAINT ACTFK1 FOREIGN KEY (opp_pk) REFERENCES opportunity (pk))", 

		"CREATE TABLE attendee ( "+
			"pk                  bigint(15) NOT NULL PRIMARY KEY, "+
			"activity_pk         bigint(15) NOT NULL, "+
			"person_pk           bigint(15) NOT NULL, "+
			"modified_by         VARCHAR(25) NOT NULL, "+
			"modified_date       DATE NOT NULL, "+
			"CONSTRAINT ATTFK1 FOREIGN KEY (activity_pk) REFERENCES activity (pk), "+
			"CONSTRAINT ATTFK2 FOREIGN KEY (person_pk) REFERENCES sales_person (pk))",

		"CREATE TABLE quote ("+
			"pk                  bigint(15) NOT NULL PRIMARY KEY,"+
			"opp_pk              bigint(15) NOT NULL,"+
			"name                VARCHAR(25) NOT NULL,"+
			"modified_by         VARCHAR(25) NOT NULL,"+
			"modified_date       DATE NOT NULL,"+
			"CONSTRAINT QUOFK1 FOREIGN KEY (opp_pk) REFERENCES opportunity (pk))",

		"CREATE TABLE sellwin_order ( "+
			"pk                  bigint(15) NOT NULL PRIMARY KEY, "+
			"opp_pk              bigint(15) NOT NULL, "+
			"order_number        bigint(15) NOT NULL, "+
			"quote_pk            bigint(15) NOT NULL, "+
			"ship_address_pk     bigint(15) NOT NULL, "+
			"bill_address_pk     bigint(15) NOT NULL, "+
			"status              VARCHAR(25) NOT NULL, "+
			"payment_form        VARCHAR(25) NOT NULL, "+
			"sub_total           decimal(10,2) NOT NULL, "+
			"tax                 decimal(10,2) NOT NULL, "+
			"shipping            decimal(10,2) NOT NULL, "+
			"total               decimal(10,2) NOT NULL, "+
			"modified_by         VARCHAR(25) NOT NULL, "+
			"modified_date       DATE NOT NULL, "+
			"CONSTRAINT ORDFK1 FOREIGN KEY (opp_pk) REFERENCES opportunity (pk), "+
			"CONSTRAINT ORDFK2 FOREIGN KEY (quote_pk) REFERENCES quote (pk), "+
			"CONSTRAINT ORDFK3 FOREIGN KEY (ship_address_pk) REFERENCES address (pk), "+
			"CONSTRAINT ORDFK4 FOREIGN KEY (bill_address_pk) REFERENCES address (pk))",

		"CREATE TABLE quote_line ("+
			"pk                  bigint(15) NOT NULL PRIMARY KEY,"+
			"quote_pk            bigint(15) NOT NULL,"+
			"product_pk          bigint(15) NOT NULL,"+
			"product_name        VARCHAR(25) NOT NULL,"+
			"product_model_no    VARCHAR(25) NOT NULL,"+
			"quantity            INTEGER NOT NULL,"+
			"cost                DECIMAL(12,2) NOT NULL,"+
			"unit_price          DECIMAL(12,2) NOT NULL,"+
			"total_price         DECIMAL(12,2) NOT NULL,"+
			"product_desc        VARCHAR(128) NOT NULL,"+
			"modified_by         VARCHAR(25) NOT NULL,"+
			"modified_date       DATE NOT NULL,"+
			"CONSTRAINT QULFK1 FOREIGN KEY (quote_pk) REFERENCES quote (pk),"+
			"CONSTRAINT QULFK2 FOREIGN KEY (product_pk) REFERENCES product (pk))",

		"CREATE TABLE user_group ("+
			"pk                  bigint(15) NOT NULL PRIMARY KEY,"+
			"name                VARCHAR(25) NOT NULL,"+
			"modified_by         VARCHAR(25) NOT NULL,"+
			"modified_date       DATE NOT NULL)",

		"CREATE TABLE user_group_member ( "+
			"pk                  bigint(15) NOT NULL PRIMARY KEY, "+
			"user_group_pk       bigint(15) NOT NULL, "+
			"user_pk             bigint(15) NOT NULL, "+
			"modified_by         VARCHAR(25) NOT NULL, "+
			"modified_date       DATE NOT NULL, "+
			"CONSTRAINT UGFK1 FOREIGN KEY (user_group_pk) REFERENCES user_group (pk), "+
			"CONSTRAINT UGFK2 FOREIGN KEY (user_pk) REFERENCES sales_person (pk))",

		"CREATE TABLE user_role ("+
			"pk                  bigint(15) NOT NULL PRIMARY KEY,"+
			"name                VARCHAR(128) NOT NULL UNIQUE,"+
			"modified_by         VARCHAR(25) NOT NULL,"+
			"modified_date       DATE NOT NULL)",

		"CREATE TABLE sales_person_role ( "+
			"pk                  bigint(15) NOT NULL PRIMARY KEY, "+
			"sp_pk               bigint(15) NOT NULL, "+
			"user_role_pk        bigint(15) NOT NULL, "+
			"modified_by         VARCHAR(25) NOT NULL, "+
			"modified_date       DATE NOT NULL)",

		"CREATE TABLE role_permission ( "+
			"pk                  bigint(15) NOT NULL PRIMARY KEY, "+
			"user_role_pk        bigint(15) NOT NULL, "+
			"name                VARCHAR(128) NOT NULL, "+
			"r               CHAR NOT NULL, "+
			"w               CHAR NOT NULL, "+
			"modified_by         VARCHAR(25) NOT NULL, "+
			"modified_date       DATE NOT NULL, "+
			"CONSTRAINT rpfk1 FOREIGN KEY (user_role_pk) REFERENCES user_role (pk))"


	};

	private final static String[] create_roles = {

		"INSERT INTO sellwin.user_role (pk, name, modified_by, modified_date) VALUES (1, 'Sales Rep', 'SYSTEM', CURRENT_DATE)",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (1, 1, 'Forecast', 'Y', 'Y', 'SYSTEM', CURRENT_DATE)",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (2, 1, 'Quote', 'Y', 'Y', 'SYSTEM', CURRENT_DATE)",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (3, 1, 'Activity', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (4, 1, 'Basic Admin', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (5, 1, 'User Admin', 'N', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (6, 1, 'Order', 'Y', 'Y', 'SYSTEM', CURRENT_DATE)", 

		"INSERT INTO user_role (pk, name, modified_by, modified_date) VALUES (2, 'Sales Mgr', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (7, 2, 'Forecast', 'N', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (8, 2, 'Quote', 'Y', 'N', 'SYSTEM', CURRENT_DATE)",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (9, 2, 'Activity', 'Y', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (10, 2, 'Basic Admin', 'Y', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (11, 2, 'User Admin', 'N', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (12, 2, 'Order', 'Y', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO user_role (pk, name, modified_by, modified_date) VALUES (3, 'Sales Support', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (13, 3, 'Forecast', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (14, 3, 'Quote', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (15, 3, 'Activity', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (16, 3, 'Basic Admin', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (17, 3, 'User Admin', 'Y', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (18, 3, 'Order', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO user_role (pk, name, modified_by, modified_date) VALUES (4, 'System Admin', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (19, 4, 'Forecast', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (20, 4, 'Quote', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (21, 4, 'Activity', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (22, 4, 'Basic Admin', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (23, 4, 'User Admin', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (24, 4, 'Order', 'Y', 'Y', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO user_role (pk, name, modified_by, modified_date) VALUES (5, 'Marketing Rep', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (25, 5, 'Forecast', 'Y', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (26, 5, 'Quote', 'Y', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (27, 5, 'Activity', 'Y', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (28, 5, 'Basic Admin', 'Y', 'N', 'SYSTEM', CURRENT_DATE) ",

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (29, 5, 'User Admin', 'N', 'N', 'SYSTEM', CURRENT_DATE) ", 

		"INSERT INTO sellwin.role_permission (pk, user_role_pk, name, r, w, modified_by, modified_date) VALUES (30, 5, 'Order', 'Y', 'N', 'SYSTEM', CURRENT_DATE) "
	};

	private final static String[] state_tax = {
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('ALABAMA','AL', 0.04,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('ALASKA','AK', 0.00,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('ARIZONA','AZ', 0.056,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('ARKANSAS','AR', 0.05125,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('CALIFORNIA','CA', 0.06,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('COLORADO','CO', 0.029,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('CONNECTICUT','CT', 0.06,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('DELAWARE','DE', 0.00,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('DISTRICT OF COLUMBIA','DC', 0.0575,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('FLORIDA','FL', 0.06,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('GEORGIA','GA', 0.04,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('HAWAII','HI', 0.04,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('IDAHO','ID', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('ILLINOIS','IL', 0.0625,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('INDIANA','IN', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('IOWA','IA', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('KANSAS','KS', 0.049,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('KENTUCKY','KY', 0.06,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('LOUISIANA','LA', 0.04,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('MAINE','ME', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('MARYLAND','MD', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('MASSACHUSETTS','MA', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('MICHIGAN','MI', 0.06,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('MINNESOTA','MN', 0.065,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('MISSISSIPPI','MS', 0.07,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('MISSOURI','MO', 0.04225,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('MONTANA','MT', 0.00,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('NEBRASKA','NE', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('NEVADA','NV', 0.0425,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('NEW HAMPSHIRE','NH', 0.00,'SYSTEM', CURRENT_DATE)",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('NEW JERSEY','NJ', 0.06,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('NEW MEXICO','NM', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('NEW YORK','NY', 0.04,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('NORTH CAROLINA','NC', 0.045,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('NORTH DAKOTA','ND', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('OHIO','OH', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('OKLAHOMA','OK', 0.045,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('OREGON','OR', 0.00,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('PENNSYLVANIA','PA', 0.06,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('RHODE ISLAND','RI', 0.07,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('SOUTH CAROLINA','SC', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('SOUTH DAKOTA','SD', 0.04,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('TENNESSEE','TN', 0.06,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('TEXAS','TX', 0.0625,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('UTAH','UT', 0.0475,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('VERMONT','VT', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('VIRGINIA','VA', 0.035,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('WASHINGTON','WA', 0.065,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('WEST VIRGINIA','WV', 0.06,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('WISCONSIN','WI', 0.05,'SYSTEM', CURRENT_DATE) ",
		"insert into state_tax (state_full_name, state_code, tax_pct, modified_by, modified_date)  values ('WYOMING','WY', 0.04,'SYSTEM', CURRENT_DATE) ",
	};

	private final static String[] office_products = {
		"insert into sellwin.product values (1, 'seating', 'Aerial', 'Size A - Aerial', 'a-1', 911.78, 888.55, 'Aerial Size A Chair', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (2, 'seating', 'Aerial', 'Size B - Aerial', 'a-2', 888.42, 777.55, 'Aerial Size B Chair', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (3, 'seating', 'Aerial', 'Size C - Aerial', 'a-3', 712.42, 727.82, 'Aerial Size C Chair', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (4, 'seating', 'Equity', 'Size A - Equity', 'ea-1', 345.42, 123.82, 'Equity Size A Chair', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (5, 'seating', 'Equity', 'Size B - Equity', 'eb-1', 145.42, 123.82, 'Equity Size B Chair', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (6, 'seating', 'Equity', 'Size C - Equity', 'ec-1', 45.42, 23.82, 'Equity Size C Chair', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (7, 'systems furniture', 'Act', 'Size A - Act', 'ac-1', 100.42, 75.82, 'Act Size A Chair', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (8, 'systems furniture', 'Act', 'Size B - Act', 'ab-8', 999.42, 787.82, 'Act Size B ', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (9, 'systems furniture', 'Act', 'Size B - Surface laminate - Act', 'ac-9', 199.42, 188.82, 'Act Surface Lam', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (10, 'systems furniture', 'R System', 'Size A - R System', 'sc-9', 101.42, 77.82, 'R System Size A', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (11, 'systems furniture', 'R System', 'Size B - R System', 'sc-9', 101.42, 77.82, 'R System Size B', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (12, 'systems furniture', 'R System', 'Size C - R System', 'sc-9', 101.42, 77.82, 'R System Size C', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (13, 'freestanding furniture', 'Eanes', 'Storage units', 'e-19', 652.42, 399.82, 'Eanes Stor Units', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (14, 'freestanding furniture', 'Eanes', 'Desks', 'e-29', 183.42, 99.82, 'Eanes Stor Units', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (15, 'freestanding furniture', 'Eanes', 'Conference table', 'e-39', 222.42, 100.82, 'Eanes Conf Table', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (16, 'freestanding furniture', 'Eanes', 'Side Table', 'e-49', 333.42, 111.82, 'Eanes Side Table', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (17, 'freestanding furniture', 'Burdine', 'Table', 'bur-1', 721.42, 100.82, 'Burdine Table', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (18, 'freestanding furniture', 'Burdine', 'Coffee Table', 'bur-2', 200.42, 100.02, 'Burdine Coff Table', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (19, 'freestanding furniture', 'Burdine', 'Desk', 'bur-3', 1721.42, 1100.82, 'Burdine Dsk', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (20, 'freestanding furniture', 'Burdine', 'Conf Table', 'bur-4', 721.42, 100.82, 'Burdine Table', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (21, 'accessories', 'Harry Miller', 'Keyboard', 'mil-5', 21.42, 10.82, 'Keyboard', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (22, 'accessories', 'Harry Miller', 'Shelves', 'mil-6', 31.42, 30.82, 'Shelves', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (23, 'accessories', 'Harry Miller', 'Paper Tray', 'mil-7', 41.42, 20.82, 'Paper Tray', 'SYSTEM', current_date) ",
		"insert into sellwin.product values (24, 'accessories', 'Harry Miller', 'CPU Holder', 'mil-8', 51.42, 19.82, 'CPU Holder', 'SYSTEM', current_date) "
	};

	private final static String[] test_data = {
		"INSERT INTO sellwin.address VALUES (1, 'Jeff', 'Roberts', 'A', '111 Spring Road', ' ', 'Mr.', '512-111-2222', ' ', ' ', ' ', 'test1998@yahoo.com', 'Dallas', 'TX', 'USA', '78732', ' ', 'SYSTEM', current_date) ",
		"INSERT INTO sellwin.address VALUES (12345, 'Jeff', 'McCormick', 'X', 'Ocean blvd ', ' ', 'Mr.', '512-111-2222', ' ', ' ', ' ', 'test1998@yahoo.com', 'Austin', 'TX', 'USA', '78732', ' ', 'SYSTEM', current_date) ",
		"INSERT INTO sellwin.sales_person VALUES (1, 'trial', 'trial', 1, 'SYSTEM', current_date, current_date) ",
		"insert into sellwin.campaign values (1, 'campaign1', 'camptype1', 'group1', 'line1', 8, 'SYSTEM', current_date) ",
		"insert into sellwin.lead values (1, 1, 12345, 'company1', 'interest1', 1, current_date, 'Y', 'SYSTEM', current_date) ",
		"insert into sellwin.user_group values (1, 'usergroup1', 'SYSTEM', current_date) ",
		"insert into sellwin.user_group_member values (1, 1, 1, 'SYSTEM', current_date) ",
		"insert into sellwin.sales_person_role values (1, 1, 4, 'SYSTEM', current_date) "
	};

	private Connection con;

	public Admin() {
	}

	/**
	 * construct specifying a particular db type
	 * @param dbType db type to assume
	 */
	public Admin(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a constructor that accepts an existing Connection
	 * to use for future operations
	 *
	 * @param con the Connection to use
	 */
	public Admin(Connection con) {
		this.con = con;
	}

	/**
	 * get the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public final void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}


	/**
	 * drop all tables
	 *
	 * @exception java.sql.SQLException 
	 */
	public final void dropAllTables() 
		throws SQLException {

		Statement stmt = null;

		try {

			stmt = con.createStatement();

			StringBuffer query = null;
			int updatedRows;

			LogWrite.write("about to drop " + drops.length);
			for (int i=0;i<drops.length;i++) {
				query = new StringBuffer();
				query.append(drops[i]);
				LogWrite.write(query.toString());
				if (Prefs.DEBUG) LogWrite.write(query.toString());
				try {
					updatedRows = stmt.executeUpdate(query.toString());
				} catch (SQLException e) {
					LogWrite.write(e.getMessage());
				}
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close(); 
			} catch (SQLException x) { }
		}
	}

	/**
	 * create all tables
	 *
	 * @exception java.sql.SQLException 
	 */
	public final void createAllTables() 
		throws SQLException {

		Statement stmt = null;

		try {

			stmt = con.createStatement();

			StringBuffer query = null;
			int updatedRows;

			LogWrite.write("about to create " + creates.length);
			for (int i=0;i<creates.length;i++) {
				query = new StringBuffer();
				query.append(creates[i]);
				LogWrite.write(query.toString());
				if (Prefs.DEBUG) LogWrite.write(query.toString());
				try {
					updatedRows = stmt.executeUpdate(query.toString());
				} catch (SQLException e) {
					LogWrite.write(e.getMessage());
				}
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close(); 
			} catch (SQLException x) { }
		}
	}


	/**	
	 * test to see if a table exists, this will tell us
	 * if we need to build the database or not
	 * @return true if the table exists, false if no table
	 */
	public final boolean testTable() {
		Statement stmt = null;
		ResultSet rs = null;
		String query = "select count(*) from sales_person";

		try {
			stmt = con.createStatement();
			rs = stmt.executeQuery(query);
			return true;
		} catch (SQLException e) {
			LogWrite.write(e);
			return false;
		} catch (Exception x) {
			LogWrite.write(x);
			return false;
		} finally {
		    try {
                if (rs != null) rs.close();
            } catch (SQLException y) { }
            try {
                if (stmt != null) stmt.close();
            } catch (SQLException z) { }
		}
	}

	public final int testTableData() {
		Statement stmt = null;
		ResultSet rs = null;
		String query = "select count(*) rupert from sales_person";

		try {
			stmt = con.createStatement();
			rs = stmt.executeQuery(query);
			rs.next();
			int ct = rs.getInt("rupert");
			return ct;
		} catch (SQLException e) {
			e.printStackTrace();
			LogWrite.write(e);
		} catch (Exception x) {
			LogWrite.write(x);
		} finally {
		    try {
                if (rs != null) rs.close();
            } catch (SQLException y) { }
            try {
                if (stmt != null) stmt.close();
            } catch (SQLException z) { }
		}

		return 0;
	}

	public final void loadTestData() 
		throws SQLException {

		Statement stmt = null;

		try {

			stmt = con.createStatement();

			StringBuffer query = null;
			int updatedRows;

			LogWrite.write("creating " + create_roles.length + " test roles");
			for (int i=0;i<create_roles.length;i++) {
				query = new StringBuffer();
				query.append(create_roles[i]);
				LogWrite.write(query.toString());
				try {
					updatedRows = stmt.executeUpdate(query.toString());
				} catch (SQLException e) {
					LogWrite.write(e.getMessage());
				}
			}
			LogWrite.write("creating " + state_tax.length + " test state tax");
			for (int i=0;i<state_tax.length;i++) {
				query = new StringBuffer();
				query.append(state_tax[i]);
				LogWrite.write(query.toString());
				try {
					updatedRows = stmt.executeUpdate(query.toString());
				} catch (SQLException e) {
					LogWrite.write(e.getMessage());
				}
			}
			LogWrite.write("creating " + test_data.length + " test data");
			for (int i=0;i<test_data.length;i++) {
				query = new StringBuffer();
				query.append(test_data[i]);
				LogWrite.write(query.toString());
				try {
					updatedRows = stmt.executeUpdate(query.toString());
				} catch (SQLException e) {
					LogWrite.write(e.getMessage());
				}
			}
			LogWrite.write("creating " + office_products.length + " test products");
			for (int i=0;i<office_products.length;i++) {
				query = new StringBuffer();
				query.append(office_products[i]);
				LogWrite.write(query.toString());
				try {
					updatedRows = stmt.executeUpdate(query.toString());
				} catch (SQLException e) {
					LogWrite.write(e.getMessage());
				}
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close(); 
			} catch (SQLException x) { }
		}
	}
		
}
