package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;
import java.util.ArrayList;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * This class implements the DBInterface for
 * the Campaign class objects which are stored
 * in the campaign database table.
 */
public class CampaignDB extends DBType implements DBInterface {
	private Connection con;

	private final static String selectAllQuery = 
			"SELECT " +
				"pk, name, type, product_group, " +
    			"product_line, lead_count," +
    			"modified_by, modified_date " +
			"FROM campaign";

	private final static String selectQuery = 
			"SELECT " +
				"name, type, product_group, " +
    			"product_line, lead_count," +
    			"modified_by, modified_date " +
			"FROM campaign " +
			"WHERE pk=";

	private final static String updateQuery = 
			"UPDATE campaign " +
			"SET  " ;

	private final static String insertQuery = 
			"INSERT INTO campaign VALUES ( ";

	private final static String deleteQuery = 
			"DELETE FROM campaign WHERE pk =";

	private final static String deleteAllQuery = 
			"DELETE FROM campaign ";


	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 *
	 */
	public CampaignDB() {
	}

	/**
	 * construct using a particular db
	 * @param dbType the database type
	 */
	public CampaignDB(int dbType) {
		DB_TYPE=dbType;
	}

	/**
	 * a constructor that accepts an existing Connection
	 * to use for future operations
	 *
	 * @param con the Connection to use
	 */
	public CampaignDB(Connection con) {
		this.con = con;
	}

	/**
	 * get the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single campaign row using the passed
	 * primary key
	 * 
	 * @param pk the primary key we are searching with
	 * @return the row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		Campaign campaign = new Campaign();
		long pkValue = ((Long)pk).longValue();
		campaign.setPK(pkValue);
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + pkValue;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			while (rs.next()) {
				i=1;
				campaign.setName(rs.getString(i)); i++;
				campaign.setType(rs.getString(i)); i++;
				campaign.setProductGroup(rs.getString(i)); i++;
				campaign.setProductLine(rs.getString(i)); i++;
				campaign.setLeadCount(new Integer(rs.getInt(i))); i++;
				campaign.setModifiedBy(rs.getString(i)); i++;
				campaign.setModifiedDate(rs.getDate(i)); 
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return campaign;
	}


	/**
	 * update a single campaign row using the passed
	 * Campaign object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param obj the object we are going to update with
	 * @exception java.sql.SQLException 
	 */
	public void updateRow(Object obj) 
		throws SQLException {

		Campaign campaign = (Campaign)obj;
		long pk = campaign.getPK();

		Statement stmt = null;
		StringBuffer query = new StringBuffer(updateQuery);

		try {
			stmt = con.createStatement();

			query.append("name=");
			query.append(JDBC.quoteMore(campaign.getName()));
			query.append("type=");
			query.append(JDBC.quoteMore(campaign.getType()));
			query.append("product_group=");
			query.append(JDBC.quoteMore(campaign.getProductGroup()));
			query.append("product_line=");
			query.append(JDBC.quoteMore(campaign.getProductLine()));
			query.append("lead_count=");
			query.append(campaign.getLeadCount().intValue());
			query.append("WHERE pk = ");
			query.append(campaign.getPK());

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int updatedRows = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close(); 
			} catch (SQLException x) { }
		}
	}

	/**
	 * insert a new campaign row using the passed
	 * Campaign object as the column values.
	 * 
	 * @param obj the object we are going to insert
	 * @param load true if this row is to be loaded, false if
	 * the row is to be added for the first time
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public long insertRow(Object obj, boolean load) 
		throws SQLException {

		Campaign campaign = (Campaign)obj;

		if (!load)
			campaign.setPK(DBUtils.generatePK());

		Statement stmt = null;
		StringBuffer query = new StringBuffer(insertQuery);

		try {
			stmt = con.createStatement();

			query.append(campaign.getPK()).append(",");
			query.append(JDBC.quoteMore(campaign.getName()));
			query.append(JDBC.quoteMore(campaign.getType()));
			query.append(JDBC.quoteMore(campaign.getProductGroup()));
			query.append(JDBC.quoteMore(campaign.getProductLine()));
			query.append(campaign.getLeadCount().intValue()).append(",");
			query.append(JDBC.quoteMore(campaign.getModifiedBy()));
			query.append(JDBC.quote(DateUtils.format(DB_TYPE,campaign.getModifiedDate())));
			query.append(")");

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int rc = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}

		return campaign.getPK();
	}

	/**
	 * delete a single campaign row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long we use to delete with
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();
		String query = deleteQuery + pkValue;
	
		Statement stmt = null;
		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }		
		}
	}

	/**
	 * select all campaign rows 
	 * 
	 * @param lastSyncDate a user's last sync date used to limit the query
	 * or null if no limit
	 * @return the Campaign(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectAllRows(java.util.Date lastSyncDate) 
		throws SQLException {

		ArrayList campaigns = new ArrayList();
		Statement stmt = null;
		ResultSet rs = null;
		StringBuffer query = new StringBuffer();
		query.append(selectAllQuery);
		if (lastSyncDate != null) {
            query.append(" WHERE campaign.modified_date > ");
            query.append(DateUtils.formatDateTime(DB_TYPE, lastSyncDate));
		}

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query.toString());
			rs = stmt.executeQuery(query.toString());

			int i;
			Campaign campaign;

			while (rs.next()) {
				i=1;
				campaign = new Campaign();
				campaign.setPK(rs.getLong(i)); i++;
				campaign.setName(rs.getString(i)); i++;
				campaign.setType(rs.getString(i)); i++;
				campaign.setProductGroup(rs.getString(i)); i++;
				campaign.setProductLine(rs.getString(i)); i++;
				campaign.setLeadCount(new Integer(rs.getInt(i))); i++;
				campaign.setModifiedBy(rs.getString(i)); i++;
				campaign.setModifiedDate(rs.getDate(i)); 
				campaigns.add(campaign);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return campaigns;
	}

	/**
	 * delete all campaign rows 
	 *
	 * @exception java.sql.SQLException 
	 */
	public final void deleteAllRows() 
		throws SQLException {

		String query = deleteAllQuery;
	
		Statement stmt = null;
		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }		
		}
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table campaign";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

}
