package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;
import java.util.ArrayList;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * This class implements the DBInterface for
 * the Contact class objects which are stored
 * in the contact database table.
 */
public class ContactDB extends DBType implements DBInterface {
	private Connection con;

	private final static String deleteQuery =
		"DELETE FROM contact WHERE pk = ";

	private final static String deleteOppRowsQuery =
		"DELETE FROM contact WHERE opp_pk = ";

	private final static String insertQuery =
		"INSERT INTO contact " +
		"VALUES (";

	private final static String selectQuery =
			"SELECT "+
    		"opp_pk," +
    		"address_pk," +
    		"modified_by," +
    		"modified_date " +
			"FROM contact "+
			"WHERE pk = ";

	private final static String selectForOppQuery =
			"SELECT "+
    		"pk," +
    		"opp_pk," +
    		"address_pk," +
  			"modified_by, modified_date " +
			"FROM contact "+
			"WHERE opp_pk = ";

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 */
	public ContactDB() {
	}

	/**
	 * construct using a particular db type
	 * @param dbType the db type to assume
	 */
	public ContactDB(int dbType) {
		DB_TYPE=dbType;
	}

	/**
	 * a version of the constructor when a connection
	 * is already obtained from somewhere else
	 *
	 * @param con the Connection to use 
	 */
	public ContactDB(Connection con) {
		this.con = con;
	}

	/**
	 * return the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public final void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single contact row using the passed
	 * primary key
	 * 
	 * @param pk the primary key we search with
	 * @return the Contact row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		Contact contact = new Contact();
		contact.setPK(((Long)pk).longValue());

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + contact.getPK();

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;

			while (rs.next()) {
				i=1;
				contact.setOppKey(rs.getLong(i)); i++;
				contact.setAddressKey(rs.getLong(i)); i++;
				contact.setModifiedBy(rs.getString(i)); i++;
				contact.setModifiedDate(rs.getDate(i)); 
			}

			AddressDB addressDB = new AddressDB(DB_TYPE);
			addressDB.setConnection(getConnection());
			Address a = (Address)addressDB.selectRow(new Long(contact.getAddressKey()));
			contact.setAddress(a);

		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return contact;
	}


	/**
	 * not yet coded
	 * @param name description
	 * @exception java.sql.SQLException 
	 */
	public final void updateRow(Object obj) 
		throws SQLException {

		Contact contact = (Contact)obj;

		//just update the contact's address, the other attributes
		//of contact are not worth updating.

		AddressDB addressDB = new AddressDB(DB_TYPE);
		addressDB.setConnection(getConnection());
		addressDB.updateRow(contact.getAddress());
	}

	/**
	 * insert a new forecast row using the passed
	 * Contact object as the column values.
	 * 
	 * @param obj the object we are going to insert
	 * @param load true if the row is to be loaded, false if the
	 * row is to be added for the first time
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public final long insertRow(Object obj, boolean load) 
		throws SQLException {

		Contact contact = (Contact)obj;

		if (!load)
			contact.setPK(DBUtils.generatePK());

		//insert the contact's address first
		AddressDB addressDB = new AddressDB(DB_TYPE);
		addressDB.setConnection(getConnection());
		contact.getAddress().setModifiedBy(contact.getModifiedBy());
		contact.getAddress().setPK(addressDB.insertRow(contact.getAddress(), load));

		StringBuffer query = new StringBuffer(insertQuery);
		Statement stmt = con.createStatement();

		query.append(contact.getPK()).append(",");
		query.append(contact.getOppKey()).append(",");
		query.append(contact.getAddress().getPK()).append(",");
		query.append(JDBC.quoteMore(contact.getModifiedBy()));
		query.append(JDBC.quote(DateUtils.format(DB_TYPE, contact.getModifiedDate())));
		query.append(")");

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int rc = stmt.executeUpdate(query.toString());

		return contact.getPK();
	}

	/**
	 * delete a single forecast row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();


		String query = deleteQuery + pkValue;

		Statement stmt = null;

		try {
			Contact c = (Contact)selectRow(new Long(pkValue));
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);

			//delete the child address 
			AddressDB addressDB = new AddressDB(DB_TYPE);
			addressDB.setConnection(getConnection());
			addressDB.deleteRow(new Long(c.getAddress().getPK()));
		} catch (SQLException e) {
			LogWrite.write(e);
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}


	/**
	 * delete all contact rows using the passed
	 * opportunity primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteOpportunityRows(Object obj) 
		throws SQLException {

		long oppPK = ((Long)obj).longValue();

		String query = deleteOppRowsQuery + oppPK;

		Statement stmt = null;

		try {
			//first, go get all the contacts for this opp
			ArrayList contacts = new ArrayList();
			selectByOppRow(oppPK, contacts);

			//second, delete all the contact's Address children
			AddressDB addressDB = new AddressDB(DB_TYPE);
			addressDB.setConnection(getConnection());
			Contact  contact;
			for (int i=0;i<contacts.size();i++) {
				contact = (Contact)contacts.get(i);
				addressDB.deleteRow(new Long(contact.getAddress().getPK()));
			}

			//lastly, delete all the contacts for this Opp
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			LogWrite.write(e);
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * select all forecast rows using the passed
	 * opportunity primary key
	 * 
	 * @param opp_pk the primary key value to search with
	 * @param contacts a list that gets populated with objects found
	 * in the search
	 * @exception java.sql.SQLException 
	 */
	public final void selectByOppRow(long opp_pk, ArrayList contacts) 
		throws SQLException {

		Contact contact = null;

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectForOppQuery + opp_pk;

		try {
			stmt = con.createStatement();

			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			long addressPK;
			AddressDB addressDB = new AddressDB(DB_TYPE);
			addressDB.setConnection(getConnection());

			while (rs.next()) {
				i=1;
				contact = new Contact();
				contact.setPK(rs.getLong(i)); i++;
				contact.setOppKey(rs.getLong(i)); i++;
				addressPK = rs.getLong(i); i++;
				contact.setAddress((Address)addressDB.selectRow(new Long(addressPK)));

				contact.setModifiedBy(rs.getString(i)); i++;
				contact.setModifiedDate(rs.getDate(i)); 
				contacts.add(contact);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table contact";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }


}
