package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.util.ArrayList;
import java.sql.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * This class implements the DBInterface for
 * the CustomerInventory class objects which are stored
 * in the customer_inventory database table.
 */
public class CustomerInventoryDB extends DBType implements DBInterface {
	private Connection con;

	private final static String deleteQuery =
		"DELETE FROM customer_inventory WHERE pk = ";

	private final static String insertQuery =
		"INSERT INTO customer_inventory " +
		" (PK,CUSTOMER_PK,PRODUCT_PK,CNT,MODIFIED_BY,MODIFIED_DATE) " +
		"VALUES (";

	private final static String updateQuery = 
			"UPDATE customer_inventory " +
			"SET ";

	private final static String selectQuery =
			"SELECT customer_pk, product_pk, cnt, " +
  			"modified_by, modified_date " +
			"FROM customer_inventory "+
			"WHERE pk = ";
	private final static String selectAllQuery =
			"SELECT customer_pk, product_pk, cnt, " +
  			"modified_by, modified_date " +
			"FROM customer_inventory "+
			"WHERE customer_pk = ";

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 *
	 */
	public CustomerInventoryDB() {
	}

	/**
	 * construct using a db type 
	 * @param dbType the db type to assume
	 */
	public CustomerInventoryDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a version of the constructor when a connection
	 * is already obtained from somewhere else
	 *
	 * @param con the Connection to use 
	 */
	public CustomerInventoryDB(Connection con) {
		this.con = con;
	}

	/**
	 * return the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public final void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single customer_inventory row using the passed
	 * primary key
	 * 
	 * @param pk the primary key to search with
	 * @return the CustomerInventory row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		CustomerInventory sp = new CustomerInventory();
		sp.setPK(((Long)pk).longValue());

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + sp.getPK();

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			long prodPK=0;

			while (rs.next()) {
				i=1;
				sp.setCustomerPK(rs.getLong(i)); i++;

				prodPK = rs.getLong(i); i++;
				ProductDB pdb = new ProductDB(DB_TYPE);
				pdb.setConnection(getConnection());
				sp.setProduct((Product)pdb.selectRow(new Long(prodPK)));

				sp.setCount(new Integer(rs.getInt(i))); i++;
				sp.setModifiedBy(rs.getString(i)); i++;
				sp.setModifiedDate(rs.getDate(i)); 
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return sp;
	}


	/**
	 * update a single customer_inventory row using the passed
	 * CustomerInventory object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param obj the object we are updating
	 * @exception java.sql.SQLException 
	 */
	public final void updateRow(Object obj) 
		throws SQLException {

		CustomerInventory customer_inventory = (CustomerInventory)obj;

		StringBuffer query = new StringBuffer(updateQuery);
		Statement stmt = con.createStatement();

		query.append("cnt=");
		query.append(customer_inventory.getCount().intValue()).append(",");
		query.append(" WHERE pk=");
		query.append(customer_inventory.getPK());

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int updatedRows = stmt.executeUpdate(query.toString());
	}

	/**
	 * insert a new customer_inventory row using the passed
	 * CustomerInventory object as the column values.
	 * 
	 * @param obj the object we are inserting
	 * @param load true if to be loaded, false if to be added
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public final long insertRow(Object obj, boolean load) 
		throws SQLException {

		CustomerInventory customer_inventory = (CustomerInventory)obj;

		//we assume the customer has already been added
		//we assume the product has already been added
		//insert the customer_inventory 
		if (!load)
			customer_inventory.setPK(DBUtils.generatePK());

		Date modified_date = new Date(customer_inventory.getModifiedDate().getTime());

		StringBuffer query = new StringBuffer(insertQuery);
		Statement stmt = con.createStatement();

		query.append(customer_inventory.getPK()).append(",");
		query.append(customer_inventory.getCustomerPK()).append(",");
		query.append(customer_inventory.getProduct().getPK()).append(",");
		query.append(customer_inventory.getCount().intValue()).append(",");
		query.append(JDBC.quoteMore(customer_inventory.getModifiedBy()));
		query.append(JDBC.quote(DateUtils.format(DB_TYPE, customer_inventory.getModifiedDate())));
		query.append(")");

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int rc = stmt.executeUpdate(query.toString());

		return customer_inventory.getPK();
	}

	/**
	 * delete a single customer_inventory row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();

		String query = deleteQuery + pkValue;

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * select all customer_inventory rows using the passed
	 * customer primary key
	 * 
	 * @param custpk the customer primary key to search with
	 * @return an ArrayList of CustomerInventory row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectAllRows(long custpk) 
		throws SQLException {

		CustomerInventory ci=null;
		ArrayList rows = new ArrayList();

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectAllQuery + custpk;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			long prodPK=0;

			while (rs.next()) {
				i=1;
				ci = new CustomerInventory();
				ci.setCustomerPK(rs.getLong(i)); i++;

				prodPK = rs.getLong(i); i++;
				ProductDB pdb = new ProductDB(DB_TYPE);
				pdb.setConnection(getConnection());
				ci.setProduct((Product)pdb.selectRow(new Long(prodPK)));

				ci.setCount(new Integer(rs.getInt(i))); i++;
				ci.setModifiedBy(rs.getString(i)); i++;
				ci.setModifiedDate(rs.getDate(i)); 
				rows.add(ci);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return rows;
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table customer_inventory";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

}
