package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;
import java.util.ArrayList;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the DBInterface for
 * the Product class objects which are stored
 * in the product database table.
 */
public class ProductDB extends DBType implements DBInterface {
	private Connection con;

	private final static String selectByGroupLineQuery = 
			"SELECT " +
				"pk, name, " +
    			"model_no, cost, price ," +
    			"description, modified_by, modified_date " +
			"FROM product " +
			"WHERE product_group= ";

	private final static String selectDistinctGroups =
			"SELECT DISTINCT product_group FROM product ORDER BY product_group";

	private final static String selectDistinctLines =
			"SELECT DISTINCT line FROM product WHERE " +
			"product_group = ";
	

	private final static String selectMatrixQuery = 
			"SELECT DISTINCT " +
				"product_group, line " +
			"FROM product ORDER BY product_group, line";

	private final static String selectByGroupLineNameQuery = 
			"SELECT " +
    			"pk, model_no, cost, price ," +
    			"description, modified_by, modified_date " +
			"FROM product " +
			"WHERE product_group=";


	private final static String selectQuery = 
			"SELECT " +
				"pk, product_group, line, name, " +
    			"model_no, cost, price ," +
    			"description, modified_by, modified_date " +
			"FROM product " +
			"WHERE pk=";

	private final static String selectAllQuery = 
			"SELECT " +
				"pk, product_group, line, name, " +
    			"model_no, cost, price ," +
    			"description, modified_by, modified_date " +
			"FROM product ";

	private final static String updateQuery = 
			"UPDATE product " +
			"SET  " ;

	private final static String insertQuery = 
			"INSERT INTO product VALUES ( ";

	private final static String deleteQuery = 
			"DELETE FROM product WHERE pk =";


	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 */
	public ProductDB() {
	}

	/**
	 * construct, specifying a db type
	 * @param dbType the db type
	 */
	public ProductDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a constructor that accepts an existing Connection
	 * to use for future operations
	 *
	 * @param con the Connection to use
	 */
	public ProductDB(Connection con) {
		this.con = con;
	}

	/**
	 * get the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single product row using the passed
	 * primary key
	 * 
	 * @param name description
	 * @return the row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		Product product = new Product();
		long pkValue = ((Long)pk).longValue();
		product.setPK(pkValue);
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + pkValue;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			while (rs.next()) {
				i=1;
				product.setPK(rs.getLong(i)); i++;
				product.setGroup(rs.getString(i)); i++;
				product.setLine(rs.getString(i)); i++;
				product.setName(rs.getString(i)); i++;
				product.setModelNo(rs.getString(i)); i++;
				product.setCost(new Double(rs.getDouble(i))); i++;
				product.setPrice(new Double(rs.getDouble(i))); i++;
				product.setDesc(rs.getString(i)); i++;
				product.setModifiedBy(rs.getString(i)); i++;
				product.setModifiedDate(rs.getDate(i)); 
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return product;
	}


	/**
	 * update a single product row using the passed
	 * Product object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param name description
	 * @exception java.sql.SQLException 
	 */
	public void updateRow(Object obj) 
		throws SQLException {

		Product product = (Product)obj;
		long pk = product.getPK();

		StringBuffer query = new StringBuffer(updateQuery);
		Statement stmt = null;

		try {
			stmt = con.createStatement();

			query.append("product_group=");
			query.append(JDBC.quoteMore(product.getGroup()));
			query.append("line=");
			query.append(JDBC.quoteMore(product.getLine()));
			query.append("name=");
			query.append(JDBC.quoteMore(product.getName()));
			query.append("model_no=");
			query.append(JDBC.quoteMore(product.getModelNo()));
			query.append("cost=");
			query.append(product.getCost().doubleValue()).append(",");
			query.append("price=");
			query.append(product.getPrice().doubleValue()).append(",");
			query.append("description=");
			query.append(JDBC.quote(product.getDesc()));
			query.append(" WHERE pk=");
			query.append(product.getPK());	

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int updatedRows = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close(); 
			} catch (SQLException x) { }
		}
	}

	/**
	 * insert a new product row using the passed
	 * Product object as the column values.
	 * 
	 * @param obj the Product to add or load
	 * @param load true if we are to load , false if we are to add
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public long insertRow(Object obj, boolean load) 
		throws SQLException {

		Product product = (Product)obj;

		if (!load)
			product.setPK(DBUtils.generatePK());

		StringBuffer query = new StringBuffer(insertQuery);
		Statement stmt = null;

		try {
			stmt = con.createStatement();

			query.append(product.getPK()).append(","); 
			query.append(JDBC.quoteMore(product.getGroup()));
			query.append(JDBC.quoteMore(product.getLine()));
			query.append(JDBC.quoteMore(product.getName()));
			query.append(JDBC.quoteMore(product.getModelNo()));
			query.append(product.getCost().doubleValue()).append(","); 
			query.append(product.getPrice().doubleValue()).append(",");
			query.append(JDBC.quoteMore(product.getDesc()));
			query.append(JDBC.quoteMore(product.getModifiedBy()));
			if (DB_TYPE == Prefs.MYSQL)
				query.append("CURRENT_DATE");
			else
				query.append("SYSDATE");
			query.append(")");

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int rc = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}

		return product.getPK();
	}

	/**
	 * delete a single product row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();
		String query = deleteQuery + pkValue;
	
		Statement stmt = null;
		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }		
		}
	}

	/**
	 * select the product "matrix" which is a set of
	 * minimal info about the products
	 * 
	 * @param name description
	 * @return the row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectMatrixRows() 
		throws SQLException {

		ArrayList rows = new ArrayList();
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectMatrixQuery;
		

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			MatrixObject mo = null;
			String group, lastGroup=null;
			String line;
			boolean rowsFound=false;
			ArrayList lines = new ArrayList();
			long pk;

			while (rs.next()) {
				i=1;

				group = rs.getString(i); i++;
				line  = rs.getString(i); i++;

				if (!rowsFound) { //first time thru
					lastGroup = group; //prime for the control break
					rowsFound = true;
				}

				if (!group.equals(lastGroup)) { //control break
					mo = new MatrixObject(lastGroup, lines);
					mo.print();
					rows.add(mo);
					lines = new ArrayList();
					lastGroup = group;
				} 

				lines.add(line);

			}

			if (rowsFound) {
				mo = new MatrixObject(lastGroup, lines);
				mo.print();
				rows.add(mo);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return rows;
	}

	/**
	 * select all product rows using the passed
	 * group and line keys
	 * 
	 * @param name description
	 * @return the Product(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectByGroupLine(String group, String line) 
		throws SQLException {

		ArrayList rows = new ArrayList();
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectByGroupLineQuery + "'" + group + "'" 
			+ " AND line = " + "'" + line + "'";

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			Product product=null;

			while (rs.next()) {
				i=1;
				product = new Product();
				product.setGroup(group);
				product.setLine(line);
				product.setPK(rs.getLong(i)); i++;
				product.setName(rs.getString(i)); i++;
				product.setModelNo(rs.getString(i)); i++;
				product.setCost(new Double(rs.getDouble(i))); i++;
				product.setPrice(new Double(rs.getDouble(i))); i++;
				product.setDesc(rs.getString(i)); i++;
				product.setModifiedBy(rs.getString(i)); i++;
				product.setModifiedDate(rs.getDate(i)); 
				rows.add(product);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}
		
		return rows;
	}

	/**
	 * select a single product row using the passed
	 * group, line, name keys
	 * 
	 * @param name description
	 * @return the row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Product selectRow(String group, String line, String name) 
		throws SQLException {

		Product product = null;
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectByGroupLineNameQuery + "'" + group + "'" +
			" AND line=" + "'" + line + "'" + " AND name=" + "'" + name +
			"'";

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			while (rs.next()) {
				i=1;
				product = new Product();
				product.setGroup(group);
				product.setLine(line);
				product.setName(name);
				product.setPK(rs.getLong(i)); i++;
				product.setModelNo(rs.getString(i)); i++;
				product.setCost(new Double(rs.getDouble(i))); i++;
				product.setPrice(new Double(rs.getDouble(i))); i++;
				product.setDesc(rs.getString(i)); i++;
				product.setModifiedBy(rs.getString(i)); i++;
				product.setModifiedDate(rs.getDate(i)); 
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}
		
		return product;
	}

	/**
	 * select all product rows 
	 * 
	 * @param lastSyncDate a date which limits the rows returned
	 * @return the row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectAllRows(java.util.Date lastSyncDate) 
		throws SQLException {

		ArrayList products = new ArrayList();
		Product product = null;
		Statement stmt = null;
		ResultSet rs = null;
		StringBuffer query = new StringBuffer();
		query.append(selectAllQuery);
		if (lastSyncDate != null) {
			query.append(" WHERE product.modified_date > ");
			query.append(DateUtils.formatDateTime(DB_TYPE, lastSyncDate));
		}

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query.toString());
			rs = stmt.executeQuery(query.toString());

			int i;
			while (rs.next()) {
				i=1;
				product = new Product();
				product.setPK(rs.getLong(i)); i++;
				product.setGroup(rs.getString(i)); i++;
				product.setLine(rs.getString(i)); i++;
				product.setName(rs.getString(i)); i++;
				product.setModelNo(rs.getString(i)); i++;
				product.setCost(new Double(rs.getDouble(i))); i++;
				product.setPrice(new Double(rs.getDouble(i))); i++;
				product.setDesc(rs.getString(i)); i++;
				product.setModifiedBy(rs.getString(i)); i++;
				product.setModifiedDate(rs.getDate(i)); 

				products.add(product);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return products;
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table product";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

}
