package sellwin.db;

import java.sql.*;
import java.util.*;

import sellwin.domain.*;
import sellwin.utils.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the DBInterface for
 * the StateTax class objects which are stored
 * in the address database table.
 */
public class StateTaxDB extends DBType implements DBInterface {

	private Connection con;

	private final static String selectQuery = 
			"SELECT " +
				"STATE_CODE, STATE_FULL_NAME, TAX_PCT, " +
				"modified_by, modified_date " +
			"FROM state_tax WHERE STATE_CODE=";

	private final static String selectAllQuery = 
			"SELECT " +
				"STATE_CODE, STATE_FULL_NAME, TAX_PCT, " +
				"modified_by, modified_date " +
			"FROM state_tax ";

	private final static String updateQuery = 
			"UPDATE state_tax  " +
			"SET ";

	private final static String insertQuery =
		"INSERT INTO state_tax VALUES ( ";

	private final static String deleteAllQuery =
		"DELETE FROM state_tax";


	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 */
	public StateTaxDB() {
	}

	/**
	 * construct using a db type
	 * @param dbType the database type to use
	 */
	public StateTaxDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a constructor that accepts an existing Connection
	 * to use for future operations
	 *
	 * @param con the Connection to use
	 */
	public StateTaxDB(Connection con) {
		this.con = con;
	}

	/**
	 * get the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single address row using the passed
	 * primary key
	 * 
	 * @param pk the primary key we are searching with
	 * @return the StateTax(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		StateTax tax = new StateTax();
		tax.setCode((String)pk);
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + "'" + tax.getCode() + "'";

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			while (rs.next()) {
				i=1;
				tax.setFullName(rs.getString(i)); i++;
				tax.setRate(rs.getDouble(i)); i++;
				tax.setModifiedBy(rs.getString(i)); i++;
				tax.setModifiedDate(rs.getDate(i)); 
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return tax;
	}


	/**
	 * update a single state_tax row using the passed
	 * StateTax object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param obj the object we want to update with
	 * @exception java.sql.SQLException 
	 */
	public void updateRow(Object obj) 
		throws SQLException {

		StateTax tax = (StateTax)obj;

		Statement stmt = null;
		StringBuffer query = new StringBuffer(updateQuery);

		try {
			stmt = con.createStatement();

			query.append("tax_pct=");
			query.append(tax.getRate());
			query.append(",modified_by='");
			query.append(tax.getModifiedBy()).append("'");
			query.append(",modified_date=");
			if (DB_TYPE == Prefs.MYSQL)
				query.append("CURRENT_DATE ");
			else
				query.append("SYSDATE ");
			query.append(" WHERE state_code='");
			query.append(tax.getCode()).append("'");	

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int updatedRows = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close(); 
			} catch (SQLException x) { }
		}
	}

	/**
	 * not implemented since we don't need insert
	 * functionality for this table
	 * 
	 * @param obj the object we are inserting
	 * @param load true if loading, false if adding
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public long insertRow(Object obj, boolean load) 
		throws SQLException {

		StateTax st = (StateTax)obj;
		
		Statement stmt = null;
		StringBuffer query = new StringBuffer(insertQuery);

		try {
			stmt = con.createStatement();
			query.append(JDBC.quoteMore(st.getCode()));
			query.append(st.getRate()).append(",");
			query.append(JDBC.quoteMore(st.getFullName()));
			query.append(JDBC.quoteMore(st.getModifiedBy()));
			query.append(JDBC.quote(DateUtils.format(DB_TYPE, st.getModifiedDate())));
			query.append(")");
			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int rc = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) {}
		}

		return 0L;
	}

	/**
	 * not needed since we don't ever delete from this table
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

	}

	/**
	 * delete all rows 
	 * @exception java.sql.SQLException 
	 */
	public final void deleteAllRows() 
		throws SQLException {

        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(deleteAllQuery);
            stmt.executeUpdate(deleteAllQuery);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
	}

	/**
	 * select all state_tax rows 
	 * 
	 * @param lastSyncDate a user's last sync date which can
	 * limit the query results, if null, no limit
	 * @return the ArrayList of StateTax(s)
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectAllRows(java.util.Date lastSyncDate) 
		throws SQLException {

		ArrayList list = new ArrayList();
		StateTax tax;
		Statement stmt = null;
		ResultSet rs = null;
		StringBuffer query = new StringBuffer();
		query.append(selectAllQuery);
		if (lastSyncDate != null) {
            query.append(" WHERE modified_date > ");
            query.append(DateUtils.formatDateTime(DB_TYPE, lastSyncDate));
		}
		query.append(" ORDER BY STATE_CODE");

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query.toString());
			rs = stmt.executeQuery(query.toString());

			int i;
			while (rs.next()) {
				i=1;
				tax = new StateTax();
				tax.setCode(rs.getString(i)); i++;
				tax.setFullName(rs.getString(i)); i++;
				tax.setRate(rs.getDouble(i)); i++;
				tax.setModifiedBy(rs.getString(i)); i++;
				tax.setModifiedDate(rs.getDate(i)); 
				list.add(tax);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return list;
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table state_tax";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

}
