package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;
import java.util.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the DBInterface for
 * the UserGroup class objects which are stored
 * in the user_group database table.
 */
public class UserGroupDB extends DBType implements DBInterface {
	private Connection con;

	private final static String deleteAllQuery =
		"DELETE FROM user_group";

	private final static String deleteByNameQuery =
		"DELETE FROM user_group WHERE name = ";

	private final static String deleteQuery =
		"DELETE FROM user_group WHERE pk = ";

	private final static String insertQuery =
		"INSERT INTO user_group " +
		"VALUES (";
	private final static String updateQuery = 
			"UPDATE user_group " +
			"SET ";
	private final static String selectQuery =
			"SELECT name, "+
  			"modified_by, modified_date " +
			"FROM user_group "+
			"WHERE pk = ";

	private final static String selectAllQuery =
			"SELECT pk, name, "+
  			"modified_by, modified_date " +
			"FROM user_group ";

	private final static String selectUsersInGroupQuery =
			"SELECT user_group_member.user_pk " +
			"FROM user_group, user_group_member " +
			"WHERE " +
			"user_group.pk = user_group_member.user_group_pk AND "+
			"user_group.name = ";

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 */
	public UserGroupDB() {
	}

	/**
	 * construct using a db type
	 * @param dbType the db type to assume
	 */
	public UserGroupDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a version of the constructor when a connection
	 * is already obtained from somewhere else
	 *
	 * @param con the Connection to use 
	 */
	public UserGroupDB(Connection con) {
		this.con = con;
	}

	/**
	 * return the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public final void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single user_group row using the passed
	 * primary key
	 * 
	 * @param pk the primary key we are searching with
	 * @return the UserGroup row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		UserGroup ug = new UserGroup("");
		ug.setPK(((Long)pk).longValue());

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + ug.getPK();

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;

			while (rs.next()) {
				i=1;
				ug.setName(rs.getString(i)); i++;
				ug.setModifiedBy(rs.getString(i)); i++;
				ug.setModifiedDate(rs.getDate(i)); 
			}

		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return ug;
	}


	/**
	 * update a single user_group row using the passed
	 * UserGroup object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param obj the object we are updating
	 * @exception java.sql.SQLException 
	 */
	public final void updateRow(Object obj) 
		throws SQLException {

		UserGroup user_group = (UserGroup)obj;

		StringBuffer query = new StringBuffer(updateQuery);
		Statement stmt = con.createStatement();

		query.append("name=");
		query.append(JDBC.quoteMore(user_group.getName()));
		query.append("modified_by=");
		query.append(JDBC.quoteMore(user_group.getModifiedBy()));
		query.append("modified_date=");
		if (DB_TYPE == Prefs.MYSQL)
			query.append("CURRENT_DATE");
		else
			query.append("SYSDATE");
		query.append(" WHERE pk=").append(user_group.getPK());

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int updatedRows = stmt.executeUpdate(query.toString());
	}

	/**
	 * insert a new user_group row using the passed
	 * UserGroup object as the column values.
	 * 
	 * @param obj the object we are inserting
	 * @param load true if loading, false if adding
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public final long insertRow(Object obj, boolean load) 
		throws SQLException {

		UserGroup user_group = (UserGroup)obj;

		if (!load)
			user_group.setPK(DBUtils.generatePK());

		StringBuffer query = new StringBuffer(insertQuery);
		Statement stmt = con.createStatement();

		query.append(user_group.getPK()).append(",");
		query.append(JDBC.quoteMore(user_group.getName()));
		query.append(JDBC.quoteMore(user_group.getModifiedBy()));
		if (DB_TYPE == Prefs.MYSQL)
			query.append("CURRENT_DATE");
		else
			query.append("SYSDATE");
		query.append(")");

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int rc = stmt.executeUpdate(query.toString());

		return user_group.getPK();
	}

	/**
	 * delete a single user_group row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();

		String query = deleteQuery + pkValue;

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * delete a single user_group row using the passed
	 * group name
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteByNameRow(String name) 
		throws SQLException {

		String query = deleteByNameQuery + "'" + name + "'";

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * select all user names for a given user group
	 * 
	 * @param groupName the group name we are searching with
	 * @return the SalesPerson row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final TreeMap selectUsersInGroup(String groupName) 
		throws SQLException {

		TreeMap users = new TreeMap();
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectUsersInGroupQuery + "'" + groupName + "'";

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			long user_pk;
			SalesPerson sp=null;
			SalesPersonDB salesPersonDB = new SalesPersonDB(DB_TYPE);
			salesPersonDB.setConnection(getConnection());

			while (rs.next()) {
				i=1;
				user_pk = rs.getLong(i); i++;
				sp = (SalesPerson)(salesPersonDB.selectRow(new Long(user_pk)));
				users.put(sp.getID(), sp);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return users;
	}

	/**
	 * select all user pks for a given user group
	 * @param groupName the group name we are searching with
	 * @return the  primary key(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectUserPKsInGroup(String groupName) 
		throws SQLException {

		ArrayList list = new ArrayList();
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectUsersInGroupQuery + "'" + groupName + "'";

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			long user_pk;

			while (rs.next()) {
				i=1;
				user_pk = rs.getLong(i); i++;
				list.add(new Long(user_pk));
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return list;
	}

	/**
	 * select all user_group rows using the passed
	 * 
	 * @param lastSyncDate a user's last sync date, if not null, is
	 * used to limit the query
	 * @return the UserGroup row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final TreeMap selectAllRows(java.util.Date lastSyncDate) 
		throws SQLException {

		UserGroup ug = null;
		TreeMap list = new TreeMap();
		Statement stmt = null;
		ResultSet rs = null;
		StringBuffer query = new StringBuffer();
		query.append(selectAllQuery);
		if (lastSyncDate != null) {
            query.append(" WHERE user_group.modified_date > ");
            query.append(DateUtils.formatDateTime(DB_TYPE, lastSyncDate));
		}
		query.append(" ORDER BY name");

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query.toString());
			rs = stmt.executeQuery(query.toString());

			int i,j;
			ArrayList users;
			SalesPerson sp;

			while (rs.next()) {
				i=1;
				ug = new UserGroup();
				ug.setPK(rs.getLong(i)); i++;
				ug.setName(rs.getString(i)); i++;
				ug.setModifiedBy(rs.getString(i)); i++;
				ug.setModifiedDate(rs.getDate(i)); 
	
				users =  selectUserPKsInGroup(ug.getName()); 
				for (j=0;j<users.size();j++) {
					ug.addUser((Long)users.get(j));
				}

				list.put(ug.getName(), ug);
			}

		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return list;
	}

	/**
	 * delete all user_group rows
	 *
	 * @exception java.sql.SQLException 
	 */
	public final void deleteAllRows() 
		throws SQLException {

		String query = deleteAllQuery;

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table user_group";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

}
