package sellwin.gui;

import sellwin.domain.*;
import sellwin.utils.*;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the Activity Panel tab
 * accessed by the users from the MainWindow ...
 * maintenance of an Opportunities Activities is
 * done here
 */
public class ActivPanel extends JPanel implements GUIChars {

	private JPopupMenu popup = new JPopupMenu();
	private ActivityTableModel tableModel = null;

    private final javax.swing.JPanel northPanel = new JPanel();
    private javax.swing.JPanel centerPanel;
	private JLabel activTableLabel;
	private JTable activTable;
	private JScrollPane activTableScrollPane;

	private Whiteboard wb;
	private JPanel thisPanel;

	private JMenuItem addMenuItem = new JMenuItem("Add");
	private JMenuItem deleteMenuItem = new JMenuItem("Delete");
	private JMenuItem recipientsMenuItem = new JMenuItem("Add Recipient");
	private JButton applyButton = new JButton("Apply");
	private JButton checkButton = new JButton("Check Alarms");

	/**
	 * construct the activity panel and initialize it 
	 */
	public ActivPanel() {
		super(new BorderLayout());

		thisPanel = this;

		wb = MainWindow.getWhiteboard();

        initComponents();

		JPanel buttonPanel = new JPanel();
		buttonPanel.add(applyButton);
		buttonPanel.add(checkButton);

		add(buttonPanel, BorderLayout.SOUTH);

		recipientsMenuItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					LogWrite.write("add recipient pressed");
				}
			}
		);

		addMenuItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					try {
						if (wb.getCurrentOpportunity() == null) {
							JOptionPane.showMessageDialog(
							thisPanel, 
             				wb.getLang().getString("selectOpp"),
                            wb.getLang().getString("error"),
							JOptionPane.INFORMATION_MESSAGE);
							return;
						}

						String name = JOptionPane.showInputDialog(wb.getLang().getString("enterAct"));
						
						Activity newActivity = new Activity(name);
						newActivity.setSubject(name);
						newActivity.setModifiedBy(wb.getCurrentUser().getAddress().getFormattedName());

						newActivity.setPK(
							wb.addActivity(
								wb.getCurrentOpportunity().getPK(),
								newActivity)
						);
						activTable.tableChanged(new TableModelEvent(tableModel));
						wb.getCurrentOpportunity().addActivity(newActivity);
						int actCt = wb.getCurrentOpportunity().getActivities().size();
						if (actCt > 0)
							activTable.setRowSelectionInterval(actCt-1, actCt-1);
						JOptionPane.showMessageDialog(
							thisPanel, "Add Successful",
							"Add Successful",
							JOptionPane.INFORMATION_MESSAGE);
					} catch (Exception f) {
						ErrorHandler.show(thisPanel, f);
					}
				}
			}
		);

		applyButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					int index = activTable.getSelectedRow();
					if (index < 0) {
                    	JOptionPane.showConfirmDialog(
                                    thisPanel,
                                    "Please select the row you wish to update.",
                                    "User Error",
                                    JOptionPane.ERROR_MESSAGE);
								return;
					}

					int opt = JOptionPane.showConfirmDialog(thisPanel, 
						"Continue with update?", "Confirm",
						JOptionPane.YES_NO_OPTION);
					if (opt == JOptionPane.YES_OPTION) {
						try {
							//get a handle to the in-memory copy of the act
							ArrayList acts = wb.getCurrentOpportunity().getActivities();
							Activity temp = (Activity)(acts.get(index));
							Activity actCopy = temp.copy();

							//update the activity in the database
							wb.updateActivity(
								wb.getCurrentOpportunity().getPK(),
								actCopy);
							JOptionPane.showMessageDialog(
								thisPanel, "Update Successful",
								"Update Successful",
								JOptionPane.INFORMATION_MESSAGE);

							//since the database update succeeded, update
							//the in-memory activity
						} catch (Exception f) {
							ErrorHandler.show(thisPanel, f);
						}
					}
				}
			}
		);
	
		deleteMenuItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					int opt = JOptionPane.showConfirmDialog(thisPanel, 
						"Continue with delete?", "Confirm",
						JOptionPane.YES_NO_OPTION);
					if (opt == JOptionPane.YES_OPTION) {
						try {
							Opportunity opp = wb.getCurrentOpportunity();
							int index = activTable.getSelectedRow();
							ArrayList acts = opp.getActivities();
							Activity a = (Activity)(acts.get(index));
							wb.deleteActivity(
								opp.getPK(),
								a.getPK());
							opp.deleteActivity(a);
							activTable.tableChanged(new TableModelEvent(tableModel));
							
							JOptionPane.showMessageDialog(
								thisPanel, "Delete Successful",
								"Delete Successful",
								JOptionPane.INFORMATION_MESSAGE);
						} catch (Exception f) {
							ErrorHandler.show(thisPanel, f);
						}
					}
				}
			}
		);



		checkButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
	                Activity alarm;
	                AlarmDialog ad;
					try {
						ArrayList alarms = wb.getAlarms();
	                	LogWrite.write("ActivityChecker sees "+ alarms.size() + " alarms");
	                	for (int i=0;i<alarms.size();i++) {
							alarm = (Activity)alarms.get(i);
							ad = new AlarmDialog(MainWindow.getMainParent(), false, alarm);
							ad.show();
						}
					} catch (AngError x) {
						ErrorHandler.show(MainWindow.getMainParent(), x);
					}
				}
			}
		);


		setColors();

		createPopup();

		activTable.addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(activTable, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(activTable, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(activTable, e.getX(), e.getY());
            }
        });
        activTable.getTableHeader().addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(activTable, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(activTable, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(activTable, e.getX(), e.getY());
            }
        });
		activTableScrollPane.addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(activTable, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(activTable, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(activTable, e.getX(), e.getY());
            }
        });


		initializeDateColumn();
		initializeDescColumn();

		setFonts();
		tableModel = new ActivityTableModel(new ArrayList());
		activTable.setModel(tableModel);
		setLang();
	}

	/**
	 * set the date columns to a certain width
	 */
	private final void initializeColumnWidths() {
		TableColumnModel colModel = activTable.getColumnModel();
		TableColumn col = colModel.getColumn(1); //the Start Date column
		col.setPreferredWidth(130);
	}

	/**
	 * iniitialize the activity 'type' column ...this
	 * ties a cell renderer to a specific table column
	 */
	private final void initializeTypeColumn() {
        JComboBox myTypes = new JComboBox(Activity.TYPES);
		TableColumnModel colModel = activTable.getColumnModel();
		TableColumn col = colModel.getColumn(2); //the type column
		col.setCellEditor(new DefaultCellEditor(myTypes));

		DefaultTableCellRenderer renderer = new DefaultTableCellRenderer();
		col.setCellRenderer(renderer);

		TableCellRenderer headerRenderer = col.getHeaderRenderer();
		if (headerRenderer instanceof DefaultTableCellRenderer) {
			((DefaultTableCellRenderer)headerRenderer).setToolTipText(
				"Click the type to see a list of choices");
		}
	}

	/**
	 * initialize the date column in the activity table 
	 */
	private final void initializeDateColumn() {

		final JButton button = new JButton();

		final DateEditor editor = new DateEditor(button, activTable);
		activTable.setDefaultEditor(Date.class, editor);

		//pass in the editor so that we can pass back the entered date
		final CalendarDialog editDialog = new CalendarDialog(MainWindow.getMainParent(), editor, true);

		//code that brings up the date editor dialog
		button.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				editDialog.setDate(editor.getDate());
				editDialog.show();
				button.setText(Prefs.dateTimeFormat.format(editor.getDate()));
			}
		});

		activTable.setDefaultRenderer(Date.class, new DateTimeRenderer());
	}

	/**
	 * initialize the desc column in the activity table 
	 */
	private final void initializeDescColumn() {

		final JButton button = new JButton();

		final DescEditor editor = new DescEditor(button, activTable);
		activTable.setDefaultEditor(String.class, editor);

		//pass in the editor so that we can pass back the entered string
		final DescEditorDialog editDialog = new DescEditorDialog(
			MainWindow.getMainParent(), editor);

		//code that brings up the date editor dialog
		button.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				editDialog.setDesc(editor.getDesc());
				editDialog.show();
				button.setText(editor.getDesc());
			}
		});

		activTable.setDefaultRenderer(String.class, new DescRenderer());
	}


	/**
	 * refresh the screen with the current opportunity 
	 */
	public final void refreshOpportunity() {
		setSecurity();
		Opportunity opp = wb.getCurrentOpportunity();

		tableModel = new ActivityTableModel(opp.getActivities());
		activTable.setModel(tableModel);
		if (opp.getActivities().size() > 0) {
			activTable.setRowSelectionInterval(0,0);
		}

		initializeTypeColumn();
		initializeColumnWidths();
	}

	/**
	 * clear the screen 
	 */
	public final void clearOpportunity() {
		tableModel = new ActivityTableModel(new ArrayList());
		activTable.setModel(tableModel);
	}

	/**
	 * set all the screen's widgets to a color 
	 */
	public final void setColors() {
        activTableLabel.setForeground(MainWindow.LETTERS);
	}

	/**
	 * set the screen's fields to a specific font
	 */
	public final void setFonts() {
    	centerPanel.setFont(MainWindow.LABEL_FONT);
		addMenuItem.setFont(MainWindow.LABEL_FONT);
		recipientsMenuItem.setFont(MainWindow.LABEL_FONT);
		deleteMenuItem.setFont(MainWindow.LABEL_FONT);
		applyButton.setFont(MainWindow.LABEL_FONT);
		checkButton.setFont(MainWindow.LABEL_FONT);
        activTableLabel.setFont(MainWindow.LABEL_FONT);
	}

	/**
	 * create the popup menu 
	 */
	private final void createPopup() {
		popup.add(addMenuItem);
		popup.add(deleteMenuItem);
		popup.add(recipientsMenuItem);
	}

	/**
	 * initialize or build the screen's components 
	 */
    private final void initComponents() {
        centerPanel = new JPanel();
		activTableLabel = new JLabel("Activities");
		activTable = new JTable();
		activTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		activTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		activTableScrollPane = new JScrollPane(activTable);
		activTableScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
		activTableScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);

        activTableScrollPane.setPreferredSize(new Dimension(600, 270));
        activTableScrollPane.setMinimumSize(new Dimension(600, 270));
        
        setBorder(new javax.swing.border.EtchedBorder());
        add(northPanel);
        
        centerPanel.setLayout(new java.awt.GridBagLayout());
        java.awt.GridBagConstraints gridBagConstraints1;
        
        centerPanel.setBorder(new javax.swing.border.EtchedBorder());
        
       
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 1;
        gridBagConstraints1.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints1.ipadx = 9;
        gridBagConstraints1.ipady = 9;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.EAST;
        centerPanel.add(activTableScrollPane, gridBagConstraints1);
        
        add(centerPanel);
    }

	/**
	 * set the screen's widgets to visible or not 
	 * depending upon a user's security profile
	 */
	private final void setSecurity() {
		try {
			if (wb.getLogin().activityWritable()) {
				addMenuItem.setEnabled(true);
				recipientsMenuItem.setEnabled(true);
				deleteMenuItem.setEnabled(true);
				applyButton.setEnabled(true);
			} else {
				addMenuItem.setEnabled(false);
				recipientsMenuItem.setEnabled(false);
				deleteMenuItem.setEnabled(false);
				applyButton.setEnabled(false);
			}
		} catch (AngError e) {
			ErrorHandler.show(thisPanel, e);
		}
	}

	/**
	 * set the screen's language
	 */
	public final void setLang() {
		tableModel.setLang();
		addMenuItem.setText(wb.getLang().getString("add"));
		recipientsMenuItem.setText(wb.getLang().getString("recipients"));
		deleteMenuItem.setText(wb.getLang().getString("delete"));
		applyButton.setText(wb.getLang().getString("apply"));
		checkButton.setText(wb.getLang().getString("checkAlarms"));
		activTableLabel.setText(wb.getLang().getString("activities"));
	}
}
