package sellwin.gui;

import sellwin.domain.*;
import sellwin.server.*;
import sellwin.utils.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.plaf.*;
import javax.swing.plaf.metal.*;
import java.rmi.*;
import java.net.*;
import java.awt.*;
import java.util.*;
import java.awt.event.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This is the GUI's main class, the overall parent
 * container of all the SellWin GUI components.  It
 * is created by the sellwin.gui.Main class upon
 * startup.
 */
public class MainWindow extends JFrame {

	final public static int OPP_TAB_INDEX = 1;
	final public static int ACT_TAB_INDEX = 2;
	final public static int FORECAST_TAB_INDEX = 3;
	final public static int CUST_TAB_INDEX = 4;
	final public static int QUOTE_TAB_INDEX = 5;
	final public static int ORDER_TAB_INDEX = 6;
	final public static int CONTACT_TAB_INDEX = 7;

	public static final int INITIAL_WIDTH=777; 
	public static final int INITIAL_HEIGHT=500;
    public static final Font LABEL_FONT = new Font("Helvetica", 1, 14);
    public static final Font FIELD_FONT = new Font("Helvetica", 0, 12);

	private static MainWindow thisFrame;
	private static Whiteboard wb = null;
	private static JTabbedPane tabbedPane = new JTabbedPane();
	private JPanel mainPanel = new JPanel(new BorderLayout());
	private static HelpDialog helpDialog = null;
	private static AboutDialog aboutDialog = null;
	private static LogonDialog logonDialog = null;
	private static OppListDialog oppList = null;
	private static SalesGroupDialog salesGroupDialog = null;
	private static CustAdminDialog custAdminDialog = null;
	private static UserAdminDialog salesPersonDialog = null;
	private static ChatDialog chatDialog = null;

	private static OppPanel oppPanel = null;
	private static ActivPanel activPanel = null;
	private static ForecastPanel forecastPanel = null;
	private static CustPanel custPanel = null;
	private static QuotesPanel quotePanel = null;
	private static OrderFulfillPanel ordersPanel = null;
	private static ContactsPanel contactsPanel = null;
	private static LeadsDialog leadsDialog = null;
	private static SyncDialog syncDialog = null;
	private static PrefsDialog prefsDialog = null;
	private static FunnelDialog funnelDialog = null;

	private static JMenu helpMenu = new JMenu("Help");
	private static JMenuItem helpItem = new JMenuItem("Help", 'H');
	private static JMenuItem aboutItem = new JMenuItem("About", 'A');

	private static JMenu adminMenu = new JMenu("Admin");
	private static JMenuItem logonItem = new JMenuItem("Logon", 'L');
	private static JMenuItem logoutItem = new JMenuItem("Logout", 'o');
	private static JMenuItem openItem = new JMenuItem("Open", 'O');
	private static JMenuItem prefsItem = new JMenuItem("Preferences", 'P');
	private static JMenuItem addSalesPersonItem = new JMenuItem("Users", 'U');
	private static JMenuItem addSalesGroupItem = new JMenuItem("Groups", 'G');
	private static JMenuItem custAdminItem = new JMenuItem("Customers", 'C');
	private static JMenuItem leadsItem = new JMenuItem("Sales Leads", 'L');
	private static JMenu dataMenu = new JMenu("Data");
	private static JMenuItem syncItem = new JMenuItem("Sync with server", 'S');

	private static JMenu analyMenu = new JMenu("Analytics");
	private static JMenuItem funnelItem = new JMenuItem("Sales Funnel", 'F');

	private static JMenu utilityMenu = new JMenu("Utilities");
	private static JMenuItem chatItem = new JMenuItem("Chat", 'C');

	private static JMenuItem exitItem = new JMenuItem("Exit", 'E');

	public static Color LETTERS    = Color.black;
	private static JMenuBar menuBar = new JMenuBar();
	private static JMenu fileMenu = new JMenu("File");

	/**
	 * the main class constructor  
	 */
	public MainWindow() 
		throws AngError, RemoteException {

		thisFrame = this;

		wb = new Whiteboard(this);

		createMenuBar();
		setTitle("SellWin");
		setSize(INITIAL_WIDTH, INITIAL_HEIGHT);


		try {
			oppPanel = new OppPanel();
			activPanel = new ActivPanel();
			forecastPanel = new ForecastPanel();
			custPanel = new CustPanel();
			quotePanel = new QuotesPanel();
			ordersPanel = new OrderFulfillPanel();
			contactsPanel = new ContactsPanel();
		} catch (RuntimeException e) {
			ErrorHandler.show(this, e);
			System.exit(-1);
		}

		setItemsOff();

		tabbedPane.add("Opportunity", oppPanel);
		tabbedPane.addTab("Activities", activPanel);
		tabbedPane.addTab("Forecasts", forecastPanel);
		tabbedPane.addTab("Customer", custPanel);
		tabbedPane.addTab("Quotes", quotePanel);
		tabbedPane.addTab("Orders", ordersPanel);
		tabbedPane.addTab("Contacts", contactsPanel);


		mainPanel.add(tabbedPane, BorderLayout.CENTER);
		getContentPane().add(mainPanel);

		setVisible(true);

		//probably later, v1.1?, there will need to be a more advanced
		//way to start and stop the activity checker thread, perhaps
		//tied to the logon-logoff process

		startActivityCheck();

		if (logonDialog == null) 	
			logonDialog = new LogonDialog(thisFrame);
		logonDialog.setLocation(calcLocation());
		logonDialog.setInitialUser();
		logonDialog.show();
		logonDialog.requestFocus();

		tabbedPane.addChangeListener(
			new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
				}
			}
		);

		setLang();

	}

	/**
	 * get the whiteboard used by this app 
	 * currently only a single whiteboard exists
	 * for each client
	 * @return the Whiteboard
	 */
	public static Whiteboard getWhiteboard() {
		return wb;
	}

	/**
	 * disable the logon button on the main
	 * menu
	 */
	public static void setLogonOff() {
		logonItem.setEnabled(false);
		logoutItem.setEnabled(true);
	}

	/**
	 * create the main menu bar 
	 */
	public final void createMenuBar() {
		menuBar = new JMenuBar();
		menuBar.setMargin(new Insets(30, 3, 10, 5));

		helpMenu.add(helpItem);
		helpItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (helpDialog == null) 	
						helpDialog = new HelpDialog(thisFrame);
					helpDialog.setVisible(true);
					helpDialog.setLocationRelativeTo(thisFrame);
					helpDialog.show();
				}
			}
		);
		helpMenu.add(aboutItem);
		aboutItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (aboutDialog == null) 	
						aboutDialog = new AboutDialog(thisFrame);
					aboutDialog.setVisible(true);
					aboutDialog.setLocationRelativeTo(thisFrame);
					aboutDialog.show();
				}
			}
		);

		logonItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (logonDialog == null) 	
						logonDialog = new LogonDialog(thisFrame);
					logonDialog.setLocation(calcLocation());
					logonDialog.show();
					logonDialog.requestFocus();
				}
			});
		fileMenu.add(logonItem);

		fileMenu.add(logoutItem);
		logoutItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					setItemsOff();
					logonItem.setEnabled(true);
				  	JOptionPane.showMessageDialog(
                                thisFrame,
                                "Logout Successful",
                                "Logout Successful",
                                JOptionPane.INFORMATION_MESSAGE);
					logonDialog.setLocation(calcLocation());
					logonDialog.setInitialUser();
					logonDialog.show();
					logonDialog.requestFocus();
				}
			});

		openItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (oppList == null)
						oppList = new OppListDialog();
					oppList.loadUser();
					oppList.setLocation(calcLocation());
					oppList.show();
				}
			});

		fileMenu.add(openItem);
		exitItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					System.exit(0);
				}
			});
		

		fileMenu.add(exitItem);

		custAdminItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (custAdminDialog == null)
						custAdminDialog = new CustAdminDialog(thisFrame, false);
					custAdminDialog.setLocation(calcLocation());
					custAdminDialog.show();
				}
			});
		adminMenu.add(custAdminItem);

		addSalesGroupItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (salesGroupDialog == null)
						salesGroupDialog = new SalesGroupDialog();
					else
						salesGroupDialog.loadData();
					salesGroupDialog.setLocation(calcLocation());
					salesGroupDialog.show();
				}
			});
		adminMenu.add(addSalesGroupItem);

		addSalesPersonItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (salesPersonDialog == null)
						salesPersonDialog = new UserAdminDialog(thisFrame, false);
					salesPersonDialog.setLocation(calcLocation());
					salesPersonDialog.show();
				}
			});
		adminMenu.add(addSalesPersonItem);

		adminMenu.add(prefsItem);
		prefsItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (prefsDialog == null)
						prefsDialog = new PrefsDialog();
					prefsDialog.setLocation(calcLocation());
					prefsDialog.show();
				}
			});

		analyMenu.add(funnelItem);
		funnelItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (funnelDialog == null)
						funnelDialog = new FunnelDialog();
					funnelDialog.setLocation(calcLocation());
					funnelDialog.show();
				}
			});

		utilityMenu.add(chatItem);
		chatItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					try {
						if (chatDialog == null) 
							chatDialog = new ChatDialog(thisFrame, false);

						chatDialog.setLocation(calcLocation());
						chatDialog.show();
					} catch (java.net.ConnectException cerr) {
						ErrorHandler.show(thisFrame, cerr, wb.getLang().getString("connectError"));
					} catch (Exception err) {
						ErrorHandler.show(thisFrame, err, "here");
					}
				}
			});

		adminMenu.add(leadsItem);
		leadsItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (leadsDialog == null)
						leadsDialog = new LeadsDialog();
					leadsDialog.setLocation(calcLocation());
					leadsDialog.loadCampaigns();
					leadsDialog.show();
				}
			});

		menuBar.add(fileMenu);
		menuBar.add(adminMenu);

		dataMenu.add(syncItem);
		syncItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (syncDialog == null)
						syncDialog = new SyncDialog(thisFrame, true);
					syncDialog.show();
				}
			});
		menuBar.add(dataMenu);
		menuBar.add(analyMenu);
		menuBar.add(utilityMenu);
		menuBar.add(helpMenu);

		this.setJMenuBar(menuBar);

		setFont();

	}

	/**
	 * disable the menu bar items 
	 */
	public static void setItemsOff() {
		logoutItem.setEnabled(false);
		addSalesGroupItem.setEnabled(false);
		addSalesPersonItem.setEnabled(false);
		prefsItem.setEnabled(false);
		tabbedPane.setEnabled(false);
		adminMenu.setEnabled(false);
		oppPanel.setEnabled(false);
		activPanel.setEnabled(false);
		forecastPanel.setEnabled(false);
		custPanel.setEnabled(false);
		quotePanel.setEnabled(false);
		ordersPanel.setEnabled(false);
		contactsPanel.setEnabled(false);
	}

	/**
	 * enable the menu bar items
	 */
	public static void setItemsOn() {
		oppPanel.setEnabled(true);
		custPanel.setEnabled(true);
		openItem.setEnabled(true);
		addSalesGroupItem.setEnabled(true);
		addSalesPersonItem.setEnabled(true);
		prefsItem.setEnabled(true);
		tabbedPane.setEnabled(true);
		adminMenu.setEnabled(true);
	}

	/**
	 * get the contacts panel
	 * @return the ContactsPanel
	 */
	public static ContactsPanel getContactsPanel() { return contactsPanel; }

	/**
	 * get the ForecastPanel 
	 * @return the ForecastPanel
	 */
	public static ForecastPanel getForecastPanel() { return forecastPanel; }

	/**
	 * get the FunnelDialog 
	 * @return the FunnelDialog
	 */
	public static FunnelDialog getFunnelDialog() { return funnelDialog; }

	/**
	 * get the ChatDialog
	 * @return the ChatDialgo
	 */
	public static ChatDialog getChatDialog() { return chatDialog; }

	/**
	 * get the ActivityPanel 
	 * @return the ActivPanel
	 */
	public static ActivPanel getActivityPanel() { return activPanel; }

	/**
	 * get the leads dialog 
	 * @return the LeadsDialog
	 */
	public static LeadsDialog getLeadsDialog() { return leadsDialog; }

	/**
	 * get the quotes panel 
	 * @return the QuotesPanel
	 */
	public static QuotesPanel getQuotesPanel() { return quotePanel; }

	/**
	 * get the orders panel 
	 * @return the OrderFulfillPanel
	 */
	public static OrderFulfillPanel getOrdersPanel() { return ordersPanel; }

	/**
	 * get the Opportunity panel 
	 * @return the OppPanel
	 */
	public static OppPanel getOppPanel() { return oppPanel; }

	/**
	 * get the customer panel 
	 * @return the CustPanel
	 */
	public static CustPanel getCustPanel() { return custPanel; }

	/**
	 * get the OppListDialog 
	 * @return the OppListDialog
	 */
	public static OppListDialog getOppList() {
		if (oppList == null)
			oppList = new OppListDialog();
		return oppList;
	} 

	/**
	 * get a handle to the main frame used by
	 * the whole app
	 * @return the JFrame of the MainWindow
	 */
	public static JFrame getMainParent() {
		return thisFrame;
	}

	/**
	 * calculate the location of where the frame should be
	 * placed
	 * @return the Point where the children dialogs should be
	 */
	public static Point calcLocation() {
		Double x = new Double(thisFrame.getWidth()/2 + thisFrame.getLocation().getX());
		Double y = new Double(thisFrame.getHeight()/2 + thisFrame.getLocation().getY()); 
		Point pt = new Point(x.intValue(), y.intValue());
		return pt;
	}

	/**
	 * update all the screens to show the current opportunity 
	 */
	public static void refreshOpportunity() {
		getOppPanel().refreshOpportunity();
		getContactsPanel().refreshOpportunity();
		getQuotesPanel().refreshOpportunity();
		getOrdersPanel().refreshOpportunity();
		getActivityPanel().refreshOpportunity();
		getForecastPanel().refreshOpportunity();
		getCustPanel().refreshOpportunity();
	}

	/**
	 * clear all the screens in the application. this is necessary
	 * when a user deletes the current opportunity being displayed
	 */
	public static void clearOpportunity() {
		getOppPanel().clearOpportunity();
		getContactsPanel().clearOpportunity();
		getQuotesPanel().clearOpportunity();
		getOrdersPanel().clearOpportunity();
		getActivityPanel().clearOpportunity();
		getForecastPanel().clearOpportunity();
		getCustPanel().clearOpportunity();
	}

	/**
	 * change all the fonts 
	 */
	private void setFont() {
		tabbedPane.setFont(LABEL_FONT);
		menuBar.setFont(LABEL_FONT);
		helpMenu.setFont(LABEL_FONT);
		helpItem.setFont(LABEL_FONT);
		aboutItem.setFont(LABEL_FONT);
		fileMenu.setFont(LABEL_FONT);
		logonItem.setFont(LABEL_FONT);
		logoutItem.setFont(LABEL_FONT);
		openItem.setFont(LABEL_FONT);
		fileMenu.setFont(LABEL_FONT);
		openItem.setFont(LABEL_FONT);
		exitItem.setFont(LABEL_FONT);
		custAdminItem.setFont(LABEL_FONT);
		addSalesGroupItem.setFont(LABEL_FONT);
		addSalesPersonItem.setFont(LABEL_FONT);
		adminMenu.setFont(LABEL_FONT);
		prefsItem.setFont(LABEL_FONT);
		funnelItem.setFont(LABEL_FONT);
		leadsItem.setFont(LABEL_FONT);
		dataMenu.setFont(LABEL_FONT);
		analyMenu.setFont(LABEL_FONT);
		utilityMenu.setFont(LABEL_FONT);
		syncItem.setFont(LABEL_FONT);
		chatItem.setFont(LABEL_FONT);
	}

	/**
	 * select the order tab
	 */
	public static void selectOrderTab() {
		tabbedPane.setSelectedComponent(ordersPanel);
	}

	/**
	 * start the activity checker thread...this is 
	 * the separate thread that checks for user alarms
	 */
	public static void startActivityCheck() {
		try {
			ActivityChecker checker = new ActivityChecker();
			checker.start();
		} catch (Exception e) {
			ErrorHandler.show(thisFrame, e);
		}
	}

	/**
	 * set the security options depending upon
	 * the user's login creditials
	 * @param login the creditials
	 */
	public static void setSecurity(Login login) {
		if (login.activityReadable())
			tabbedPane.setEnabledAt(ACT_TAB_INDEX, true);
		else
			tabbedPane.setEnabledAt(ACT_TAB_INDEX, false);

		if (login.forecastReadable())
			tabbedPane.setEnabledAt(FORECAST_TAB_INDEX, true);
		else
			tabbedPane.setEnabledAt(FORECAST_TAB_INDEX, false);

		if (login.quoteReadable())
			tabbedPane.setEnabledAt(QUOTE_TAB_INDEX, true);
		else
			tabbedPane.setEnabledAt(QUOTE_TAB_INDEX, false);

		if (login.orderReadable())
			tabbedPane.setEnabledAt(ORDER_TAB_INDEX, true);
		else
			tabbedPane.setEnabledAt(ORDER_TAB_INDEX, false);


		//default to no access
		addSalesPersonItem.setEnabled(false);
		addSalesGroupItem.setEnabled(false);
		custAdminItem.setEnabled(false);

		if (login.isSystemAdmin()) {
			addSalesPersonItem.setEnabled(true);
			addSalesGroupItem.setEnabled(true);
			custAdminItem.setEnabled(true);
		 	return;
		}

		if (login.userAdminReadable()) {
			addSalesPersonItem.setEnabled(true);
			addSalesGroupItem.setEnabled(true);
		}

		if (login.basicAdminReadable()) {
			custAdminItem.setEnabled(true);
		}
	}

	/**
	 * set the language of the whole app
	 */
	public final void setLang() {
		tabbedPane.setTitleAt(0, wb.getLang().getString("opportunity"));
		tabbedPane.setTitleAt(1, wb.getLang().getString("activities"));
		tabbedPane.setTitleAt(2, wb.getLang().getString("forecasts"));
		tabbedPane.setTitleAt(3, wb.getLang().getString("customer"));
		tabbedPane.setTitleAt(4, wb.getLang().getString("quotes"));
		tabbedPane.setTitleAt(5, wb.getLang().getString("orders"));
		tabbedPane.setTitleAt(6, wb.getLang().getString("contacts"));

		helpMenu.setText(wb.getLang().getString("help"));
		helpItem.setText(wb.getLang().getString("help"));
		adminMenu.setText(wb.getLang().getString("admin"));
		logonItem.setText(wb.getLang().getString("logon"));
		logoutItem.setText(wb.getLang().getString("logout"));
		openItem.setText(wb.getLang().getString("open"));
		prefsItem.setText(wb.getLang().getString("prefs"));
		addSalesPersonItem.setText(wb.getLang().getString("users"));
		addSalesGroupItem.setText(wb.getLang().getString("groups"));
		custAdminItem.setText(wb.getLang().getString("customers"));
		funnelItem.setText(wb.getLang().getString("salesFunnel"));
		leadsItem.setText(wb.getLang().getString("salesLeads"));
		dataMenu.setText(wb.getLang().getString("data"));
		analyMenu.setText(wb.getLang().getString("analytics"));
		utilityMenu.setText(wb.getLang().getString("utilities"));
		syncItem.setText(wb.getLang().getString("syncWithServer"));
		exitItem.setText(wb.getLang().getString("exit"));
		fileMenu.setText(wb.getLang().getString("file"));

		Utility.setLang(getRootPane());

		oppPanel.setLang();
		activPanel.setLang();
		forecastPanel.setLang();
		custPanel.setLang();
		quotePanel.setLang();
		ordersPanel.setLang();
		contactsPanel.setLang();

		if (helpDialog != null) helpDialog.setLang();
		if (aboutDialog != null) aboutDialog.setLang();
		if (logonDialog != null) logonDialog.setLang();
		if (oppList != null) oppList.setLang();
		if (salesGroupDialog != null) salesGroupDialog.setLang();
		if (custAdminDialog != null) custAdminDialog.setLang();
		if (salesPersonDialog != null) salesPersonDialog.setLang();
		if (chatDialog != null) chatDialog.setLang();
		if (funnelDialog != null) funnelDialog.setLang();
		if (leadsDialog != null) leadsDialog.setLang();
		if (syncDialog != null) syncDialog.setLang();
		if (prefsDialog != null) prefsDialog.setLang();
	}

	/**
	 * get a handle on the sales group dialog
	 * this is used by the user admin dialog after
	 * a user has been added to update the users
	 * in the sales group dialog
 	 * @return a reference to the SalesGroupDialog
	 */
	public final static SalesGroupDialog getSalesGroupDialog() {
		if (salesGroupDialog == null)
			salesGroupDialog = new SalesGroupDialog();
		return salesGroupDialog;
	}
}
