package sellwin.gui;

import sellwin.domain.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the opportunity list
 * dialog which a user will be shown their list
 * of opportunities and from which they can add
 * or delete opportunities.
 */
public class OppListDialog extends JDialog implements GUIChars {

	private JPopupMenu popup = new JPopupMenu();

	private Whiteboard wb;
	private JDialog thisDialog = null;	
	private final JList list = new JList();
	private JButton openButton = new JButton("Open");
	private JButton closeButton = new JButton("Close");
	private JMenuItem addMenuButton = new JMenuItem("Add");
	private JMenuItem delMenuButton = new JMenuItem("Delete ");
	private JPanel mainPanel = new JPanel(new BorderLayout());
	private ArrayList oppNames;

	/**
	 * construct the opp list dialog 
	 */
	public OppListDialog() {
		super();

		thisDialog = this;
		wb = MainWindow.getWhiteboard();

		setTitle(wb.getLang().getString("opps"));

		setSize(450, 300);

		setFonts();
		setLang();

		JScrollPane pane = new JScrollPane(list);

		mainPanel.add(pane, BorderLayout.CENTER);

		JPanel buttonPanel = new JPanel();
		buttonPanel.add(openButton);
		buttonPanel.add(closeButton);
		
		popup.add(addMenuButton);
		popup.add(delMenuButton);

		mainPanel.add(buttonPanel, BorderLayout.SOUTH);

		getContentPane().add(mainPanel);

 		WindowListener l = new WindowAdapter() {
			public void windowClosed(WindowEvent e) {
			}

            public void windowClosing(WindowEvent e) {
                hide();
            }
        };
 
        addWindowListener(l);

		closeButton.addActionListener(            
			new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					thisDialog.hide();
                }
            }
		);

		delMenuButton.addActionListener(            
			new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					int listIndex = list.getSelectedIndex();
					System.out.println("selected index =" + listIndex);
					try {
						ArrayList oppIndex = wb.getOpportunityIndex();
						OppIndex oi = (OppIndex)oppIndex.get(listIndex);
						int option = JOptionPane.showConfirmDialog(thisDialog, "Delete "+ oi.getName() + " ?");
						if (option != JOptionPane.YES_OPTION)
							return;

						thisDialog.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
						wb.deleteOpportunity(oi.getPK());
						if (oi.getPK() == wb.getCurrentOpportunity().getPK()) 
							MainWindow.clearOpportunity();
						ArrayList names = wb.getOpportunityNames();
						setData(names.toArray());
						thisDialog.setCursor(Cursor.getDefaultCursor());
					} catch (Exception f) {
						ErrorHandler.show(thisDialog, f);
					}
					thisDialog.hide();
                }
            }
		);

		addMenuButton.addActionListener(            
			new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					try {
						ArrayList groups = wb.getGroupsForUser();
						UserGroup ug;
						ArrayList groupNames = new ArrayList();
						for (int i=0;i<groups.size();i++) {
							ug = (UserGroup)groups.get(i);
							groupNames.add(ug.getName());
						}
						Object[] groupNamesList = groupNames.toArray();

						String val=JOptionPane.showInputDialog(wb.getLang().getString("enterOpp"));
						Opportunity opp = new Opportunity(val, wb.getCurrentUser().getAddress().getFormattedName());

						String gName = (String)JOptionPane.showInputDialog(
							thisDialog,
							wb.getLang().getString("chooseGroup"),
							wb.getLang().getString("selectGroup"),
							JOptionPane.PLAIN_MESSAGE,
							null,
							groupNamesList, 
							groupNamesList[0]);

						opp.setGroupName(gName);
						//the following takes some defaults, these could be added to a wizard later on
						opp.setPrimeSalesPerson(wb.getCurrentUser());

						TreeMap custNames = wb.getAllCustNames(true);
						Object[] custNamesList = custNames.values().toArray();

						if (custNamesList.length == 0) {
							JOptionPane.showMessageDialog(
								thisDialog,
								"There are no Customers defined, please add a customer first.",
								"ERROR",
								JOptionPane.ERROR_MESSAGE);
								return;
						}

						String custName = (String)JOptionPane.showInputDialog(
							thisDialog,
							wb.getLang().getString("chooseCustomer"),
							wb.getLang().getString("selectCustomer"),
							JOptionPane.PLAIN_MESSAGE,
							null,
							custNamesList, 
							custNamesList[0]);

						Customer cust = wb.getCustomer(custName);

						opp.setCustomer(cust);

						thisDialog.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
						wb.addOpportunity(opp);
						thisDialog.setCursor(Cursor.getDefaultCursor());
						wb.setCurrentOpportunity(opp);
						MainWindow.refreshOpportunity();
					} catch (Exception f) {
						ErrorHandler.show(thisDialog, f);
					}
					thisDialog.hide();
                }
            }
		);

		list.addMouseListener(
			new MouseAdapter() {
				public void mousePressed(MouseEvent e) {
					if (e.isPopupTrigger())
						popup.show(list, e.getX(), e.getY());
				}
				public void mouseReleased(MouseEvent e) {
					if (e.isPopupTrigger())
						popup.show(list, e.getX(), e.getY());
				}

                public void mouseClicked(MouseEvent e) {
					if (e.isPopupTrigger())
						popup.show(list, e.getX(), e.getY());
					else
					if (e.getClickCount() == 2) {
						clickOpp();
                	}
				}
            }
		);
		openButton.addActionListener(            
			new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					clickOpp();
                }
            }
		);

		getRootPane().setDefaultButton(openButton);

	}

	/**
	 * handle the case of an opportunity being
	 * clicked on
	 */
	public final void clickOpp() {
		String oppName = (String)list.getSelectedValue();
		try {
			Opportunity opp = wb.getOpportunityByName(oppName);
			wb.setCurrentOpportunity(opp);
			MainWindow.refreshOpportunity();
			MainWindow.setItemsOn();
			thisDialog.hide();
		} catch (AngError e) {
			ErrorHandler.show(thisDialog, e);
		}
	}

	/**
	 * load a user's opp names 
	 */
	public final void loadUser() {
		try {

			oppNames = wb.getOpportunityNames();

			setData(oppNames.toArray());
		} catch (Exception e) {
			ErrorHandler.show(thisDialog, e);
		}
	}	

	/**
	 * set the opp list data  
	 * @param items  the list of opps
	 */
	public final void setData(Object[] items) {
		DefaultListModel model = new DefaultListModel();
		for (int i=0;i<items.length;i++)
			model.addElement(items[i]);

		list.setModel(model);

		if (items.length > 0) 
			list.setSelectedIndex(0);
	}

	/**
	 * add a customer name to the list model 
	 * @param c the Customer to add info from
	 */
	public final void addCustomer(Customer c) {
		DefaultListModel model = (DefaultListModel)list.getModel();
		model.addElement(c.getName());
	}

	/**
	 * set the screen's fonts 
	 */
	public final void setFonts() {
		list.setFont(MainWindow.FIELD_FONT);
		openButton.setFont(MainWindow.LABEL_FONT);
		closeButton.setFont(MainWindow.LABEL_FONT);
		addMenuButton.setFont(MainWindow.LABEL_FONT);
		delMenuButton.setFont(MainWindow.LABEL_FONT);
	}

	public void setColors() {}

	/**
	 * set the screen's language
	 */
	public final void setLang() {
		setTitle(wb.getLang().getString("opps"));
		openButton.setText(wb.getLang().getString("open"));
		closeButton.setText(wb.getLang().getString("close"));
		addMenuButton.setText(wb.getLang().getString("add"));
		delMenuButton.setText(wb.getLang().getString("delete"));
	}
}
