package sellwin.gui;

import sellwin.domain.*;
import sellwin.utils.*;

import java.util.*;
import java.rmi.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
 
// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class lets a user select a product
 */
public class ProductSelectionDialog extends JDialog implements GUIChars {
	private Whiteboard wb = null;
	private String currentGroup, currentLine;
	private ArrayList currentProds;
	private JDialog thisDialog;
	private InventoryListener parent = null;

    private JPanel buttonPanel;
    private JButton addButton;
    private JButton cancelButton;
    private JPanel mainPanel;
    private JSplitPane splitPane;
    private JScrollPane treeScrollPane;
    private JTree tree;
    private JScrollPane listScrollPane;
    private JList list;
    private JPanel countPanel;
    private JLabel countLabel;
    private JTextField count;

    /** Creates new form ProductSelectionDialog */
    public ProductSelectionDialog(InventoryListener p) {
        super(MainWindow.getMainParent(), true);

		thisDialog = this;
		parent = p;

		wb = MainWindow.getWhiteboard();
		setSize(430, 450);

        initComponents();
		setLang();

		addButton.addActionListener(
            new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    QuoteLine x = getQuoteLineFromScreen();
                    parent.addQuoteLine(x);
                    thisDialog.hide();
                }
            });

        getRootPane().setDefaultButton(addButton);

        cancelButton.addActionListener(
            new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    thisDialog.hide();
                }
            });

		loadProducts();

        tree.addTreeSelectionListener(
            new TreeSelectionListener() {
                public void valueChanged(TreeSelectionEvent e) {
                    TreePath path = e.getPath();
                    Object[] nodes = path.getPath();
                    DefaultMutableTreeNode parentNode, node;
                    for (int i=0;i<nodes.length;i++) {
                        node = (DefaultMutableTreeNode)nodes[i];
                        if (node.isLeaf()) {
                            parentNode = (DefaultMutableTreeNode)node.getParent();
                            updateModelsList(parentNode.toString(), node.toString());
                        }
                    }
                }
            }
        );

        pack();

    }

	/**
	 * update the model list on the screen
	 * @param group the product group to search with
	 * @param line the product line to search with
	 */
    public final void updateModelsList(String group, String line) {
        cleanModelList();
        try {
            //
            //this is where some caching will be required if
            //the user has already downloaded products for this group
            //and line combo
            //

            currentGroup = group;
            currentLine = line;

            currentProds = wb.getProductsForLine(group, line);
            ListModel lmodel = list.getModel();
            if (lmodel != null) {
                DefaultListModel model = (DefaultListModel)lmodel;
                Product prod;
                for (int i=0;i<currentProds.size();i++) {
                    prod = (Product)(currentProds.get(i));
                    model.addElement(prod.getName());
                }
                if (currentProds.size() > 0)
                    list.setSelectedIndex(0);
            }

        } catch (Exception e) {
            ErrorHandler.show(this, e);
        }
    }

	/**
	 * load the products onto the screen 
	 */
    public final void loadProducts() {
        try {
            ArrayList matrix = wb.getProductMatrix();
            tree.setModel(createTreeModel(matrix));
            cleanModelList();
        } catch (Exception e) {
            ErrorHandler.show(this, e);
        }
    }

	/**
	 * clean the model list 
	 */
    private final void cleanModelList() {
        ListModel lmodel = list.getModel();
        if (lmodel != null) {
            DefaultListModel model = (DefaultListModel)lmodel;
            model.removeAllElements();
        }
    }

	/**
	 *  create the product tree using a new
	 * product matrix
	 * @param matrix an array of product info to display
	 * @return the tree's model of data
	 */
    private final DefaultTreeModel createTreeModel(ArrayList matrix) {
        DefaultMutableTreeNode root = new DefaultMutableTreeNode("Groups");
        DefaultMutableTreeNode groupNode, lineNode;
        ArrayList lines;
        String line;
        MatrixObject mo;
        for (int i=0;i<matrix.size();i++) {
            mo = (MatrixObject)matrix.get(i);
			mo.print();
            groupNode = new DefaultMutableTreeNode(mo.getGroup());
            lines = mo.getLines();
            for (int j=0;j<lines.size();j++) {
                line = (String)lines.get(j);
                lineNode = new DefaultMutableTreeNode(line);
                groupNode.add(lineNode);
            }
            root.add(groupNode);
        }
        DefaultTreeModel model = new DefaultTreeModel(root);
        return model;
    }

	/**
	 * get a quote line from the screen
	 * @return the QuoteLine
	 */
    public final QuoteLine getQuoteLineFromScreen() {
        QuoteLine line = new QuoteLine();
        String name = (String)list.getSelectedValue();
        //retrieve the rest of the product info for the product selected
        try {
            Product product = wb.getProduct(currentGroup, currentLine, name);
			line.setProduct(product);
            line.setQuantity(new Integer(count.getText()));
        } catch (AngError e) {
            ErrorHandler.show(this, e);
        }

        return line;
    }

    /** This method is called from within the constructor to
     * initialize the form.
     */
    private final void initComponents() {
        buttonPanel = new JPanel();
        addButton = new JButton();
        cancelButton = new JButton();
        mainPanel = new JPanel();
        splitPane = new JSplitPane();
        treeScrollPane = new JScrollPane();
        tree = new JTree();
        listScrollPane = new JScrollPane();
        list = new JList(new DefaultListModel());
        countPanel = new JPanel();
        countLabel = new JLabel();
        count = new JTextField("1");
        
        setTitle(wb.getLang().getString("prodDialog"));
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog(evt);
            }
        });
        
        buttonPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));
        
        addButton.setText(wb.getLang().getString("ok"));
        buttonPanel.add(addButton);
        
        cancelButton.setText(wb.getLang().getString("cancel"));
        buttonPanel.add(cancelButton);
        
        getContentPane().add(buttonPanel, java.awt.BorderLayout.SOUTH);
        
        mainPanel.setLayout(new java.awt.BorderLayout());
        
        mainPanel.setBorder(new javax.swing.border.EtchedBorder());
        treeScrollPane.setViewportBorder(new javax.swing.border.EtchedBorder());
        treeScrollPane.setViewportView(tree);
        
        splitPane.setLeftComponent(treeScrollPane);
        
        listScrollPane.setViewportBorder(new javax.swing.border.EtchedBorder());
        listScrollPane.setViewportView(list);
        
        splitPane.setRightComponent(listScrollPane);
        
        mainPanel.add(splitPane, java.awt.BorderLayout.CENTER);
        
        countPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.LEFT));
        
        countLabel.setText(wb.getLang().getString("quantity"));
        countLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        countPanel.add(countLabel);
        
        count.setText("1");
        count.setPreferredSize(new Dimension(80, Prefs.FIELD_HEIGHT));
        count.setMinimumSize(new Dimension(80, Prefs.FIELD_HEIGHT));
        countPanel.add(count);
        
        mainPanel.add(countPanel, java.awt.BorderLayout.SOUTH);
        
        getContentPane().add(mainPanel, java.awt.BorderLayout.CENTER);
        
        pack();
    }

    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt) {
        setVisible(false);
        dispose();
    }

	/**
	 * set the screen's language
	 */
	public final void setLang() {
        setTitle(wb.getLang().getString("prodDialog"));
        countLabel.setText(wb.getLang().getString("quantity"));
        addButton.setText(wb.getLang().getString("ok"));
        cancelButton.setText(wb.getLang().getString("cancel"));
        countLabel.setText(wb.getLang().getString("quantity"));
	}

	public void setFonts() {}
	public void setColors() {}

}
