package sellwin.gui;

import sellwin.domain.*;
import sellwin.server.*;
import sellwin.utils.*;

import java.util.*;
import java.rmi.*;
import javax.naming.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.awt.*;
import java.awt.event.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class handles the process of uploading changed and/or
 * deleted objects from the local data to the remote database.
 * It also handles the downloading of a user's data FROM
 * the remote database to the local laptop disk.  These two
 * tasks are initiated via the GUI by the user whenever they
 * want to SYNChronize their work.
 */
public class SyncDialog extends JDialog implements GUIChars {

	private Whiteboard wb=null;
	private BizServices localServer;
	private SellwinSession remoteServer;
	private java.awt.Frame parent;
    private JPanel buttonPanel, mainPanel, timesPanel;
    private JButton syncButton, completeRefreshButton, closeButton;

	private JLabel lastSyncLabel, currEndLabel, currStartLabel;
	private JTextField currEndField, currStartField, lastSyncField;
	private static final int FIELD_WIDTH=140;

	private ArrayList stats = new ArrayList();
	private JTable statTable = new JTable();
	private JScrollPane statTableScrollPane = new JScrollPane();
	private StatTableModel statModel;
	private String READY_STAT = "Ready";
	private String RUNNING_STAT = "Running";
	private String FINISH_STAT = "Finished";
	private final static int STAT_ROLES 		=0;
	private final static int STAT_GROUPS		=1;
	private final static int STAT_GROUP_MEMBERS =2;
	private final static int STAT_SALES_PERSONS =3;
	private final static int STAT_PRODUCTS 		=4;
	private final static int STAT_CAMPAIGNS		=5;
	private final static int STAT_LEADS			=6;
	private final static int STAT_CUSTOMERS		=7;
	private final static int STAT_OPPS			=8;
	private final static int STAT_TAX			=9;

    /** 
	 * Creates new form SyncDialog 
	 * @param parent the SyncDialog's parent 
	 * @param modal signifies the type of SyncDialog modality
	 */
    public SyncDialog(java.awt.Frame parent, boolean modal) {
        super(parent, modal);

		this.parent = parent;

		wb = MainWindow.getWhiteboard();
        initComponents();
		initStatTable();

		setLang();
		setFonts();
		setColors();
		getInterfaces();
		setSize(455, 365);
    }

	/**
	 * get the local and remote interfaces, both
	 * are required to get data from the server and
	 * store it locally
	 */
	private final void getInterfaces() {
		SellwinSession w = wb.getRemIF();
		try {
			if (w instanceof BizServices) { 
				localServer = (BizServices)w;
				Properties props = wb.getProps();
				String serverName = (String)(props.getProperty(Prefs.SERVER_NAME));
				InitialContext ic = Utility.getEJBContext(serverName, "1099");
				SellwinSessionHome sellwinHome = (SellwinSessionHome) ic.lookup("SellWinSessionBean");

				remoteServer = sellwinHome.create();
			}
			else {
				remoteServer = w;
				localServer = new BizServices(Prefs.MYSQL);
				localServer.init2Tier("jdbc:mysql:///sellwin", "jmccormi", "jeffery2");
			}
		} catch (Exception e) {
			ErrorHandler.show(parent, e);
		}
	}

    /** 
	 * This method is called from within the constructor to
     * initialize the form.
     */
    private final void initComponents() {
		statTable = new JTable();
        buttonPanel = new JPanel();
        syncButton = new JButton();
        completeRefreshButton = new JButton();
        closeButton = new JButton();
        mainPanel = new JPanel();

        lastSyncLabel = new javax.swing.JLabel();
        lastSyncField = new javax.swing.JTextField();
		lastSyncField.setMinimumSize(new Dimension(FIELD_WIDTH, Prefs.FIELD_HEIGHT));
		lastSyncField.setPreferredSize(new Dimension(FIELD_WIDTH, Prefs.FIELD_HEIGHT));
		if (wb.getCurrentUser().getSyncDate() == null)
			lastSyncField.setText("");
		else
			lastSyncField.setText(
				Prefs.dateTimeFormat.format(wb.getCurrentUser().getSyncDate()));

        currStartLabel = new javax.swing.JLabel();
        currStartField = new javax.swing.JTextField();
		currStartField.setMinimumSize(new Dimension(FIELD_WIDTH, Prefs.FIELD_HEIGHT));
		currStartField.setPreferredSize(new Dimension(FIELD_WIDTH, Prefs.FIELD_HEIGHT));
        currEndLabel = new javax.swing.JLabel();
        currEndField = new javax.swing.JTextField();
		currEndField.setMinimumSize(new Dimension(FIELD_WIDTH, Prefs.FIELD_HEIGHT));
		currEndField.setPreferredSize(new Dimension(FIELD_WIDTH, Prefs.FIELD_HEIGHT));

     
        setTitle(wb.getLang().getString("syncDialog"));
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog();
            }
        });
        
        syncButton.setText(wb.getLang().getString("sync"));
        syncButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
				java.util.Date lastSyncDate = wb.getCurrentUser().getSyncDate();
				System.out.println("syncing with sync_date="+lastSyncDate);
                syncAction(evt, lastSyncDate);
            }
        });
        
        buttonPanel.add(syncButton);

        completeRefreshButton.setText(wb.getLang().getString("completeRefresh"));
        completeRefreshButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {

				//use a date in 1969 as the sync date to cause a complete
				//refresh of data during the synchronization

				java.util.Date lastSyncDate = new java.util.Date(1);

				//
				//remove the local data
				//
				try {
					localServer.truncateDB();

                	syncAction(evt, lastSyncDate);
				} catch (Exception e) {
					ErrorHandler.show(parent, e);
				}
            }
        });
        
        buttonPanel.add(completeRefreshButton);
        
        closeButton.setText(wb.getLang().getString("close"));
        closeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                closeDialog();
            }
        });
        
        buttonPanel.add(closeButton);
        
        getContentPane().add(buttonPanel, java.awt.BorderLayout.SOUTH);
       
		statTable.setPreferredSize(new Dimension(350, 180));
		statTable.setMaximumSize(new Dimension(350, 180));
		statTable.setMinimumSize(new Dimension(350, 180));
		statTable.setAutoResizeMode(javax.swing.JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
		statTableScrollPane.setMinimumSize(new Dimension(350, 180));
		statTableScrollPane.setMaximumSize(new Dimension(350, 180));
		statTableScrollPane.setPreferredSize(new Dimension(350, 180));
		statTableScrollPane.setHorizontalScrollBarPolicy(javax.swing.JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
		statTableScrollPane.setVerticalScrollBarPolicy(javax.swing.JScrollPane.VERTICAL_SCROLLBAR_NEVER);
		statTableScrollPane.setViewportView(statTable);

        mainPanel.setLayout(new java.awt.GridBagLayout());
        java.awt.GridBagConstraints gridBagConstraints1;
        
        mainPanel.setBorder(new javax.swing.border.EtchedBorder());
        
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 0;
        gridBagConstraints1.gridwidth = 2;
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(statTableScrollPane, gridBagConstraints1);
        
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 1;
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(currStartLabel, gridBagConstraints1);
        
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 1;
        gridBagConstraints1.gridy = 1;
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(currStartField, gridBagConstraints1);
        
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 2;
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(currEndLabel, gridBagConstraints1);
        
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 1;
        gridBagConstraints1.gridy = 2;
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(currEndField, gridBagConstraints1);

        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 3;
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(lastSyncLabel, gridBagConstraints1);
        
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 1;
        gridBagConstraints1.gridy = 3;
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(lastSyncField, gridBagConstraints1);
        
        getContentPane().add(mainPanel, java.awt.BorderLayout.CENTER);
        
        pack();

    }

	/**
	 * the action that gets performed when a user presses
	 * the 'Sync' button
	 * @param evt the action event
	 */
    private final void syncAction(java.awt.event.ActionEvent evt, java.util.Date lastSyncDate) {

		try {

			currStartField.setText(
				Prefs.dateTimeFormat.format(new java.util.Date())
			);
			currEndField.setText("");

			//perform upload tasks before the download tasks
			//get the list of locally deleted objects 

			//
			//process locally deleted data
			//
			ArrayList deletes = localServer.getDeletes();
			DeleteInfo d;
			System.out.println("Delete List follows ("+deletes.size()+")...");
			for (int i=0;i<deletes.size();i++) {
				d = (DeleteInfo)deletes.get(i);
				System.out.println("PK="+d.pk +" Class="+ d.className);
			}
			remoteServer.uploadDeletes(deletes);
			localServer.resetDeletes();

			//
			//look for and process all the locally updated data
			//
			ArrayList opps = localServer.getOpportunities(wb.getCurrentUser(), null);
			uploadChanges(opps);		


			//
			//load from remote to local
			//


			/**
			 * ROLES
			 */
			int localUpdates=0, localInserts=0;
			setStat(STAT_ROLES, RUNNING_STAT, localInserts, localUpdates);

			ArrayList roles = remoteServer.getAllUserRoles(lastSyncDate);
			System.out.println("got " + roles.size() + " roles from remote");
			UserRole localUserRole, remoteUserRole;
			for (int i=0;i<roles.size();i++) {
				remoteUserRole = (UserRole)roles.get(i);
				localUserRole = localServer.getUserRole(remoteUserRole.getName());
				if (localUserRole == null) {
					localServer.loadUserRole(remoteUserRole);
					localInserts++;
				} else {
					localServer.updateUserRole(remoteUserRole);
					localUpdates++;
				}
			}

			setStat(STAT_ROLES, FINISH_STAT, localInserts, localUpdates);

			/**
			 * GROUPS
			 */
			localInserts=0; localUpdates=0;
			setStat(STAT_GROUPS, RUNNING_STAT, localInserts, localUpdates);

			Object[] values = remoteServer.getUserGroups(lastSyncDate);
			System.out.println("got " + values.length + " user groups from remote");
			UserGroup remoteUserGroup, localUserGroup;
			for (int j=0;j<values.length;j++) {
				remoteUserGroup = (UserGroup)values[j];
				localUserGroup = localServer.getUserGroup(remoteUserGroup.getPK());
				if (localUserGroup == null) {
					localServer.loadUserGroup(remoteUserGroup);
					localInserts++;
				} else {
					localUpdates++; 
					//no need to update these as this version had no 
					//fields to update...just update the stats to show
					//some kind of status anyway
				}
			}

			setStat(STAT_GROUPS, FINISH_STAT, localInserts, localUpdates);

			/**
			 * SALES PERSONS
			 */
			localInserts=0; localUpdates=0;
			setStat(STAT_SALES_PERSONS, RUNNING_STAT, localInserts, localUpdates);

			values = remoteServer.getSalesPersons(lastSyncDate);
			System.out.println("got " + values.length + " sales persons from remote");
			SalesPerson remoteSalesPerson, localSalesPerson;
			for (int k=0;k<values.length;k++) {
				remoteSalesPerson = (SalesPerson)values[k];
				localSalesPerson = localServer.getSalesPerson(remoteSalesPerson.getPK());

				if (localSalesPerson == null) {
					localServer.loadSalesPerson(remoteSalesPerson);
					localInserts++;
				} else {
					localServer.updateSalesPerson(remoteSalesPerson);
					localUpdates++;
				}
			}

			setStat(STAT_SALES_PERSONS, FINISH_STAT, localInserts, localUpdates);

			/**
			 * GROUP MEMBERS
			 */
			localInserts=0; localUpdates=0;
			setStat(STAT_GROUP_MEMBERS, RUNNING_STAT, localInserts, localUpdates);

			ArrayList remoteMembers = remoteServer.getGroupMembers(lastSyncDate);
			System.out.println("got " + remoteMembers.size() + " group members from remote");
			UserGroupMember remoteGroupMember, localGroupMember;
			for (int jj=0;jj<remoteMembers.size();jj++) {
				remoteGroupMember = (UserGroupMember)remoteMembers.get(jj);
				localGroupMember = localServer.getGroupMember(remoteGroupMember.getPK());
				if (localGroupMember == null) {
					localServer.loadGroupMember(remoteGroupMember);
					localInserts++;
				} else {
					localUpdates++; 
					//nothing to update on this object so skip the update
					//show updates in the stats for some visual
				}
			}

			setStat(STAT_GROUP_MEMBERS, FINISH_STAT, localInserts, localUpdates);

			/**
	 	     * PRODUCTS
			 */
			localInserts=0; localUpdates=0;
			setStat(STAT_PRODUCTS, RUNNING_STAT, localInserts, localUpdates);

			ArrayList remoteProds = remoteServer.getProducts(lastSyncDate);
			System.out.println("got " + remoteProds.size() + " products from remote");
			Product localProd, remoteProd;
			for (int m=0;m<remoteProds.size();m++) {
				remoteProd = (Product)remoteProds.get(m);
				localProd = localServer.getProduct(remoteProd.getPK());
				if (localProd == null) {
					localServer.loadProduct(remoteProd);
					localInserts++;
				} else {
					localServer.updateProduct(remoteProd);
					localUpdates++;
				}
			}

			setStat(STAT_PRODUCTS, FINISH_STAT, localInserts, localUpdates);

			/**
			 * CAMPAIGNS
			 */
			localInserts=0; localUpdates=0;
			setStat(STAT_CAMPAIGNS, RUNNING_STAT, localInserts, localUpdates);

			ArrayList remoteCamps = remoteServer.getCampaigns(lastSyncDate);
			System.out.println("got " + remoteCamps.size() + " campaigns from remote");
			Campaign remoteCampaign, localCampaign;
			for (int n=0;n<remoteCamps.size();n++) {
				remoteCampaign = (Campaign)remoteCamps.get(n);
				localCampaign = localServer.getCampaign(remoteCampaign.getPK());
				if (localCampaign == null) {
					localServer.loadCampaign(remoteCampaign);
					localInserts++;
				} else {
					localUpdates++; 
					//nothing to update right now, maybe in the future
					//show updates for a visual in the stats table
				}
			}
			setStat(STAT_CAMPAIGNS, FINISH_STAT, localInserts, localUpdates);

			/**
	 		 * LEADS
			 */
			localInserts=0; localUpdates=0;
			setStat(STAT_LEADS, RUNNING_STAT, localInserts, localUpdates);

			ArrayList remoteLeads = remoteServer.getCampaignLeads(lastSyncDate);
			System.out.println("got " + remoteLeads.size() + " leads from remote");
			Lead remoteLead, localLead;
			for (int aj=0;aj < remoteLeads.size();aj++) {
				remoteLead = (Lead)remoteLeads.get(aj);
				localLead = localServer.getLead(remoteLead.getPK());
				if (localLead == null) {
					localServer.loadLead(remoteLead);
					localInserts++;
				} else {
					localServer.updateLead(remoteLead.getCampaignKey(), remoteLead);
					localUpdates++;
				}
			}
			setStat(STAT_LEADS, FINISH_STAT, localInserts, localUpdates);

			/**
			 * CUSTOMERS
			 */
			localInserts=0; localUpdates=0;
			setStat(STAT_CUSTOMERS, RUNNING_STAT, localInserts, localUpdates);

			ArrayList remoteCusts = remoteServer.getCustomers(lastSyncDate);
			System.out.println("got " + remoteCusts.size() + " custs from remote");
			Customer localCust, remoteCust;
			for (int p=0;p<remoteCusts.size();p++) {
				remoteCust = (Customer)remoteCusts.get(p);
				localCust = localServer.getCustomer(remoteCust.getName());
				if (localCust == null) {
					localServer.loadCustomer(remoteCust);
					localInserts++;
				} else {
					localServer.updateCustomer(remoteCust);
					localUpdates++;
				}
			}

			setStat(STAT_CUSTOMERS, FINISH_STAT, localInserts, localUpdates);

			/**
			 * OPPS
			 */
			localInserts=0; localUpdates=0;
			setStat(STAT_OPPS, RUNNING_STAT, localInserts, localUpdates);

			ArrayList remoteOpps = remoteServer.getOpportunities(wb.getCurrentUser(), lastSyncDate);
			System.out.println("got " + remoteOpps.size() + " opps from remote");
			Opportunity remoteOpp, localOpp;
			for (int x=0;x<remoteOpps.size();x++) {
				remoteOpp = (Opportunity)remoteOpps.get(x);
				localOpp = localServer.getOpportunity(remoteOpp.getPK());
				if (localOpp == null) {
					localServer.loadOpportunity(remoteOpp);
					localInserts++;
				} else {
					localServer.updateOpportunity(remoteOpp);
					localUpdates++;
				}
			}
			setStat(STAT_OPPS, FINISH_STAT, localInserts, localUpdates);

			/**
			 * STATE_TAX
			 */
			localInserts=0; localUpdates=0;
			setStat(STAT_TAX, RUNNING_STAT, localInserts, localUpdates);
	
			ArrayList remoteTaxes = remoteServer.getStateTax(lastSyncDate);
			System.out.println("got " + remoteTaxes.size() + " taxes from remote");
			StateTax remoteTax, localTax;
			for (int r=0;r<remoteTaxes.size();r++) {
				remoteTax = (StateTax)remoteTaxes.get(r);
				localTax = localServer.getTax(remoteTax.getCode());
				if (localTax == null) {
					localServer.loadTax(remoteTax);
					localInserts++;
				} else {
					localServer.updateTax(remoteTax);
					localUpdates++;
				}
			}

			setStat(STAT_TAX, FINISH_STAT, localInserts, localUpdates);

			/**
			 * update this user's sync date to reflect this synchronization
	 		 * has occurred
			 */
			wb.getCurrentUser().setSyncDate(new java.util.Date());
			wb.updateSalesPerson(wb.getCurrentUser());

			currEndField.setText(Prefs.dateTimeFormat.format(new java.util.Date()));

		} catch (Exception e) {
			e.printStackTrace();
		}
    }

    /** 
	 * Closes the dialog 
	 * @param evt the action event
	 */
    private final void closeDialog() {
		resetStats();
        setVisible(false);
        dispose();
    }



	public void setColors() {
	}

	public void setFonts() {
		statTable.setFont(MainWindow.FIELD_FONT);
		syncButton.setFont(MainWindow.LABEL_FONT);
		closeButton.setFont(MainWindow.LABEL_FONT);
		completeRefreshButton.setFont(MainWindow.LABEL_FONT);
	}

	/**
	 * set the language type for this dialog
	 */
	public final void setLang() {
		System.out.println("SyncDialog setLang() called");
		statModel.setLang();
		currEndLabel.setText(wb.getLang().getString("currEnd"));
		currStartLabel.setText(wb.getLang().getString("currStart"));
		lastSyncLabel.setText(wb.getLang().getString("lastSync"));

        setTitle(wb.getLang().getString("syncDialog"));
        syncButton.setText(wb.getLang().getString("sync"));
        completeRefreshButton.setText(wb.getLang().getString("completeRefresh"));
        closeButton.setText(wb.getLang().getString("close"));

		READY_STAT = wb.getLang().getString("ready");
		RUNNING_STAT = wb.getLang().getString("running");
		FINISH_STAT = wb.getLang().getString("finished");

		Stat stat;
		stat = (Stat)stats.get(STAT_ROLES);
		stat.dataType = (String)wb.getLang().getString("roles");
		stat = (Stat)stats.get(STAT_GROUPS);
		stat.dataType = (String)wb.getLang().getString("groups");
		stat = (Stat)stats.get(STAT_GROUP_MEMBERS);
		stat.dataType = (String)wb.getLang().getString("groupMembers");
		stat = (Stat)stats.get(STAT_SALES_PERSONS);
		stat.dataType = (String)wb.getLang().getString("salesPersons");
		stat = (Stat)stats.get(STAT_PRODUCTS);
		stat.dataType = (String)wb.getLang().getString("products");
		stat = (Stat)stats.get(STAT_CAMPAIGNS);
		stat.dataType = (String)wb.getLang().getString("campaigns");
		stat = (Stat)stats.get(STAT_LEADS);
		stat.dataType = (String)wb.getLang().getString("leads");
		stat = (Stat)stats.get(STAT_CUSTOMERS);
		stat.dataType = (String)wb.getLang().getString("customers");
		stat = (Stat)stats.get(STAT_OPPS);
		stat.dataType = (String)wb.getLang().getString("opps");
		stat = (Stat)stats.get(STAT_TAX);
		stat.dataType = (String)wb.getLang().getString("tax");
		reloadStatTable();
	}

	/**
	 * upload all changes to the local set of
	 * opportuntiies from the local disk to
	 * the remote server
	 * @param opps the current set of Opportunities
	 */
	private final void uploadChanges(ArrayList opps) {
		Opportunity o;

		for (int i=0;i<opps.size();i++) {
			o = (Opportunity)opps.get(i);
			uploadOpportunityChanges(o);
			uploadOrderChanges(o);
			uploadActivityChanges(o);
			uploadContactChanges(o);
			uploadQuoteChanges(o);
			uploadForecastChanges(o);
		}
	}

	/**
	 * upload the changes for a given Opportunity
	 * @param o the Opportunity we are to process
	 */
	private final void uploadOpportunityChanges(Opportunity o) {
		try {
			if (o.getAddedLocally()) {
				System.out.println("upload log: Opp " + o.getPK() + " was added");
				remoteServer.addOpportunity(o);
			}

			if (o.getUpdatedLocally()) {
				System.out.println("upload log: Opp " + o.getPK() + " was updated");
				remoteServer.updateOpportunity(o);
			}
		} catch (Exception e) {
				System.out.println("UPLOAD ERROR: Opportunity : OPP pk=" + o.getPK());
				System.out.println(e.getMessage());
		}
	}

	/**
	 * upload some Order changes within a given
	 * Opportunity
	 * @param o the Opportunity we are processing
	 */
	private final void uploadOrderChanges(Opportunity o) {
		ArrayList orders = o.getOrders();
		Order order;
		for (int i=0;i<orders.size();i++) {
			order = (Order)orders.get(i);
			try {
				if (order.getAddedLocally()) {
					System.out.println("upload log: Order " + order.getPK() + " was added");
					remoteServer.addOrder(o.getPK(), order);
				}

				if (order.getUpdatedLocally()) {
					System.out.println("upload log: Order " + order.getPK() + " was updated");
					remoteServer.updateOrder(o.getPK(), order);
				}
			} catch (Exception e) {
				System.out.println("UPLOAD ERROR: Order : pk="+order.getPK() + " OPP pk=" + o.getPK());
				System.out.println(e.getMessage());
			}
		}
	}

	/**
	 * upload activity changes for a given opportunity
	 * @param o the Opportunity we are processing
	 */
	private final void uploadActivityChanges(Opportunity o) {
		ArrayList acts = o.getActivities();
		Activity act;
		for (int i=0;i<acts.size();i++) {
			act = (Activity)acts.get(i);
			try {
				if (act.getAddedLocally()) {
					System.out.println("upload log: Activity " + act.getPK() + " was added");
					remoteServer.addActivity(o.getPK(), act);
				}

				if (act.getUpdatedLocally()) {
					System.out.println("upload log: Activity " + act.getPK() + " was updated");
					remoteServer.updateActivity(o.getPK(), act);
				}
			} catch (Exception e) {
				System.out.println("UPLOAD ERROR: Activity : pk="+act.getPK() + " OPP pk=" + o.getPK());
				System.out.println(e.getMessage());
			}
		}
	}

	/**
	 * upload contact changes for a given opportunity
	 * @param o the Opportunity we are processing
	 */
	private final void uploadContactChanges(Opportunity o) {
		ArrayList contacts = o.getContacts();
		Contact con;
		for (int i=0;i<contacts.size();i++) {
			con = (Contact)contacts.get(i);

			try {
				if (con.getAddedLocally()) {
					System.out.println("upload log: Contact " + con.getPK() + " was added");
					remoteServer.addContact(o.getPK(), con);
				}

				if (con.getUpdatedLocally()) {
					System.out.println("upload log: Contact " + con.getPK() + " was updated");
					remoteServer.updateContact(o.getPK(), con);
				}
			} catch (Exception e) {
				System.out.println("UPLOAD ERROR: Contact : pk="+con.getPK() + " OPP pk=" + o.getPK());
				System.out.println(e.getMessage());
			}	
		}
	}

	/**
	 * upload some quote changes for a given opportunity
	 * @param o the Opportunity we are processing
	 */
	private final void uploadQuoteChanges(Opportunity o) {
		ArrayList quotes = o.getQuotes();
		Quote quote;

		for (int i=0;i<quotes.size();i++) {
			quote = (Quote)quotes.get(i);
			try {
				if (quote.getAddedLocally()) {
					System.out.println("upload log: Quote " + quote.getPK() + " was added");
					remoteServer.addQuote(o.getPK(), quote);
				}

				if (quote.getUpdatedLocally()) {
					System.out.println("upload log: Quote " + quote.getPK() + " was updated");
					remoteServer.updateQuote(o.getPK(), quote);
				}
			} catch (Exception e) {
				System.out.println("UPLOAD ERROR: Quote : pk="+quote.getPK() + " OPP pk=" + quote.getPK());
				System.out.println(e.getMessage());
			}	
		}
	}

	/**
	 * upload forecast changes for a given Opportunity
	 * @param o the Opportunity we are processing
	 */
	private final void uploadForecastChanges(Opportunity o) {
		ArrayList forecasts = o.getForecasts();
		Forecast f;
		for (int i=0;i<forecasts.size();i++) {
			f = (Forecast)forecasts.get(i);
			try {
				if (f.getAddedLocally()) {
					System.out.println("upload log: Forecast " + f.getPK() + " was added");
					remoteServer.addForecast(o.getPK(), f);
				}

				if (f.getUpdatedLocally()) {
					System.out.println("upload log: Forecast " + f.getPK() + " was updated");
					remoteServer.updateForecast(o.getPK(), f);
				}
			} catch (Exception e) {
				System.out.println("UPLOAD ERROR: Forecast : pk="+f.getPK() + " OPP pk=" + o.getPK());
				System.out.println(e.getMessage());
			}
		}
	}

	private final void resetStats() {

		for (int i=0;i<stats.size();i++) 
			setStat(i, READY_STAT, 0, 0);

		if (wb.getCurrentUser().getSyncDate() == null)
			lastSyncField.setText("");
		else
			lastSyncField.setText(
				Prefs.dateTimeFormat.format(wb.getCurrentUser().getSyncDate()));
        currStartField.setText("");
        currEndField.setText("");
	}

	private final void setStat(int dataType, String syncStatus, 
		int localInserts, int localUpdates) {

		Stat stat = (Stat)stats.get(dataType);
		stat.syncStatus = syncStatus;
		stat.localInserts = new Integer(localInserts);
		stat.localUpdates = new Integer(localUpdates);

		statTable.tableChanged(new TableModelEvent(statModel));
	}

	private final void initStatTable() {
		stats = new ArrayList();
		Stat stat;
		stat = new Stat(wb.getLang().getString("roles"), READY_STAT, 0, 0);
		stats.add(stat);
		stat = new Stat(wb.getLang().getString("groups"), READY_STAT, 0, 0);
		stats.add(stat);
		stat = new Stat(wb.getLang().getString("groupMembers"), READY_STAT, 0, 0);
		stats.add(stat);
		stat = new Stat(wb.getLang().getString("salesPersons"), READY_STAT, 0, 0);
		stats.add(stat);
		stat = new Stat(wb.getLang().getString("products"), READY_STAT, 0, 0);
		stats.add(stat);
		stat = new Stat(wb.getLang().getString("campaigns"), READY_STAT, 0, 0);
		stats.add(stat);
		stat = new Stat(wb.getLang().getString("leads"), READY_STAT, 0, 0);
		stats.add(stat);
		stat = new Stat(wb.getLang().getString("customers"), READY_STAT, 0, 0);
		stats.add(stat);
		stat = new Stat(wb.getLang().getString("opps"), READY_STAT, 0, 0);
		stats.add(stat);
		stat = new Stat(wb.getLang().getString("tax"), READY_STAT, 0, 0);
		stats.add(stat);

		reloadStatTable();
	}

	private final void reloadStatTable() {
		statModel = new StatTableModel(stats);
		statTable.setModel(statModel);
	}
}
