package sellwin.server;

import sellwin.domain.*;
import sellwin.db.*;
import sellwin.utils.*;

import java.rmi.*;
import java.sql.*;
import java.util.*;

import javax.ejb.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class is the remote server implementation of 
 * the SellwinSession interface.  All client to server
 * communications goes thru this implementation at the
 * moment.  I've coded this to work in 2-tier form
 * as well as via EJB or CORBA.  Typically, calls to
 * this class are initiated via the GUI, specifically
 * the Whiteboard class.  The Whiteboard class keeps
 * the rest of the GUI ignorant of how data is retrieved
 * from the remote interface, this is good mister fubu.
 */
public class BizServices implements SellwinSession {

	private int DB_TYPE;

	//some security state info for a given  user session is
	//stored in 'login'
	private Login login;

	//create the low-level database objects
	//these are declared protected since the EJB interface
	//will extend this class and needs access to these
	protected ActivityDB activityDB;
	protected AddressDB addressDB;
	protected Admin adminDB;
	protected AttendeeDB attendeeDB;
	protected CampaignDB campaignDB;
	protected CustomerDB customerDB;
	protected DeleteInfoDB deleteInfoDB;
	protected StateTaxDB stateTaxDB;
	protected ContactDB contactDB;
	protected CustomerInventoryDB custInventoryDB;
	protected ForecastDB forecastDB;
	protected LeadDB leadDB;
	protected OpportunityDB oppDB;
	protected OrderDB orderDB;
	protected ProductDB productDB;
	protected QuoteDB quoteDB;
	protected QuoteLineDB quoteLineDB;
	protected RolePermissionDB rolePermissionDB;
	protected SalesPersonDB salesPersonDB;
	protected SalesPersonRoleDB salesPersonRoleDB;
	protected UserGroupDB userGroupDB;
	protected UserGroupMemberDB ugMemberDB;
	protected UserRoleDB userRoleDB;

	private Connection con=null;

	/**
	 * construct a BizServices
	 */
	public BizServices() {
		Properties props = System.getProperties();
		String temp = props.getProperty("DBTYPE");
		if (temp == null) {
			DB_TYPE = Prefs.ORACLE;
			System.out.println("ORACLE DB [DEFAULT] TYPE WILL BE USED");
		}
		else
		if (temp.equals("MYSQL")) {
			System.out.println("MYSQL DB TYPE WILL BE USED");
			DB_TYPE = Prefs.MYSQL;
		}
		else {
			DB_TYPE = Prefs.ORACLE;	
			System.out.println("unknown DBTYPE specified = [" + temp + "]");
			System.out.println("ORACLE DB TYPE WILL BE USED AS DEFAULT");
		}
		init(DB_TYPE);
	}

	/***
	 * construct a BizServices specifying a database type
	 * @param dbType the database type , types found in Prefs.java
	 */
	public BizServices(int dbType) {
		DB_TYPE = dbType;
		init(DB_TYPE);
	}

	/**
	 * construct a 2 tier connection to a database
	 * this will be called only when SellWin is to
	 * be configured to operate in a 2 tier model.
	 * @exception AngError thrown when we can't get
	 * a database connection
	 * @param dbURL the JDBC URL of the database
	 * @param dbID the database ID we are to log in with
	 * @param dbPSW the database password to log in with
	 */
	public void init2Tier(String dbURL, String dbID, String dbPSW) 
		throws AngError {

        try {
			if (DB_TYPE == Prefs.ORACLE) {
				DriverManager.registerDriver(new oracle.jdbc.driver.OracleDriver());
				System.out.println("registered Oracle Driver");
				System.out.println("Oracle URL=["+dbURL+"] ID=[" + dbID +"] PSW=["+ dbPSW + "]");	
				con = DriverManager.getConnection(dbURL, dbID, dbPSW);
			}
			else
			if (DB_TYPE == Prefs.MYSQL) {
            	DriverManager.registerDriver(new org.gjt.mm.mysql.Driver());
				System.out.println("registered MySQL mm Driver");
				String mysqlURL = dbURL + "?user=" + dbID + "&password=" + dbPSW;
				System.out.println("MYSQL URL=["+mysqlURL+"]");
				con = DriverManager.getConnection(mysqlURL);
			}
			else {
				throw new AngError("Unsupported database: " + DB_TYPE);
			}
		
            con.setAutoCommit(false);

			activityDB.setConnection(con);
			addressDB.setConnection(con);
			adminDB.setConnection(con);
			attendeeDB.setConnection(con);
			campaignDB.setConnection(con);
			customerDB.setConnection(con);
			deleteInfoDB.setConnection(con);
			stateTaxDB.setConnection(con);
			contactDB.setConnection(con);
			custInventoryDB.setConnection(con);
			forecastDB.setConnection(con);
			leadDB.setConnection(con);
			oppDB.setConnection(con);
			orderDB.setConnection(con);
			productDB.setConnection(con);
			quoteDB.setConnection(con);
			quoteLineDB.setConnection(con);
			rolePermissionDB.setConnection(con);
			salesPersonDB.setConnection(con);
			salesPersonRoleDB.setConnection(con);
			userGroupDB.setConnection(con);
			ugMemberDB.setConnection(con);
			userRoleDB.setConnection(con);

        } catch (Exception e) {
            System.out.println ("\n  Error connecting to Database...\n");
			e.printStackTrace();
            throw new AngError(e.getMessage());
        }
	}


	/**
	 * update a Lead in the database 
	 * @param campPK a Campaign primary key that is the
	 * container of this Lead
	 * @param l the Lead we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateLead(long campPK, Lead l)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.updateLead");
		try {
			l.setCampaignKey(campPK);
			leadDB.updateRow(l);
			leadDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { leadDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * get the Opportunity Index array structure 
	 * @param u the SalesPerson we are searching with
	 * @return an ArrayList of OppIndex objects
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getOpportunityIndex(SalesPerson u) 
		throws RemoteException, AngError {
	
		if (Prefs.DEBUG) System.out.println("BizServices.getOpportunityIndex");
		ArrayList rows=null;
	
		try {
			if (Prefs.DEBUG) System.out.println("getOpportunityIndex ");
			//u.print();
			rows = oppDB.selectOppIndexRows(u);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}

		return rows;
	}
		
	/**
	 * get the product matrix ArrayList which is a big
	 * blob of Products that we build quotes with
	 * @return an ArrayList of Product objects
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductMatrix()
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getProductMatrix");

		ArrayList matrix=null;

		try {
			matrix = productDB.selectMatrixRows();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	
		return matrix;
	}

	/**
	 * get the list of Product groups used in the quoting
	 * process
	 * @return an ArrayList of product group objects
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductGroups()
		throws RemoteException, AngError {

		//currently not used, so not implemented here

		if (Prefs.DEBUG) System.out.println("BizServices.getProductGroups");
		return new ArrayList();
	}

	/**
	 * get the ArrayList of Product Line objects used
	 * in the quoting process
	 * @param group  a product group to search for
	 * @return the ArrayList of found Product Line objs
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductLines(String group)
		throws RemoteException, AngError {

		//currently not used, so not implemented here

		if (Prefs.DEBUG) System.out.println("BizServices.getProductLines");
		return new ArrayList();
	}

	/**
	 * get all the Products for a given product line
	 * @param group the product group to search in
	 * @param line the product line to search in
	 * @return the ArrayList of foun products
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductsForLine(String group, String line)
		throws RemoteException, AngError {
		ArrayList prods = null;

		if (Prefs.DEBUG) System.out.println("BizServices.getProductsForLine");
		try {
			prods = productDB.selectByGroupLine(group, line);
		} catch (SQLException e) {
			e.printStackTrace();
		 	throw new AngError(e.getMessage());
		}
	
		return prods;
	}

	/**
	 * get a Product object using some criteria
	 * @param group the product group to search
	 * @param line the product line to search
	 * @param name the product name to search
	 * @return the found Product
	 * @exception AngError thrown when an app error occurs
	 */
	public Product getProduct(String group, String line, String name)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getProduct");
		Product product=null;

		try {
			product = productDB.selectRow(group, line, name);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}

		return product;
	}

	/**
	 * get a Product object using it's primary key
	 * @param pk the primary key
	 * @return the found Product or null if not found
	 * @exception AngError thrown when an app error occurs
	 */
	public Product getProduct(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getProduct");
		Product product=null;

		try {
			product = (Product)productDB.selectRow(new Long(pk));
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}

		return product;
	}

	/**
	 * update a Product
	 * @param product the Product to update
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateProduct(Product product)
		throws RemoteException, AngError {

		try {
			productDB.updateRow(product);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add a Forecast to the database
	 * @param opportunityPK the Opportunity primary key of the
	 * parent to this Forecast being added
	 * @param a the Forecast we are adding
	 * @return the newly assigned Forecast primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addForecast(long opportunityPK, Forecast a)
		throws RemoteException, AngError {

		long pk=0L;
		if (Prefs.DEBUG) System.out.println("BizServices.addForecast");

		try {
			a.setOppKey(opportunityPK);
			pk = forecastDB.insertRow(a, false);
			forecastDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { forecastDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}

		return pk;
	}


	/**
	 * update the Forecast 
	 * @param oppPK the Opportunity primary key of the parent
	 * to this Forecast
	 * @param a the Forecast we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateForecast(long oppPK, Forecast a)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.updateForecast");
		try {
			a.setOppKey(oppPK);
			forecastDB.updateRow(a);
			forecastDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { forecastDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * delete a Forecast 
	 * @param opportunityPK the parent Opportunity's primary key
	 * @param forecastPK the Forecast primary key we delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteForecast(long opportunityPK, long forecastPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteForecast");
		try {
			forecastDB.deleteRow(new Long(forecastPK));
			forecastDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { forecastDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add an Order to the database
	 * @param oppPK the parent Opportunity primary key
	 * @param q the Order we are adding
	 * @return the newly assigned primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addOrder(long oppPK, Order q)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addOrder");
		long pk=0L;

		try {
			q.setOppKey(oppPK);
			pk = orderDB.insertRow(q, false);
			orderDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { orderDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
		return pk;
	}

	/**
	 * update an Order in the database
	 * @param oppPK the parent Opportunity primary key
	 * @param q the Order we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateOrder(long oppPK, Order q)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.updateOrder");
		try {
			q.setOppKey(oppPK);
			orderDB.updateRow(q);
			orderDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { orderDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add a Quote to the database 
	 * @param oppPK the parent Opportunity primary key
	 * @param q the Quote we are adding
	 * @return the newly assigned Quote primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addQuote(long oppPK, Quote q)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addQuote");
		long pk = 0L;

		try {
			q.setOppPK(oppPK);
			pk = quoteDB.insertRow(q, false);
			quoteDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { quoteDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}		

		return pk;
	}

	/**
	 * update a Quote in the database 
	 * @param oppPK the parent Opportunity primary key
	 * @param q the Quote we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateQuote(long oppPK, Quote q)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.updateQuote");
		try {
			q.setOppPK(oppPK);
			quoteDB.updateRow(q);
			quoteDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { quoteDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}		
	}

	/**
	 * delete a Quote in the database
	 * @param oppPK the parent Opportunity primary key
	 * @param quotePK the Quote primary key we delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteQuote(long oppPK, long quotePK)
		throws RemoteException, AngError {
		
		if (Prefs.DEBUG) System.out.println("BizServices.deleteQuote");
		try {
			quoteDB.deleteRow(new Long(quotePK));
			quoteDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { quoteDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}		
	}

	/**
	 *  add a QuoteLine to the database
	 * @param oppPK the parent Opportunity primary key
	 * @param quotePK the parent Quote primary key
	 * @param a the QuoteLine we are adding
	 * @return the newly assigned QuoteLine primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addQuoteLine(long oppPK, long quotePK, QuoteLine a) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addQuoteLine");
		long pk = 0L;

		try {
			a.setQuotePK(quotePK);
			pk = quoteLineDB.insertRow(a, false);
			quoteLineDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { quoteLineDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
		return pk;
	}

	/**
	 * delete a QuoteLine from the database
	 * @param oppPK the primary key of the parent Opportunity
	 * @param quotePK the primary key of the parent Quote
	 * @param linePK the primary key of the QuoteLine we are deleting
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteQuoteLine(long oppPK, long quotePK, long linePK) 
		throws RemoteException, AngError {
		if (Prefs.DEBUG) System.out.println("BizServices.deleteQuoteLine");

		try {
			quoteLineDB.deleteRow(new Long(linePK));
			quoteLineDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { quoteLineDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * get all the Alarms for a given SalesPerson 
	 * only for the current day
	 * @param salesPersonPK the primary key of the sales person
	 * we are searching for
	 * @return  an ArrayList of all Alarms that pertain
	 * to this sales person
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getAlarms(long salesPersonPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getAlarms");
		try {
			ArrayList acts =  activityDB.selectActiveAlarms(salesPersonPK);
			System.out.println("BizServices.getAlarms acts="+ acts.size());
			return acts;
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}
	
	/**
	 * add an Activity to the database
	 * @param opportunityPK the parent opportunity
	 * @return description
	 * @exception AngError thrown when an app error occurs
	 */
	public long addActivity(long opportunityPK, Activity a)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addActivity");
		long pk = 0L;

		try {
			a.setOppPK(opportunityPK);
			pk = activityDB.insertRow(a, false);
			activityDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { activityDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
		return pk;
	}

	/**
	 * update an Activity 
	 * @param oppPK the containing Opportunity primary key
	 * @param a the Activity we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateActivity(long oppPK, Activity a)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.updateActivity");
		try {
			a.setOppPK(oppPK);
			activityDB.updateRow(a);
			activityDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { activityDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * delete an Activity
	 * @param opportunityPK the containing opportunity primary key
	 * @param activityPK the primary key of the Activity we are deleting
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteActivity(long opportunityPK, long activityPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteActivity");
		try {
			activityDB.deleteRow(new Long(activityPK));
			activityDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { activityDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}


	/**
	 * get all the user roles in the system
	 * @param lastSyncDate a user's last sync date we use in the
	 * query to limit results
	 * @return an ArrayList of all the found data
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getAllUserRoles(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getAllUserRoles");
		ArrayList roles=null;
		try {
			roles = userRoleDB.selectAllRows(lastSyncDate);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return roles;
	}

	/**
	 * load a user role to the system 
	 * @param role the user role we are adding
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadUserRole(UserRole role)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.loadUserRole");
		long pk=0L;

		try {
			pk = userRoleDB.insertRow(role, true);
			userRoleDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userRoleDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}

	}


	/**
	 * add a user role to the system 
	 * @param role the user role we are adding
	 * @exception AngError thrown when an app error occurs
	 */
	public void addUserRole(UserRole role)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addUserRole");
		long pk=0L;

		try {
			pk = userRoleDB.insertRow(role, false);
			userRoleDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userRoleDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}

	}

	/**
	 * deleta all user roles
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllUserRole() 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteAllUserRole");
		try {
			userRoleDB.deleteAllRows();
			userRoleDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userRoleDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * deleta a user role
	 * @param name the name of the user role to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteUserRole(long pk) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteUserRole");
		try {
			userRoleDB.deleteRow(new Long(pk));
			userRoleDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userRoleDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}


	/**
	 * update a user role 
	 * @param role the UserRole we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateUserRole(UserRole role)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.updateUserRole");
		try {
			userRoleDB.updateRow(role);
			userRoleDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userRoleDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	
	/**
	 * get a user role 
	 * @param name the name of the User Role we are getting
	 * @return the found UserRole
	 * @exception AngError thrown when an app error occurs
	 */
	public UserRole getUserRole(String name)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getUserRole");
		UserRole ur = null;
		try {
			ur = userRoleDB.selectRowByName(name);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return ur;
	}


	/**
	 * logon to the system  
	 * @param id the logon id to try
	 * @param psw the logon psw to try
	 * @return a Login credential if successful
	 * @exception AngError thrown when an app error occurs
	 */
	public Login logon(String id, String psw) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println(id + "/" + psw + " BizServices.logon");
		try {
			SalesPerson sp = salesPersonDB.selectRow(id, psw);
			login = new Login();
			login.setSalesPerson(sp);
		} catch (SellwinNotFoundException f) {
			throw new AngError("Logon failed.  Check ID and Password.");
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return login;
	}

	/**
	 * delete a customer by pk
	 * @param cust the customer to delete 
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteCustomer(Customer cust)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteCustomer");
		try {
			customerDB.deleteRow(cust);
			customerDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { customerDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}


	/**
	 * update a customer
	 * @param c the Customer we update
	 * @exception AngError thrown when an app error occurs
	 */

	/**
	 * delete a customer
	 * @param name the customer name to delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteCustomer(String name)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteCustomer");
		try {
			customerDB.deleteByNameRow(name);
			customerDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { customerDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}


	/**
	 * update a customer
	 * @param c the Customer we update
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateCustomer(Customer c)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.udpateCustomer");
		try {
			customerDB.updateRow(c);
			customerDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { customerDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * load a customer 
	 * @param c the Customer to load
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadCustomer(Customer c)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.loadCustomer");
		try {
			long pk = customerDB.insertRow(c, true);
			customerDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { customerDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add a customer 
	 * @param c the Customer to add
	 * @return the primary key of the newly added row
	 * @exception AngError thrown when an app error occurs
	 */
	public long addCustomer(Customer c)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addCustomer");
		long pk=0L;
		try {
			pk = customerDB.insertRow(c, false);
			customerDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { customerDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
		return pk;
	}


	/**
	 * get a customer by name
	 * @param name the customer name to search with
	 * @return the found Customer
	 * @exception AngError thrown when an app error occurs
	 */
	public Customer getCustomer(String name) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getCustomer");
		Customer cust=null;
		try {
			cust = customerDB.selectByNameRow(name);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return cust;
	}


	/**
	 * get all the customer names 
	 * @return an array of customer names found
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getAllCustomerNames() 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getAllCustomerNames");
		TreeMap names = null;

		try {
			names = customerDB.selectAllNames();
			return names.values().toArray();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * get a StateTax
	 * @param stateCode the code to key with
	 * @return a StateTax if found null if not found
	 */
	public StateTax getTax(String code) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getTax");
		try {
			return (StateTax)stateTaxDB.selectRow(code);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * update a state tax row
	 * @param stateTax a StateTax to update
	 */
	public void updateTax(StateTax tax)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.updateTax");

		try {
			stateTaxDB.updateRow(tax);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * delete all StateTax rows
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllTax() 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteAllTax");

		try {
			stateTaxDB.deleteAllRows();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}


	/**
	 * load a StateTax
	 * @param row  StateTax to load
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadTax(StateTax row) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.loadTax");

		try {
			stateTaxDB.insertRow(row, true);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}


	/**
	 *  get all the State tax codes
	 * @param lastSyncDate the user's last sync date which limits
	 * the query, if null no limit
	 * @return an ArrayList of StateTax found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getStateTax(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getStateTax");
		ArrayList tax = null;

		try {
			tax = stateTaxDB.selectAllRows(lastSyncDate);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}

		return tax;
	}


	/**
	 *  get all the Customers
	 * @param lastSyncDate a user's last sync date used to limit the
	 * query or null if no limit
	 * @return an ArrayList of Customers found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCustomers(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getCustomers");
		ArrayList customers = null;

		try {
			customers = customerDB.selectAllRows(lastSyncDate);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}

		return customers;
	}

	/**
	 * load a Product 
	 * @param p the Product to add
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadProduct(Product p)
		throws RemoteException, AngError {

		long pk=0L;
		if (Prefs.DEBUG) System.out.println("BizServices.loadProduct");

		try {
			pk = productDB.insertRow(p, true);
			productDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { productDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}

	}


	/**
	 * add a Product 
	 * @param p the Product to add
	 * @exception AngError thrown when an app error occurs
	 */
	public void addProduct(Product p)
		throws RemoteException, AngError {

		long pk=0L;
		if (Prefs.DEBUG) System.out.println("BizServices.addProduct");

		try {
			pk = productDB.insertRow(p, false);
			productDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { productDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}

	}

	/**
	 * delete all the Products 
	 * @param prod Product to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteProduct(Product prod) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteProduct");
		try {
			productDB.deleteRow(new Long(prod.getPK())); 
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	
	/**
	 * get all the Products 
	 * @param a date criteria to use for the query
	 * @return an ArrayList of Products
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProducts(java.util.Date afterDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getProducts");
		ArrayList products = null;;
		try {
			products = productDB.selectAllRows(afterDate);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		
		return products;
	}


	/**
	 * get all the Opportunity names for a given user
	 * @param u the SalesPerson to search for
	 * @return an ArrayList of Opportunity Names (Strings)
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getOpportunityNames(SalesPerson u)
		throws RemoteException, AngError {
		
		if (Prefs.DEBUG) System.out.println("BizServices.getOpportunityNames");
		ArrayList names = null;
		try {
			names = oppDB.selectNames(u.getPK());
			if (Prefs.DEBUG) System.out.println("opp names has "+names.size());
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return names;
	}


	/**
	 * get all the Opportunities for a given user 
	 * @param u the SalesPerson to search for
	 * @param lastSyncDate a user's last sync date used to limit
	 * the query, null if no limit
	 * @return an ArrayList of Opportunity rows found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getOpportunities(SalesPerson u, java.util.Date lastSyncDate)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getOpportunities");
		ArrayList opps = null;
		try {
			opps = oppDB.selectByUserRows(u.getPK(), lastSyncDate);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return opps;
	}


	/**
	 * get an Opportunity
	 * @param pk search by the Opportunity's primary key
	 * @return the found Opportunity
	 * @exception AngError thrown when an app error occurs
	 */
	public Opportunity getOpportunity(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getOpportunity");
		Opportunity opp=null;

		try {
			opp = (Opportunity)(oppDB.selectRow(new Long(pk)));
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return opp;
	}


	/**
	 * delete an Opportunity
	 * @param pk the primary key of an Opportunity to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteOpportunity(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteOpportunity");
		try {
			oppDB.deleteRow(new Long(pk));
			oppDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { productDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * load an Opportunity
	 * @param o the Opportunity we are loading
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadOpportunity(Opportunity o)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.loadOpportunity");

		try {
			long pk= oppDB.insertRow(o, true);
			oppDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { oppDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add an Opportunity
	 * @param o the Opportunity we are adding
	 * @return the newly added Opportunity's primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addOpportunity(Opportunity o)
		throws RemoteException, AngError {

		long pk;
		if (Prefs.DEBUG) System.out.println("BizServices.addOpportunity");

		try {
			pk= oppDB.insertRow(o, false);
			oppDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { oppDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
		return pk;
	}


	/**
	 * update an Opportunity
	 * @param o the Opportunity to update
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateOpportunity(Opportunity o)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.updateOpportunity");
		try {
			oppDB.updateRow(o);
			oppDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { oppDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}


	/**
	 * delete a Contact 
	 * @param oppPk the containing Opportunity primary key
	 * @param contPk the Contact primary key to delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteContact(long oppPk, long contPk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteContact");
		try {
			contactDB.deleteRow(new Long(contPk));
			contactDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { contactDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}


	/**
	 * update a Contact
	 * @param oppPK the containing Opportunity primary key
	 * @param c the the Contact being added
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateContact(long oppPK, Contact c) 
		throws RemoteException, AngError {
		if (Prefs.DEBUG) System.out.println("BizServices.udpateContact");
		try {
			contactDB.updateRow(c);
			contactDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { contactDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add a new Contact to the database belonging to the 
	 * current opportunity
	 *
	 * @param oppPk the primary key of the parent opportunity
	 * @param c the Address entered for this new contact
	 * @return the primary key of the Contact object that is created
	 * @exception AngError general application error 
	 */
	public long addContact(long oppPk, Contact contact)
		throws RemoteException, AngError {
		
		if (Prefs.DEBUG) System.out.println("BizServices.addContact");
		long pk;
		try {
			contact.setOppKey(oppPk);
			pk = contactDB.insertRow(contact, false);
			contactDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { contactDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
		return pk;
	}


	/**
	 * get all SalesPersons in the system 
	 * @param lastSyncDate a user's last sync date or null, used to
	 * limit the query
	 * @return an array of SalesPerson objects
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getSalesPersons(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getSalesPersons");
		TreeMap users = null;

		try {
			users = salesPersonDB.selectAllRows(lastSyncDate);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return users.values().toArray();
	}

	/**
	 * get a SalesPerson using a formatted name
	 * @param pk primary key to search with
	 * @return the found SalesPerson
	 * @exception AngError thrown when an app error occurs
	 */
	public SalesPerson getSalesPerson(long pk) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getSalesPerson");
		SalesPerson sp=null;
		try {
			sp = (SalesPerson)salesPersonDB.selectRow(new Long(pk));
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return sp;
	}


	/**
	 * update a SalesPerson
	 * @param s the SalesPerson to update with
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateSalesPerson(SalesPerson s)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.updateSalesPerson");
		try {
			salesPersonDB.updateRow(s);
			salesPersonDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { salesPersonDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * delete a user from the system 
	 * @param pk the SalesPerson primary key to delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteSalesPerson(SalesPerson per) 
		throws RemoteException, AngError {
		if (Prefs.DEBUG) System.out.println("BizServices.deleteSalesPerson");

		try {
			salesPersonDB.deleteRow(per);
			salesPersonDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { salesPersonDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * load a user to the system
	 * @param s the SalesPerson to add
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadSalesPerson(SalesPerson s) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.loadSalesPerson");
		long pk;
		try {
			pk= salesPersonDB.insertRow(s, true);
			salesPersonDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { salesPersonDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add a user to the system
	 * @param s the SalesPerson to add
	 * @return the newly added row's primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addSalesPerson(SalesPerson s) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addSalesPerson");
		long pk;
		try {
			pk = salesPersonDB.insertRow(s, false);
			salesPersonDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { salesPersonDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
		return pk;
	}


	/**
	 * get all sales person id's
	 * @return an ArrayList of user IDs
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getSalesPersonIDs() 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getSalesPersonIDs");
		ArrayList idList = null;

		try {
			idList = salesPersonDB.selectAllIDs();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return idList;

	}

	/**
	 * get all the users names in 'formatted' form
	 * @return an ArrayList of formatted user names
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getSalesPersonNames() 
		throws RemoteException, AngError {
		
		if (Prefs.DEBUG) System.out.println("BizServices.getSalesPersonNames");
		ArrayList names=null;
		try {
			names = salesPersonDB.selectAllNames();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return names;
	}


	/**
	 * add a user to a UserGroup
	 * @param userPK the primary key of the user being added
	 * @param group the UserGroup begin added to
	 * @exception AngError thrown when an app error occurs
	 */
	public void addUserToGroup(long userPK, UserGroup group) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addUserToGroup");
		try {
			UserGroupMember ugm = new UserGroupMember();
			ugm.setUserGroupPK(group.getPK());
			ugm.setModifiedBy(group.getModifiedBy());
			ugm.setUserPK(userPK);
			long pk = ugMemberDB.insertRow(ugm, false);
			ugMemberDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { ugMemberDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}


	/**
	 *  get all the SalesPersons in a UserGroup
	 * @param groupName the UserGroup we are searching with
	 * @return an array of SalesPersons found
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getUsersInGroup(String groupName) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getUsersInGroup");
		try {
			TreeMap members = userGroupDB.selectUsersInGroup(groupName);
			return members.values().toArray();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * get a UserGroup in the system by primary key
	 * @param pk a UserGroup primary key to search with
	 * @return a UserGroup that was found or null if not found
	 * @exception AngError thrown when an app error occurs
	 */
	public UserGroup getUserGroup(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getUserGroup");
		try {
			UserGroup ug = (UserGroup)userGroupDB.selectRow(new Long(pk));
			return ug;
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * get all UserGroups in the system
	 * @param lastSyncDate a user's last sync date which limits the query
	 * @return an array of UserGroups that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getUserGroups(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getUserGroups");
		try {
			TreeMap userGroups = userGroupDB.selectAllRows(lastSyncDate);
			return userGroups.values().toArray();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * load a UserGroup to the system
	 * @param g a UserGroup we are adding to the system
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadUserGroup(UserGroup g) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addUserGroup");
		try {
			long pk = userGroupDB.insertRow(g, true);
			userGroupDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userGroupDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add a UserGroup to the system
	 * @param g a UserGroup we are adding to the system
	 * @exception AngError thrown when an app error occurs
	 */
	public void addUserGroup(UserGroup g) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addUserGroup");
		try {
			long pk = userGroupDB.insertRow(g, false);
			userGroupDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userGroupDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * delete all UserGroups
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllUserGroup() 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteAllUserGroup");
		try {
			userGroupDB.deleteAllRows();
			userGroupDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userGroupDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 *  delete a UserGroup
	 * @param groupName the UserGroup name we delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteUserGroup(String groupName) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteUserGroup");
		try {
			userGroupDB.deleteByNameRow(groupName);
			userGroupDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userGroupDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}


	/**
	 * delete a user in a UserGroup
	 * @param g the containing UserGroup 
	 * @param userPK the user's primary key we are deleting with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteUserInGroup(UserGroup g, long userPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteUserInGroup");
		try {
			ArrayList users = ugMemberDB.selectUsersInGroup(g.getPK());
			SalesPerson sp;
			for (int i=0;i<users.size();i++) {
				sp = (SalesPerson)users.get(i);
				if (sp.getPK() == userPK) {
					ugMemberDB.deleteByUserRow(sp.getPK());
					break;
				}
			}
			ugMemberDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { ugMemberDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}


	/**
	 * get the UserGroups for a given user
	 * @param userPK the SalesPerson primary key we are searching for
	 * @return an ArrayList of UserGroup(s) this user belongs to 
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getGroupsForUser(long userPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getGroupsForUser");
		ArrayList userGroups = null;

		try {
			userGroups = ugMemberDB.selectGroupsForUser(userPK);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return userGroups;
	}

	/**
	 * delete the Campaigns 
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllCampaign()
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteAllCampaign");

		try {
			campaignDB.deleteAllRows();
			campaignDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * load a campaign into the campaigns table
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadCampaign(Campaign c)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.loadCampaign");

		try {
			long pk = campaignDB.insertRow(c, true);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * get a Campaign
	 * @param pk a primary key
	 * @return a Campaign if found null if not found
	 * @exception AngError thrown when an app error occurs
	 */
	public Campaign getCampaign(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getCampaign");

		try {
			return (Campaign)campaignDB.selectRow(new Long(pk));
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * get the Campaigns 
	 * @param lastSyncDate a user's last sync date which limits the
	 * query or null if no limit 
	 * @return an ArrayList of Campaigns that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCampaigns(java.util.Date lastSyncDate)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getCampaigns");
		ArrayList campaigns = null;

		try {
			campaigns = campaignDB.selectAllRows(lastSyncDate);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return campaigns;
	}


	/**
	 * get the Leads for a given Campaign 
	 * @param campaignPK the containing Campaign's primary key
	 * @return an ArrayList of Leads that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCampaignLeads(long campaignPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getCampaignLeads");
		ArrayList leads = null;

		try {
			leads = leadDB.selectByCampaignRows(campaignPK);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return leads;
	}

	/**
	 * get all Leads 
	 * @param lastSyncDate a user's last sync date used to limit the
	 * query or null if no limit
	 * @return an ArrayList of Leads that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCampaignLeads(java.util.Date lastSyncDate)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getCampaignLeads all");
		ArrayList leads = null;

		try {
			leads = leadDB.selectAllRows(lastSyncDate);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return leads;
	}

	/**
	 * get a Lead
	 * @param pk the Lead primary key
	 * @return the Lead that was found or null if not found
	 */
	public Lead getLead(long pk)
		throws RemoteException, AngError {
	
		if (Prefs.DEBUG) System.out.println("BizServices.getLead");
		try {
			return (Lead)leadDB.selectRow(new Long(pk));
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}
	
	/**
	 * process the items that were deleted by a user when
	 * operating in a 'disconnected' mode, each object
	 * in the passed list will be deleted from the database
	 *
	 * @param deletes the ArrayList of DeleteInfo objects
	 * @exception AngError thrown when an app error occurs
	 */
	public void uploadDeletes(ArrayList deletes) 
		throws RemoteException, AngError {
	
		if (Prefs.DEBUG) System.out.println("uploadDeletes is working on "+ deletes.size());

		DeleteInfo del;
		Connection currentConnection=null;

		//get the Connection from any one of the database
		//routines, they all share the same connection so 
		//the ActivityDB will do just fine 

		currentConnection = activityDB.getConnection();

		try {
			for (int i=0;i<deletes.size();i++) {
				del = (DeleteInfo)deletes.get(i);
				if (del.className.equals("Activity")) {
					activityDB.deleteRow(new Long(del.pk));
				} else
				if (del.className.equals("Forecast")) {
					forecastDB.deleteRow(new Long(del.pk));
				} else
				if (del.className.equals("Quote")) {
					quoteDB.deleteRow(new Long(del.pk));
				} else
				if (del.className.equals("QuoteLine")) {
					quoteLineDB.deleteRow(new Long(del.pk));
				} else
				if (del.className.equals("Contact")) {
					contactDB.deleteRow(new Long(del.pk));
				} else
				if (del.className.equals("Opportunity")) {
					oppDB.deleteRow(new Long(del.pk));
				} else
					throw new AngError(del.className + " not implemented yet");

				currentConnection.commit();

			}
		} catch (SQLException e) {
			e.printStackTrace();
			try { currentConnection.rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add a customer inventory
	 * @param ci the CustomerInventory to add
	 * @return the primary key of the newly added row
	 * @exception AngError thrown when an app error occurs
	 */
	public long addCustomerInventory(CustomerInventory c)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.addCustomerInventory");
		long pk=0L;
		try {
			pk = custInventoryDB.insertRow(c, false);
			custInventoryDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { custInventoryDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
		return pk;
	}

	/**
	 * delete a customer inventory
	 * @param ci the customer inventory to delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteCustomerInventory(CustomerInventory ci)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteCustomerInventory");
		try {
			custInventoryDB.deleteRow(new Long(ci.getPK()));
			custInventoryDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { custInventoryDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * get the CustomerInventory for a given customer
	 * @param cust the Customer we are searching with
	 * @return an ArrayList of CustomerInventory objects
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCustomerInventory(long custPK) 
		throws RemoteException, AngError {
	
		if (Prefs.DEBUG) System.out.println("BizServices.getCustomerInventory");
		ArrayList rows=null;
	
		try {
			rows = custInventoryDB.selectAllRows(custPK);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
		return rows;
	}

	/**
	 * get the list of deleted rows
	 * @return the list of deleted data
	 */
	public ArrayList getDeletes() 
		throws RemoteException, AngError {
		if (Prefs.DEBUG) System.out.println("BizServices.getDeletes");
	
		try {
			return deleteInfoDB.selectAllRows();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
 	 * reset the deletes
	 */
	public void resetDeletes() 
		throws RemoteException, AngError {
		if (Prefs.DEBUG) System.out.println("BizServices.resetDeletes");
	
		try {
			deleteInfoDB.deleteAllRows();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * write out some delete info for the sync processing
	 * @param objType class name of object that got deleted
	 * @param pk primary key of object that got deleted
	 * @exception AngError thrown when an app error occurs
	 */
	public final void writeDelete(DeleteInfo deleteInfo)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.writeDelete");
	
		try {
			long pk = deleteInfoDB.insertRow(deleteInfo, false);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	public void init(int DB_TYPE) {
		activityDB = new ActivityDB(DB_TYPE);
		addressDB = new AddressDB(DB_TYPE);
		adminDB = new Admin(DB_TYPE);
		attendeeDB = new AttendeeDB(DB_TYPE);
		campaignDB = new CampaignDB(DB_TYPE);
		customerDB = new CustomerDB(DB_TYPE);
		deleteInfoDB = new DeleteInfoDB(DB_TYPE);
		stateTaxDB = new StateTaxDB(DB_TYPE);
		contactDB = new ContactDB(DB_TYPE);
		custInventoryDB = new CustomerInventoryDB(DB_TYPE);
		forecastDB = new ForecastDB(DB_TYPE);
		leadDB = new LeadDB(DB_TYPE);
		oppDB = new OpportunityDB(DB_TYPE);
		orderDB = new OrderDB(DB_TYPE);
		productDB = new ProductDB(DB_TYPE);
		quoteDB = new QuoteDB(DB_TYPE);
		quoteLineDB = new QuoteLineDB(DB_TYPE);
		rolePermissionDB = new RolePermissionDB(DB_TYPE);
		salesPersonDB = new SalesPersonDB(DB_TYPE);
		salesPersonRoleDB = new SalesPersonRoleDB(DB_TYPE);
		userGroupDB = new UserGroupDB(DB_TYPE);
		ugMemberDB = new UserGroupMemberDB(DB_TYPE);
		userRoleDB = new UserRoleDB(DB_TYPE);
	}

	/**
	 * delete all SalesPersons in the system 
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllSalesPerson() 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.deleteAllSalesPerson");

		try {
			salesPersonDB.deleteAllRows();
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	public void truncateDB()
		throws RemoteException, AngError {

		try {
			//truncate in same order as 'drop' script

			deleteInfoDB.truncate();
			deleteInfoDB.getConnection().commit();

			stateTaxDB.truncate();
			stateTaxDB.getConnection().commit();

			deleteInfoDB.truncate();
			deleteInfoDB.getConnection().commit();

			rolePermissionDB.truncate();
			rolePermissionDB.getConnection().commit();

			userRoleDB.truncate();
			userRoleDB.getConnection().commit();

			ugMemberDB.truncate();
			ugMemberDB.getConnection().commit();

			userGroupDB.truncate();
			userGroupDB.getConnection().commit();

			orderDB.truncate();
			orderDB.getConnection().commit();

			quoteLineDB.truncate();
			quoteLineDB.getConnection().commit();

			quoteDB.truncate();
			quoteDB.getConnection().commit();

			attendeeDB.truncate();
			attendeeDB.getConnection().commit();

			activityDB.truncate();
			activityDB.getConnection().commit();

			leadDB.truncate();
			leadDB.getConnection().commit();

			campaignDB.truncate();
			campaignDB.getConnection().commit();

			custInventoryDB.truncate();
			custInventoryDB.getConnection().commit();

			productDB.truncate();
			productDB.getConnection().commit();

			customerDB.truncate();
			customerDB.getConnection().commit();

			salesPersonDB.truncate();
			salesPersonDB.getConnection().commit();

			salesPersonRoleDB.truncate();
			salesPersonRoleDB.getConnection().commit();

			forecastDB.truncate();
			forecastDB.getConnection().commit();

			contactDB.truncate();
			contactDB.getConnection().commit();

			oppDB.truncate();
			oppDB.getConnection().commit();

			addressDB.truncate();
			addressDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * get all the UserGroupMembers
	 * @param lastSyncDate a user's last sync date or null, used to
	 * limit the query
	 * @return an ArrayList of UserGroupMember objects
	 */
	public ArrayList getGroupMembers(java.util.Date lastSyncDate)
		throws RemoteException, AngError {
		
		try {
			return ugMemberDB.selectAllRows(lastSyncDate);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * select a user group member 
	 * @param pk the UserGroupMember primary key 
	 * @return the UserGroupMember or null if not found
	 * @exception AngError thrown when an app error occurs
	 */
	public UserGroupMember getGroupMember(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.getGroupMember");

		try {
			UserGroupMember ugm = (UserGroupMember)ugMemberDB.selectRow(new Long(pk));
			return ugm;
		} catch (SQLException e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * load a user group member to the system 
	 * @param ugm the UserGroupMember we are loading
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadGroupMember(UserGroupMember ugm)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.loadGroupMember");
		long pk=0L;

		try {
			pk = ugMemberDB.insertRow(ugm, true);
			ugMemberDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userRoleDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * load a lead 
	 * @param lead the Lead we are loading
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadLead(Lead lead)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("BizServices.loadLead");
		long pk=0L;

		try {
			pk = leadDB.insertRow(lead, true);
			leadDB.getConnection().commit();
		} catch (SQLException e) {
			e.printStackTrace();
			try { userRoleDB.getConnection().rollback(); } catch (SQLException x) {}
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * test a local table to see if it exists, if not
	 * then return false, this test is used by the GUI client
	 * to see if it needs to build the local database
	 * @return true if the local table exists, false if not
	 */
	public final boolean testTable()
		throws RemoteException, AngError {
		
		return adminDB.testTable();
	}

	/**
	 * test a local table to see if data exists in it
	 * @return the row count is returned
	 */
	public final int testTableData()
		throws RemoteException, AngError {
		
		return adminDB.testTableData();
	}

	/**
	 * create local database
	 * drop the tables first then create them
	 */
	public final void createAllTables() 
		throws RemoteException, AngError {

		try {
			adminDB.dropAllTables();
			adminDB.createAllTables();
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * load some test data for trial users
	 */
	public final void loadTestData()
		throws RemoteException, AngError {
		try {
			adminDB.loadTestData();
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	public void remove() {}
	public boolean isIdentical(EJBObject x) { return true; }
	public Object getPrimaryKey() { return null; }
	public EJBHome getEJBHome() { return null; }
	public Handle getHandle() { return null; }
}
