package sellwin.server;

import java.io.*;
import java.net.*;
import java.util.*;
import sellwin.domain.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * this class handles chat client connections, basically
 * implements the simple chat protocol (LOGON, QUERY, SEND)
 */
public class ChatHandler extends Thread {

	private static Vector handlers = new Vector( 10 );
	private Socket socket;
	private ObjectInputStream ois=null;
	private ObjectOutputStream oos=null;
	private PrintWriter out;
	private TreeMap onlineUsers = null;

	/**
	 * construct a handler for a client's socket
	 * @param socket the client's socket we read-write from-to	
	 * @param onlineUsers the global list of online users 
	 */
    public ChatHandler(Socket socket, TreeMap onlineUsers) throws IOException {

		this.onlineUsers = onlineUsers;
        this.socket = socket;
		ois = new ObjectInputStream(socket.getInputStream());
		oos = new ObjectOutputStream(socket.getOutputStream());
    }

	/**
	 * this handler is a Thread, so this is the Thread's run() method
	 * that gets executed upon a start()
	 * handles the chat packets read from the various chat clients
	 *
	 */
    public void run() {

        String line;

		synchronized(handlers) {
	    	handlers.addElement(this);
		}

		try {
	    	while(true) {

	    		ChatPacket msg = (ChatPacket)ois.readObject();

				switch 	(msg.getAction()) {

				case ChatPacket.QUERY_USERS:
					queryRequest(msg);
					break;

				case ChatPacket.LOGON:
					logonRequest( oos, msg);
					break;
	
				case ChatPacket.SEND_MSG:
					userCheck(msg);
					msg.print();
					sendReply(msg);
					break;
				default:
					System.out.println("outrageous error here");
					System.exit(-1);
				}

				for (int i = 0; i < handlers.size(); i++) {	
					synchronized(handlers) {
		            	ChatHandler handler =
			        	(ChatHandler)handlers.elementAt(i);
					}
				}
	    	}
		} catch(ClassNotFoundException cl) {
	    	cl.printStackTrace();
		} catch(IOException ioe) {
	    	ioe.printStackTrace();
		} finally {
	    	try {
				ois.close();
				oos.close();
				socket.close();
	    	} catch(IOException ioe) {
				ioe.printStackTrace();
	    	} finally {
				synchronized(handlers) {
		    		handlers.removeElement(this);
				}
	    	}
		}
    }

	/**
	 * check to see if the chat client user is in the online user
	 * list, if not, add them
	 * @param msg the online Chat user's message just received
	 */
	private void userCheck(ChatPacket msg) {

		if (onlineUsers.containsKey(msg.getSender()))
			System.out.println(msg.getSender() + " already in online list");
		else {
			ChatConnInfo info = new ChatConnInfo(msg.getSender(), oos, new java.util.Date());
			System.out.println("adding " + msg.getSender() + " at " + info.getConnectedSince());
			onlineUsers.put(msg.getSender(), info);
		}
	}

	/**
 	 * handle the chat client's logon request
	 * @param oos the client's output stream to reply to
	 * @param msg the logon request packet received
	 */
	private void logonRequest(ObjectOutputStream oos, ChatPacket msg) {
		System.out.println("logon request...");
		ChatConnInfo info = new ChatConnInfo(msg.getSender(), oos, new java.util.Date()); 
		onlineUsers.put(msg.getSender(), info);
	}

	/**
	 * send a reply to the clients specified in the SEND request
	 * @param msg the client's packet we received
	 */
	private void sendReply(ChatPacket msg) {

		ArrayList toUsers = msg.getToList();
		String toUser=null;
		ObjectOutputStream userOOS = null;

		for (int i=0; i<toUsers.size(); i++) {
			toUser = (String)toUsers.get(i);	
			System.out.println("sending reply to " + toUser);
			userOOS = lookupUserOutputStream(toUser);

			try {
				if (userOOS != null) {
					System.out.println("actually wrote back a reply");
					userOOS.writeObject(msg);
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * find a recipient's output stream so we can send them a
	 * message
	 * @param toUser the key or user ID we index into the global
	 * list of online users
	 * @return the user's output stream
	 */
	private ObjectOutputStream  lookupUserOutputStream(String toUser) {

    	ChatConnInfo connInfo = (ChatConnInfo)onlineUsers.get(toUser);
        if ((connInfo != null) && (connInfo.stillConnected())) 
			return connInfo.getOOS();
		else
			return null;
	}

	/**
	 * handle a user's query request for the currently logged on
	 * list of users
	 * @param msg the user's query request message
	 */
	private void queryRequest(ChatPacket msg) {
		System.out.println("query request...");

		//update the user's timestamp to mark them as 'online'..accomplishes
		//the heartbeat  of the protocol

		ChatConnInfo info = new ChatConnInfo(msg.getSender(), oos, new java.util.Date());
		System.out.println("msg.getSender = " + msg.getSender());
		System.out.println("info.getSender=" + info.getSender());
		onlineUsers.put(msg.getSender(), info);

		//get the list of online users

		Set keys = onlineUsers.keySet();
		Object[] keysArray = keys.toArray();
		ArrayList tempList = new ArrayList();
		for (int i=0;i<onlineUsers.size();i++) {
			info = (ChatConnInfo)onlineUsers.get(keysArray[i]);
			System.out.println((String)keysArray[i] + "-" + info.getSender());
			tempList.add(keysArray[i]);
		}

		//send back the ArrayList of online users

		msg = new ChatPacket();
		msg.setSender("server");
		msg.setOnlineUsers(tempList);
		msg.setAction(ChatPacket.QUERY_USERS);
		try {
			oos.writeObject(msg);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
}
