package sellwin.server;

import sellwin.domain.*;
import sellwin.db.*;
import sellwin.utils.*;

import java.rmi.*;
import java.rmi.server.*;
import java.sql.*;
import java.util.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class is an RMI wrapper around the business services offered
 * by the BizServices class.  Using RMI requires this wrapper technique
 * since RMI servers have to extend the UnicastRemoteObject RMI class.;
 */
public class RMIBizServices extends UnicastRemoteObject 
	implements ServerInterface {

	//some security state info for a given  user session is
	//stored in 'login'
	private Login login;

	private BizServices biz;

	public RMIBizServices(String serverName) 
		throws RemoteException {

		super();	
		try {
			Naming.rebind(serverName, this);

			String dbType = Prefs.getDatabaseType();
			if ((dbType == null) || (!dbType.equals("ORACLE") && !dbType.equals("MYSQL")))
				raiseError("Please specify DATABASE_HOST property.  ORACLE or MYSQL");

			if (dbType.equals("ORACLE"))
				biz = new BizServices(Prefs.ORACLE);
			else
			if (dbType.equals("MYSQL"))
				biz = new BizServices(Prefs.MYSQL);

			String dbHost = Prefs.getDatabaseHost();
			if (dbHost == null) 
				raiseError("Please specify DATABASE_HOST property");

			String dbPort = Prefs.getDatabasePort();
			if (dbPort == null)
				raiseError("Please specify DATABASE_PORT property");

			String dbName = Prefs.getDatabaseName();
			if (dbName == null)
				raiseError("Please specify DATABASE_NAME property");


			String dbID = Prefs.getDatabaseID();
			if (dbID == null)
				raiseError("Please specify DATABASE_ID property");

			String dbPassword = Prefs.getDatabasePassword();
			if (dbPassword == null)
				raiseError("Please specify DATABASE_PASSWORD property");

			String jdbcurl;

			if (dbType.equals("ORACLE")) {
				jdbcurl = "jdbc:oracle:thin:@" + dbHost + ":" + dbPort + ":" + dbName;
				System.out.println("JDBC URL: " + jdbcurl); 
				biz.init2Tier(jdbcurl, dbID, dbPassword);
			} else {
				jdbcurl ="jdbc:mysql://"+ dbHost + "/" + dbName;
				System.out.println("JDBC URL: " + jdbcurl); 
				biz.init2Tier(jdbcurl, dbID, dbPassword);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * update a Lead in the database 
	 * @param campPK a Campaign primary key that is the
	 * container of this Lead
	 * @param l the Lead we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateLead(long campPK, Lead l)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.updateLead");

		biz.updateLead(campPK, l);

	}

	/**
	 * get the Opportunity Index array structure 
	 * @param u the SalesPerson we are searching with
	 * @return an ArrayList of OppIndex objects
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getOpportunityIndex(SalesPerson u) 
		throws RemoteException, AngError {
	
		if (Prefs.DEBUG) System.out.println("RMIBizServices.getOpportunityIndex");

		return biz.getOpportunityIndex(u);
	}
		
	/**
	 * get the product matrix ArrayList which is a big
	 * blob of Products that we build quotes with
	 * @return an ArrayList of Product objects
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductMatrix()
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getProductMatrix");

		return biz.getProductMatrix();
	}

	/**
	 * get a product by its primary key
	 * @param pk the primary key
	 * @return the Product if found or null if not found
	 */
	public Product getProduct(long pk) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getProduct");
		return biz.getProduct(pk);
	}

	/**
	 * update a product
	 * @param product the Product to update
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateProduct(Product product)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.updateProduct");
		biz.updateProduct(product);
	}
	
	/**
	 * get the list of Product groups used in the quoting
	 * process
	 * @return an ArrayList of product group objects
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductGroups()
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getProductGroups");
		return biz.getProductGroups();
	}

	/**
	 * get the ArrayList of Product Line objects used
	 * in the quoting process
	 * @param group  a product group to search for
	 * @return the ArrayList of found Product Line objs
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductLines(String group)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getProductLines");
		return biz.getProductLines(group);
	}

	/**
	 * get all the Products for a given product line
	 * @param group the product group to search in
	 * @param line the product line to search in
	 * @return the ArrayList of foun products
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductsForLine(String group, String line)
		throws RemoteException, AngError {
		ArrayList prods = null;

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getProductsForLine");
		return biz.getProductsForLine(group, line);
	}

	/**
	 * get a Product object using some criteria
	 * @param group the product group to search
	 * @param line the product line to search
	 * @param name the product name to search
	 * @return the found Product
	 * @exception AngError thrown when an app error occurs
	 */
	public Product getProduct(String group, String line, String name)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getProduct");

		return biz.getProduct(group, line, name);
	}

	/**
	 * add a Forecast to the database
	 * @param opportunityPK the Opportunity primary key of the
	 * parent to this Forecast being added
	 * @param a the Forecast we are adding
	 * @return the newly assigned Forecast primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addForecast(long opportunityPK, Forecast a)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addForecast");
		return biz.addForecast(opportunityPK, a);
	}


	/**
	 * update the Forecast 
	 * @param oppPK the Opportunity primary key of the parent
	 * to this Forecast
	 * @param a the Forecast we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateForecast(long oppPK, Forecast a)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.updateForecast");
		biz.updateForecast(oppPK, a);
	}

	/**
	 * delete a Forecast 
	 * @param opportunityPK the parent Opportunity's primary key
	 * @param forecastPK the Forecast primary key we delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteForecast(long opportunityPK, long forecastPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteForecast");
		biz.deleteForecast(opportunityPK, forecastPK);
	}

	/**
	 * add an Order to the database
	 * @param oppPK the parent Opportunity primary key
	 * @param q the Order we are adding
	 * @return the newly assigned primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addOrder(long oppPK, Order q)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addOrder");
		return biz.addOrder(oppPK, q);
	}

	/**
	 * update an Order in the database
	 * @param oppPK the parent Opportunity primary key
	 * @param q the Order we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateOrder(long oppPK, Order q)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.updateOrder");
		biz.updateOrder(oppPK, q);
	}

	/**
	 * add a Quote to the database 
	 * @param oppPK the parent Opportunity primary key
	 * @param q the Quote we are adding
	 * @return the newly assigned Quote primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addQuote(long oppPK, Quote q)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addQuote");
		return biz.addQuote(oppPK, q);
	}

	/**
	 * update a Quote in the database 
	 * @param oppPK the parent Opportunity primary key
	 * @param q the Quote we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateQuote(long oppPK, Quote q)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.updateQuote");
		biz.updateQuote(oppPK, q);
	}

	/**
	 * delete a Quote in the database
	 * @param oppPK the parent Opportunity primary key
	 * @param quotePK the Quote primary key we delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteQuote(long oppPK, long quotePK)
		throws RemoteException, AngError {
		
		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteQuote");
		biz.deleteQuote(oppPK, quotePK);
	}

	/**
	 * add a QuoteLine to the database
	 * @param oppPK the parent Opportunity primary key
	 * @param quotePK the parent Quote primary key
	 * @param a the QuoteLine we are adding
	 * @return the newly assigned QuoteLine primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addQuoteLine(long oppPK, long quotePK, QuoteLine a) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addQuoteLine");
		return biz.addQuoteLine(oppPK, quotePK, a);
	}

	/**
	 * delete a QuoteLine from the database
	 * @param oppPK the primary key of the parent Opportunity
	 * @param quotePK the primary key of the parent Quote
	 * @param linePK the primary key of the QuoteLine we are deleting
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteQuoteLine(long oppPK, long quotePK, long linePK) 
		throws RemoteException, AngError {
		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteQuoteLine");
		biz.deleteQuoteLine(oppPK, quotePK, linePK);
	}

	/**
	 * get all the Alarms for a given SalesPerson 
	 * only for the current day
	 * @param salesPersonPK the primary key of the sales person
	 * we are searching for
	 * @return  an ArrayList of all Alarms that pertain
	 * to this sales person
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getAlarms(long salesPersonPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getAlarms");
		ArrayList acts =  biz.getAlarms(salesPersonPK);
		System.out.println("RMIBizServices.getAlarms acts="+ acts.size());
		return acts;
	}
	
	/**
	 * add an Activity to the database
	 * @param opportunityPK the parent opportunity
	 * @return description
	 * @exception AngError thrown when an app error occurs
	 */
	public long addActivity(long opportunityPK, Activity a)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addActivity");
		return biz.addActivity(opportunityPK, a);
	}

	/**
	 * update an Activity 
	 * @param oppPK the containing Opportunity primary key
	 * @param a the Activity we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateActivity(long oppPK, Activity a)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.updateActivity");
		biz.updateActivity(oppPK, a);
	}

	/**
	 * delete an Activity
	 * @param opportunityPK the containing opportunity primary key
	 * @param activityPK the primary key of the Activity we are deleting
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteActivity(long opportunityPK, long activityPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteActivity");
		biz.deleteActivity(opportunityPK, activityPK);
	}


	/**
	 * get all the user roles in the system
	 * @param lastSyncDate a user's last sync date we use to limit
	 * the query
	 * @return an ArrayList of all the found data
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getAllUserRoles(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getAllUserRoles");
		return biz.getAllUserRoles(lastSyncDate);
	}

	/**
	 * load a user role to the system 
	 * @param role the user role we are adding
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadUserRole(UserRole role)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.loadUserRole");
		biz.loadUserRole(role);
	}



	/**
	 * add a user role to the system 
	 * @param role the user role we are adding
	 * @exception AngError thrown when an app error occurs
	 */
	public void addUserRole(UserRole role)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addUserRole");
		biz.addUserRole(role);
	}


	/**
	 * deleta a user role
	 * @param pk the pk of the user role to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteUserRole(long pk) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteUserRole");
		biz.deleteUserRole(pk);
	}


	/**
	 * update a user role 
	 * @param role the UserRole we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateUserRole(UserRole role)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.updateUserRole");
		biz.updateUserRole(role);
	}

	
	/**
	 * get a user role 
	 * @param name the name of the User Role we are getting
	 * @return the found UserRole
	 * @exception AngError thrown when an app error occurs
	 */
	public UserRole getUserRole(String name)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getUserRole");
		return biz.getUserRole(name);
	}


	/**
	 * logon to the system  
	 * @param id the logon id to try
	 * @param psw the logon psw to try
	 * @return a Login credential if successful
	 * @exception AngError thrown when an app error occurs
	 */
	public Login logon(String id, String psw) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println(id + "/" + psw + " BizServices.logon");
		System.out.println("before logon call");
		return biz.logon(id, psw);
	}

	/**
	 * delete a customer
	 * @param cust the customer to delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteCustomer(Customer cust)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteCustomer");
		biz.deleteCustomer(cust);
	}

	/**
	 * delete a customer
	 * @param name the customer name to delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteCustomer(String name)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteCustomer");
		biz.deleteCustomer(name);
	}


	/**
	 * update a customer
	 * @param c the Customer we update
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateCustomer(Customer c)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.udpateCustomer");
		biz.updateCustomer(c);
	}

	/**
	 * load a customer 
	 * @param c the Customer to load
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadCustomer(Customer c)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.loadCustomer");
		biz.loadCustomer(c);
	}

	/**
	 * add a customer 
	 * @param c the Customer to add
	 * @return the primary key of the newly added row
	 * @exception AngError thrown when an app error occurs
	 */
	public long addCustomer(Customer c)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addCustomer");
		return biz.addCustomer(c);
	}


	/**
	 * get a customer by name
	 * @param name the customer name to search with
	 * @return the found Customer
	 * @exception AngError thrown when an app error occurs
	 */
	public Customer getCustomer(String name) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getCustomer");
		return biz.getCustomer(name);
	}


	/**
	 * get all the customer names 
	 * @return an array of customer names found
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getAllCustomerNames() 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getAllCustomerNames");
		return biz.getAllCustomerNames();
	}

	/**
	 * get a StateTax
	 * @param code the code to key with	
	 * @return a StateTax if found, null if not found
	 * @exception AngError thrown when an app error occurs
	 */
	public StateTax getTax(String code)
		throws RemoteException, AngError {
	
		if (Prefs.DEBUG) System.out.println("RMIBizServices.getTax");
		return biz.getTax(code);
	}

	/**
	 * update a state tax
	 * @param tax the StateTax to update
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateTax(StateTax tax)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.updateTax");
		biz.updateTax(tax);
	}

	/**
	 * delete all StateTax rows
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllTax()
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteAllTax");
		biz.deleteAllTax();
	}

	/**
	 * load a StateTax row
	 * @param row the StateTax row to load
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadTax(StateTax row)
		throws RemoteException, AngError {
		if (Prefs.DEBUG) System.out.println("RMIBizServices.getStateTax");
		biz.loadTax(row);
	}

	/**
	 *  get all the State tax codes
	 * @param lastSyncDate a user's last sync date which limits the
	 * query or null if no limit
	 * @return an ArrayList of StateTax found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getStateTax(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getStateTax");
		return biz.getStateTax(lastSyncDate);
	}


	/**
	 * get all the Customers
	 * @param lastSyncDate a user's last sync date used to limit the
	 * query or null if no limit
	 * @return an ArrayList of Customers found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCustomers(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getCustomers");
		return biz.getCustomers(lastSyncDate);
	}

	/**
	 * load a Product 
	 * @param p the Product to add
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadProduct(Product p)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.loadProduct");
		biz.loadProduct(p);
	}

	/**
	 * add a Product 
	 * @param p the Product to add
	 * @exception AngError thrown when an app error occurs
	 */
	public void addProduct(Product p)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addProduct");
		biz.addProduct(p);
	}

	/**
	 * delete all the Products
	 * @param prod Product to delete 
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteProduct(Product prod) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteAllProduct");
		biz.deleteProduct(prod);
	}

	
	/**
	 * get all the Products 
	 * @param a date to compare rows with in the query
	 * @return an ArrayList of Products
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProducts(java.util.Date afterDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getProducts");
		return biz.getProducts(afterDate);
	}


	/**
	 * get all the Opportunity names for a given user
	 * @param u the SalesPerson to search for
	 * @return an ArrayList of Opportunity Names (Strings)
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getOpportunityNames(SalesPerson u)
		throws RemoteException, AngError {
		
		if (Prefs.DEBUG) System.out.println("RMIBizServices.getOpportunityNames");
		return biz.getOpportunityNames(u);
	}


	/**
	 * get all the Opportunities for a given user 
	 * @param u the SalesPerson to search for
	 * @param lastSyncDate a user's last sync date to limit the query
	 * with or null if no limit
	 * @return an ArrayList of Opportunity rows found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getOpportunities(SalesPerson u, java.util.Date lastSyncDate)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getOpportunities");
		return biz.getOpportunities(u, lastSyncDate);
	}


	/**
	 * get an Opportunity
	 * @param pk search by the Opportunity's primary key
	 * @return the found Opportunity
	 * @exception AngError thrown when an app error occurs
	 */
	public Opportunity getOpportunity(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getOpportunity");
		return biz.getOpportunity(pk);
	}


	/**
	 * delete an Opportunity
	 * @param pk the primary key of an Opportunity to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteOpportunity(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteOpportunity");
		biz.deleteOpportunity(pk);
	}

	/**
	 * load an Opportunity
	 * @param o the Opportunity we are loading
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadOpportunity(Opportunity o)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.loadOpportunity");
		biz.loadOpportunity(o);
	}

	/**
	 * add an Opportunity
	 * @param o the Opportunity we are adding
	 * @return the newly added Opportunity's primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addOpportunity(Opportunity o)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addOpportunity");
		return biz.addOpportunity(o);
	}


	/**
	 * update an Opportunity
	 * @param o the Opportunity to update
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateOpportunity(Opportunity o)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.updateOpportunity");
		biz.updateOpportunity(o);
	}


	/**
	 * delete a Contact 
	 * @param oppPk the containing Opportunity primary key
	 * @param contPk the Contact primary key to delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteContact(long oppPk, long contPk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteContact");
		biz.deleteContact(oppPk, contPk);
	}


	/**
	 * update a Contact
	 * @param oppPK the containing Opportunity primary key
	 * @param c the the Contact being added
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateContact(long oppPK, Contact c) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.udpateContact");
		biz.updateContact(oppPK, c);
	}

	/**
	 * add a new Contact to the database belonging to the 
	 * current opportunity
	 *
	 * @param oppPk the primary key of the parent opportunity
	 * @param c the Address entered for this new contact
	 * @return the primary key of the Contact object that is created
	 * @exception AngError general application error 
	 */
	public long addContact(long oppPk, Contact contact)
		throws RemoteException, AngError {
		
		if (Prefs.DEBUG) System.out.println("RMIBizServices.addContact");
		return biz.addContact(oppPk, contact);
	}


	/**
	 * get all SalesPersons in the system 
	 * @param lastSyncDate a user's last sync date or null, used to
	 * limit the query
	 * @return an TreeMap of SalesPersons
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getSalesPersons(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getSalesPersons");
		return biz.getSalesPersons(lastSyncDate);
	}

	/**
	 * get a SalesPerson using a formatted name
	 * @param pk primary key to search with
	 * @return the found SalesPerson
	 * @exception AngError thrown when an app error occurs
	 */
	public SalesPerson getSalesPerson(long pk) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getSalesPerson");
		return biz.getSalesPerson(pk);
	}


	/**
	 * update a SalesPerson
	 * @param s the SalesPerson to update with
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateSalesPerson(SalesPerson s)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.updateSalesPerson");
		biz.updateSalesPerson(s);
	}

	/**
	 * delete all user from the system 
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllSalesPerson() 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteAllSalesPerson");
		biz.deleteAllSalesPerson();
	}

	/**
	 * delete a user from the system 
	 * @param pk the SalesPerson primary key to delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteSalesPerson(SalesPerson per) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteSalesPerson");
		biz.deleteSalesPerson(per);
	}

	/**
	 * load a user to the system
	 * @param s the SalesPerson to add
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadSalesPerson(SalesPerson s) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.loadSalesPerson");
		biz.loadSalesPerson(s);
	}

	/**
	 * add a user to the system
	 * @param s the SalesPerson to add
	 * @return the newly added row's primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addSalesPerson(SalesPerson s) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addSalesPerson");
		return biz.addSalesPerson(s);
	}


	/**
	 * get all sales person id's
	 * @return an ArrayList of user IDs
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getSalesPersonIDs() 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getSalesPersonIDs");
		return biz.getSalesPersonIDs();
	}

	/**
	 * get all the users names in 'formatted' form
	 * @return an ArrayList of formatted user names
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getSalesPersonNames() 
		throws RemoteException, AngError {
		
		if (Prefs.DEBUG) System.out.println("RMIBizServices.getSalesPersonNames");
		return biz.getSalesPersonNames();
	}


	/**
	 * add a user to a UserGroup
	 * @param userPK the primary key of the user being added
	 * @param group the UserGroup begin added to
	 * @exception AngError thrown when an app error occurs
	 */
	public void addUserToGroup(long userPK, UserGroup group) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addUserToGroup");
		biz.addUserToGroup(userPK, group);
	}


	/**
	 * get all the SalesPersons in a UserGroup
	 * @param groupName the UserGroup we are searching with
	 * @return an array of SalesPersons found
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getUsersInGroup(String groupName) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getUsersInGroup");
		return biz.getUsersInGroup(groupName);
	}

	/**
	 * get a UserGroup in the system
	 * @param pk a primary key 
	 * @return a UserGroup that was found or null if not found
	 * @exception AngError thrown when an app error occurs
	 */
	public UserGroup getUserGroup(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getUserGroup");
		return biz.getUserGroup(pk);
	}

	/**
	 * get all UserGroups in the system
	 * @param lastSyncDate a user's last sync date which limits the query
	 * @return an array of UserGroups that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getUserGroups(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getUserGroups");
		return biz.getUserGroups(lastSyncDate);
	}

	/**
	 * delete all the user groups
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllUserGroup() 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteAllUserGroups");
		biz.deleteAllUserGroup();
	}

	/**
	 * load a UserGroup to the system
	 * @param g a UserGroup we are adding to the system
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadUserGroup(UserGroup g) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.loadUserGroup");
		biz.loadUserGroup(g);
	}

	/**
	 * add a UserGroup to the system
	 * @param g a UserGroup we are adding to the system
	 * @exception AngError thrown when an app error occurs
	 */
	public void addUserGroup(UserGroup g) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addUserGroup");
		biz.addUserGroup(g);
	}


	/**
	 * delete a UserGroup
	 * @param groupName the UserGroup name we delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteUserGroup(String groupName) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteUserGroup");
		biz.deleteUserGroup(groupName);	
	}


	/**
	 * delete a user in a UserGroup
	 * @param g the containing UserGroup 
	 * @param userPK the user's primary key we are deleting with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteUserInGroup(UserGroup g, long userPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteUserInGroup");
		biz.deleteUserInGroup(g, userPK);
	}


	/**
	 * get the UserGroups for a given user
	 * @param userPK the SalesPerson primary key we are searching for
	 * @return an ArrayList of UserGroup(s) this user belongs to 
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getGroupsForUser(long userPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getGroupsForUser");
		return biz.getGroupsForUser(userPK);
	}

	
	/**
	 * load a Campaign
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadCampaign(Campaign c)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.loadCampaign");
		biz.loadCampaign(c);
	}

	/**
	 * get a Campaign 
	 * @param pk a primary key
	 * @return a Campaign that was found or null if not found
	 * @exception AngError thrown when an app error occurs
	 */
	public Campaign getCampaign(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getCampaign");
		return biz.getCampaign(pk);
	}

	/**
	 * get the Campaigns 
	 * @param lastSyncDate a user's last sync date which limits the
	 * query or null if no limit
	 * @return an ArrayList of Campaigns that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCampaigns(java.util.Date lastSyncDate)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getCampaigns");
		return biz.getCampaigns(lastSyncDate);
	}

	/**
	 * delete the campaigns
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllCampaign()
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteAllCampaign");
		biz.deleteAllCampaign();
	}


	/**
	 * get a Lead
	 * @param pk the Lead primary key
	 * @return the Lead
	 */
	public Lead getLead(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getLead");
		return biz.getLead(pk);
	}

	/**
	 * get the Leads for a given Campaign 
	 * @param campaignPK the containing Campaign's primary key
	 * @return an ArrayList of Leads that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCampaignLeads(long campaignPK)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getCampaignLeads");
		return biz.getCampaignLeads(campaignPK);
	}

	/**
	 * get all Leads
	 * @param lastSyncDate a user's last sync date used to limit the 
	 * query or null if no limit
	 * @return an ArrayList of Leads that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCampaignLeads(java.util.Date lastSyncDate)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getCampaignLeads all");
		return biz.getCampaignLeads(lastSyncDate);
	}

	/**
	 * process the items that were deleted by a user when
	 * operating in a 'disconnected' mode, each object
	 * in the passed list will be deleted from the database
	 *
	 * @param deletes the ArrayList of DeleteInfo objects
	 * @exception AngError thrown when an app error occurs
	 */
	public void uploadDeletes(ArrayList deletes) 
		throws RemoteException, AngError {
	
		if (Prefs.DEBUG) System.out.println("RMIBizServices.getCampaignLeads");
		biz.uploadDeletes(deletes);
	}

	/**
	 * add a customer inventory
	 * @param ci the CustomerInventory to add
	 * @return the primary key of the newly added row
	 * @exception AngError thrown when an app error occurs
	 */
	public long addCustomerInventory(CustomerInventory c)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.addCustomerInventory");
		return biz.addCustomerInventory(c);
	}

	/**
	 * delete a customer inventory
	 * @param ci the customer inventory to delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteCustomerInventory(CustomerInventory ci)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteCustomerInventory");
		biz.deleteCustomerInventory(ci);
	}

	/**
	 * get the CustomerInventory for a given customer
	 * @param cust the Customer we are searching with
	 * @return an ArrayList of CustomerInventory objects
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCustomerInventory(long custPK) 
		throws RemoteException, AngError {
	
		if (Prefs.DEBUG) System.out.println("RMIBizServices.getCustomerInventory");
		return biz.getCustomerInventory(custPK);
	}

	/**
	 * see BizServices 
	 */
	public void deleteAllUserRole() 
		throws RemoteException, AngError {
		
		if (Prefs.DEBUG) System.out.println("RMIBizServices.deleteAllUserRole");
		biz.deleteAllUserRole();
	}

	/**
	 * see BizServices
	 */
	public ArrayList getGroupMembers(java.util.Date lastSyncDate) 
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getGroupMembers");
		return biz.getGroupMembers(lastSyncDate);
	}

	/**
	 * see BizServices
	 */
	public UserGroupMember getGroupMember(long pk)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.getGroupMember");
		return biz.getGroupMember(pk);
	}

	/**
	 * see BizServices
	 */
	public void loadGroupMember(UserGroupMember ugm)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.loadGroupMember");
		biz.loadGroupMember(ugm);
	}

	/**
	 * see BizServices
	 */
	private void loadLead(Lead lead)
		throws RemoteException, AngError {

		if (Prefs.DEBUG) System.out.println("RMIBizServices.loadLead");
		biz.loadLead(lead);
	}

	/**
	 * utility to display an error then exit
	 */
	 private void raiseError(String msg) {
		System.out.println(msg);
		System.exit(1);
	 }

}
