package sellwin.utils;

import java.util.*;
import java.io.*;
import java.text.*;
import java.awt.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class offers some routines that manage the user
 * preferences that are stored in the '.sellwinrc' file
 * in the user's home directory.  Also, other basic user
 * or system constants are stored here as well just as
 * a common place to store such things.
 */
public class Prefs {

	public final static boolean DEBUG=true;

	//database types that are supported
	public final static int ORACLE = 1;
	public final static int MYSQL  = 2;

	//AWT inset constants for grid bag field padding
	public final static int I_TOP 	= 7;
	public final static int I_LF 	= 5;
	public final static int I_BOTT 	= 7;
	public final static int I_RT 	= 5;

	public final static int FIELD_HEIGHT = 20;

	private static Properties systemProps=null;
   	public final static String CONNECTED="Connected";
   	public final static String TWO_TIER="2Tier";
    public final static String LOCAL="Local";

	public final static String[] LANGS = { "English", "French", "Spanish", "German" };
   	public final static String DEF_LANG = "defaultLanguage";
    public final static String CONN_MODE = "connectionMode";
    public final static String LOCAL_DB= "sellwin";
    public final static String LOCAL_DB_ID= "localDBid";
    public final static String LOCAL_DB_PSW= "localDBpsw";
    public final static String SERVER_NAME= "serverName";
    public final static String LAST_ID= "default";
	public final static String PREF_BASE= ".sellwinrc";
	public final static String SELLWIN_HOME= "sellwin";
	public final static String DATA_BASE= "data";

	public final static NumberFormat wholeMoney = new DecimalFormat("#,###,###,##0");
	public final static NumberFormat money = new DecimalFormat("#,###,###,##0.00");
	public final static SimpleDateFormat dateFormat = new SimpleDateFormat("MM/dd/yy");
	public final static SimpleDateFormat dateTimeFormat = new SimpleDateFormat("MM/dd/yy hh:mm aa");


	/**
	 * get the user's default language (e.g. English, Spanish) 
	 * @return a string such as 'English' or 'Spanish'
	 */
	public static String getDefaultLang() {
		Properties props = getApplProps();
		String lang = (String)(props.getProperty(DEF_LANG));
		return lang;
	}

	/**
	 * get the user's default data directory where 
	 * downloaded data will be stored
	 * @return the pathname of the data directory
	 */
	public static String getDataDir() {
		Properties props = getApplProps();
		String dir = (String)(props.getProperty(LOCAL_DB));
		return dir;
	}

	/**
 	 * get the user's last login ID
	 * @return the last login name
	 */
	public static String getLastID() {
		Properties props = getApplProps();
		String lastID = (String)(props.getProperty(LAST_ID));
		if (lastID == null)
			return "default";
		else
			return lastID;
	}
	/**
 	 * get the user's default server host name
	 * for connections to a SellWin server
	 * @return the server name
	 */
	public static String getServerName() {
		Properties props = getApplProps();
		String serverName = (String)(props.getProperty(SERVER_NAME));
		if (serverName == null)
			return "localhost";
		else
			return serverName;
	}

	/**
	 * get the user's connection type, this is either
	 * 'connected' or 'disconnected'
	 * @return the connection type
	 */
	public static String getConnectionMode() {
		Properties props = getApplProps();
		String mode = (String)(props.getProperty(CONN_MODE));
		return mode;
	}

	/**
	 * store the user properties in the .sellwinrc file
	 * in the user's home directory
	 * @param props the Properties we are storing to disk
	 */
	public static void saveApplProps(Properties props) {
		FileOutputStream outStream=null;
  		try {
			System.out.println("saving props to "+ Prefs.getApplPropsPath());
            outStream = new FileOutputStream(Prefs.getApplPropsPath());
            props.store(outStream, "SellWin application properties");
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                if (outStream != null)
                    outStream.close();
            } catch (Exception f) {
                f.printStackTrace();
            }
        }
	}

	/**
	 * get the user's default Properties from disk
	 * @return the user's Properties
	 */
	public static Properties getApplProps() {
		String appPropsPath = getApplPropsPath();
		Properties props = new Properties();

   		FileInputStream inStream=null;
        try {
            inStream = new FileInputStream(appPropsPath);
            props.load(inStream);
        } catch (FileNotFoundException e) {
            System.out.println(appPropsPath + " does not exist...creating with default values");
			props.setProperty(DEF_LANG, LANGS[0]);
			props.setProperty(CONN_MODE, LOCAL);
			props.setProperty(LAST_ID, "trial");
			props.setProperty(LOCAL_DB, "sellwin");
			props.setProperty(LOCAL_DB_ID, "localid");
			props.setProperty(LOCAL_DB_PSW, "localpsw");
			props.setProperty(SERVER_NAME, "localhost");
			saveApplProps(props);
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                if (inStream != null)
                    inStream.close();
            } catch (Exception f) {
                f.printStackTrace();
            }
        }
		return props;
	}

	/**
 	 * get the database server host (RMI Server only)
	 * @return the database server IP or hostname
	 */
	public static String getDatabaseHost() {
		Properties props = System.getProperties();
		return props.getProperty("DATABASE_HOST");
	}

	/**
 	 * get the database server port (RMI Server only)
	 * @return the database server port
	 */
	public static String getDatabasePort() {
		Properties props = System.getProperties();
		return props.getProperty("DATABASE_PORT");
	}

	/**
 	 * get the database server type (RMI Server only)
	 * @return the database server type
	 */
	public static String getDatabaseType() {
		Properties props = System.getProperties();
		return props.getProperty("DATABASE_TYPE");
	}

	/**
 	 * get the database name
	 * @return the database name
	 */
	public static String getDatabaseName() {
		Properties props = System.getProperties();
		return props.getProperty("DATABASE_NAME");
	}

	/**
 	 * get the database user id 
	 * @return the database user id
	 */
	public static String getDatabaseID() {
		Properties props = System.getProperties();
		return props.getProperty("DATABASE_ID");
	}

	/**
 	 * get the database server user password 
	 * @return the database user password
	 */
	public static String getDatabasePassword() {
		Properties props = System.getProperties();
		return props.getProperty("DATABASE_PASSWORD");
	}

	/**
 	 * get the rmi server host IP or hostname (RMI server only)
	 * @return the RMI server host or IP
	 */
	public static String getServerHost() {
		Properties props = System.getProperties();
		return props.getProperty("SERVER_HOST");
	}
	/**
 	 * get the local database name
	 * @return the local database name
	 */
	public static String getLocalDB() {
		Properties props = System.getProperties();
		return props.getProperty("LOCAL_DB");
	}
	/**
 	 * get the local database id
	 * @return the local database id
	 */
	public static String getLocalDBID() {
		Properties props = System.getProperties();
		return props.getProperty("LOCAL_DB_ID");
	}
	/**
 	 * get the local database password
	 * @return the local database password
	 */
	public static String getLocalDBPSW() {
		Properties props = System.getProperties();
		return props.getProperty("LOCAL_DB_PSW");
	}

	/**
	 * get the user's home directory 
	 * @return the user's home directory path 
	 */
	public static String getHomeDir() { 
		if (systemProps == null)
			systemProps = System.getProperties();
		return systemProps.getProperty("user.home");
	}

	/**
	 * get the system file separator character 
	 * @return the file separator character
	 */
	public static String getFileSep() {
		if (systemProps == null)
			systemProps = System.getProperties();

		return systemProps.getProperty("file.separator");
	}

	/**
	 * get the full path name to the user's default
	 * property file
	 * @return the path name of the property file
	 */
	public static String getApplPropsPath() {
		String homeDir = getHomeDir();
		return homeDir + getFileSep() + PREF_BASE;
	}
}

