package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;
import java.util.ArrayList;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * This class implements the DBInterface for
 * the Activity class objects which are stored
 * in the activity database table.
 */
public class ActivityDB extends DBType  implements DBInterface {

	private Connection con;

	private final static String selectByOppQuery = 
			"SELECT " +
				"pk, subject, " +
    			"type, place, message, start_date, duration, " +
    			"group_alarm, group_alarm_ack, " +
    			"alarm, alarm_ack, modified_by, modified_date " +
			"FROM activity " +
			"WHERE opp_pk=";

	private final static String selectAlarmsQuery = 
			"SELECT " +
				"activity.pk, opportunity.pk, activity.subject, " +
    			"activity.type, activity.place, activity.message, " +
				"activity.start_date, activity.duration," +
    			"activity.group_alarm, activity.group_alarm_ack, " +
    			"activity.alarm, activity.alarm_ack, " +
				"activity.modified_by, activity.modified_date " +
			"FROM activity, opportunity, user_group, user_group_member " +
			"WHERE opportunity.pk = activity.opp_pk AND " +
			"opportunity.group_name = user_group.name AND " +
			"user_group.pk = user_group_member.user_group_pk AND " +
			"activity.alarm = 'Y' AND activity.alarm_ack = 'N' AND ";

	private final static String selectQuery = 
			"SELECT " +
				"opp_pk, subject, " +
    			"type, place, message, start_date, duration, " +
    			"group_alarm, group_alarm_ack, " +
    			"alarm, alarm_ack, modified_by, modified_date " +
			"FROM activity " +
			"WHERE pk=";

	private final static String updateQuery = 
			"UPDATE activity " +
			"SET " ;

	private final static String insertQuery = 
			"INSERT INTO activity VALUES ( ";

	private final static String deleteQuery = 
			"DELETE FROM activity WHERE pk =";

	private final static String deleteOppQuery =
			"DELETE FROM activity where opp_pk = ";
			

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 *
	 */
	public ActivityDB() {
	}

	/**
	 * construct using a particular database type
	 * @param dbType the type of database found in Prefs.java
	 */
	public ActivityDB(int dbType) {
		DB_TYPE=dbType;
	}

	/**
	 * a constructor that accepts an existing Connection
	 * to use for future operations
	 *
	 * @param con the Connection to use
	 */
	public ActivityDB(Connection con) {
		this.con = con;
	}

	/**
	 * get the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single activity row using the passed
	 * primary key
	 * 
	 * @param pk the primary key to search with
	 * @return the Activity(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		Activity activity = new Activity("");
		long pkValue = ((Long)pk).longValue();
		activity.setPK(pkValue);
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + pkValue;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			while (rs.next()) {
				i=1;
				activity.setOppPK(rs.getLong(i)); i++;
				activity.setSubject(rs.getString(i)); i++;
				activity.setType(rs.getString(i)); i++;
				activity.setPlace(rs.getString(i)); i++;
				activity.setMessage(rs.getString(i)); i++;
				activity.setStartDate(rs.getDate(i)); i++;
				activity.setDuration(new Integer(rs.getInt(i))); i++;
				boolean x;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setGroupAlarm(x); i++;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setGroupAlarmAck(x); i++;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setAlarm(new Boolean(x)); i++;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setAlarmAck(x); i++;
				activity.setModifiedBy(rs.getString(i)); i++;
				activity.setModifiedDate(rs.getDate(i)); 
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return activity;
	}


	/**
	 * update a single activity row using the passed
	 * Activity object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param obj the object we want to update with
	 * @exception java.sql.SQLException 
	 */
	public void updateRow(Object obj) 
		throws SQLException {

		Activity activity = (Activity)obj;
		long pk = activity.getPK();

		Statement stmt = null;
		StringBuffer query = new StringBuffer(updateQuery);

		try {
			stmt = con.createStatement();

			query.append("subject=");
			query.append(JDBC.quoteMore(activity.getSubject()));
			query.append("type=");
			query.append(JDBC.quoteMore(activity.getType()));
			query.append("place=");
			query.append(JDBC.quoteMore(activity.getPlace()));
			query.append("message=");
			query.append(JDBC.quoteMore(activity.getMessage()));
			query.append("start_date=");
			query.append(JDBC.quoteMore(DateUtils.format(DB_TYPE,activity.getStartDate())));
			query.append("duration=");
			query.append(activity.getDuration()).append(",");
			String x;
			if (activity.getGroupAlarm()) x="Y"; else x="N";
			query.append("group_alarm=");
			query.append(JDBC.quoteMore(x));
			if (activity.getGroupAlarmAck()) x="Y"; else x="N";
			query.append("group_alarm_ack=");
			query.append(JDBC.quoteMore(x));
			if (activity.getAlarm().booleanValue()) x="Y"; else x="N";
			query.append("alarm=");
			query.append(JDBC.quoteMore(x));
			if (activity.getAlarmAck()) x="Y"; else x="N";
			query.append("alarm_ack=");
			query.append(JDBC.quoteMore(x));
			query.append("modified_by=");
			query.append(JDBC.quoteMore(activity.getModifiedBy()));
			if (DB_TYPE == Prefs.MYSQL)
				query.append("modified_date=CURRENT_DATE ");
			else
				query.append("modified_date=SYSDATE ");
			query.append("WHERE pk=").append(activity.getPK());	

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int updatedRows = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close(); 
			} catch (SQLException x) { }
		}
	}

	/**
	 * insert a new activity row using the passed
	 * Activity object as the column values.
	 * 
	 * @param obj the object we are trying to insert
	 * @param load true if the row is to be loaded, false if
	 * the row is to be added for the first time
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public long insertRow(Object obj, boolean load) 
		throws SQLException {

		Activity activity = (Activity)obj;
		if (!load)
			activity.setPK(DBUtils.generatePK());

		Statement stmt = null;
		StringBuffer query = new StringBuffer(insertQuery);

		try {
			stmt = con.createStatement();

			query.append(activity.getPK()).append(",");
			query.append(activity.getOppPK()).append(",");
			query.append(JDBC.quoteMore(activity.getSubject()));
			query.append(JDBC.quoteMore(activity.getType()));
			query.append(JDBC.quoteMore(activity.getPlace()));
			query.append(JDBC.quoteMore(activity.getMessage()));
			query.append(JDBC.quoteMore(DateUtils.format(DB_TYPE, activity.getStartDate())));
			query.append(activity.getDuration()).append(",");
			String x;
			if (activity.getGroupAlarm()) x="Y"; else x="N";
			query.append(JDBC.quoteMore(x));
			if (activity.getGroupAlarmAck()) x="Y"; else x="N";
			query.append(JDBC.quoteMore(x));
			if (activity.getAlarm().booleanValue()) x = "Y"; else x="N";
			query.append(JDBC.quoteMore(x));
			if (activity.getAlarmAck()) x = "Y"; else x="N";
			query.append(JDBC.quoteMore(x));
			query.append(JDBC.quoteMore(activity.getModifiedBy()));
			if (DB_TYPE == Prefs.MYSQL)
				query.append("CURRENT_DATE");
			else
				query.append("SYSDATE");
			query.append(")");

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int rc = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}

		return activity.getPK();
	}

	/**
	 * delete a single activity row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();
		String query = deleteQuery + pkValue;
	
		Statement stmt = null;
		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }		
		}
	}

	/**
	 * delete all activity rows using the passed
	 * opportunity primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteOpportunityRows(Object obj) 
		throws SQLException {

		long opp_pk = ((Long)obj).longValue();
		String query = deleteOppQuery + opp_pk;
	
		Statement stmt = null;
		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }		
		}
	}

	/**
	 * select all activity rows for a given opportunity
	 * 
	 * @param opp_pk the opportunity primary key we are searching with
	 * @param list the list of activities owned by the opportunity
 	 * which gets filled in by this routine
	 * @exception java.sql.SQLException 
	 */
	public final void selectByOppRows(long opp_pk, ArrayList list) 
		throws SQLException {

		Activity activity = null;
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectByOppQuery + opp_pk;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			while (rs.next()) {
				i=1;
				activity = new Activity();
				activity.setPK(rs.getLong(i)); i++;
				activity.setOppPK(opp_pk);
				activity.setSubject(rs.getString(i)); i++;
				activity.setType(rs.getString(i)); i++;
				activity.setPlace(rs.getString(i)); i++;
				activity.setMessage(rs.getString(i)); i++;
				activity.setStartDate(rs.getDate(i)); i++;
				activity.setDuration(new Integer(rs.getInt(i))); i++;
				boolean x;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setGroupAlarm(x); i++;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setGroupAlarmAck(x); i++;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setAlarm(new Boolean(x)); i++;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setAlarmAck(x); i++;
				activity.setModifiedBy(rs.getString(i)); i++;
				activity.setModifiedDate(rs.getDate(i)); 
				list.add(activity);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}
	}

	/**
	 * select all activity rows for a given user that
	 * have active alarms earlier than tomorrow 
	 * 
	 * @param user_pk the opportunity primary key we are searching with
	 * @param list the list of activities owned by the user
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectActiveAlarms(long user_pk)
		throws SQLException {

		ArrayList acts = new ArrayList();
		Activity activity = null;
		Statement stmt = null;
		ResultSet rs = null;
		StringBuffer query = new StringBuffer(selectAlarmsQuery);

		if (DB_TYPE == Prefs.MYSQL)
			query.append( "activity.start_date < DATE_ADD(CURRENT_DATE, INTERVAL 1 DAY) AND ");
		else
			query.append( "activity.start_date < SYSDATE + 1 AND ");
		
		query.append("user_group_member.user_pk = ");
		query.append(user_pk);

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query.toString());
			rs = stmt.executeQuery(query.toString());

			int i;
			while (rs.next()) {
				i=1;
				activity = new Activity();
				activity.setPK(rs.getLong(i)); i++;
				activity.setOppPK(rs.getLong(i)); i++;
				activity.setSubject(rs.getString(i)); i++;
				activity.setType(rs.getString(i)); i++;
				activity.setPlace(rs.getString(i)); i++;
				activity.setMessage(rs.getString(i)); i++;
				activity.setStartDate(rs.getDate(i)); i++;
				activity.setDuration(new Integer(rs.getInt(i))); i++;
				boolean x;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setGroupAlarm(x); i++;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setGroupAlarmAck(x); i++;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setAlarm(new Boolean(x)); i++;
				if (rs.getString(i).charAt(0)=='Y') x=true; else x=false;
				activity.setAlarmAck(x); i++;
				activity.setModifiedBy(rs.getString(i)); i++;
				activity.setModifiedDate(rs.getDate(i)); 
				acts.add(activity);
			}
		} catch (SQLException e) {
			LogWrite.write(e);
			throw e;
		} catch (Exception f) {
			LogWrite.write(f);
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		LogWrite.write("ActivityDB.selectActiveAlarms acts="+ acts.size());
		return acts;
	}

	/**
	 * truncate the whole table
	 *
	 * @exception java.sql.SQLException 
	 */
	public final void truncate()
		throws SQLException {

		String query = "truncate table activity";
	
		Statement stmt = null;
		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }		
		}
	}
}
