package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * This class implements the DBInterface for
 * the Address class objects which are stored
 * in the address database table.
 */
public class AddressDB extends DBType implements DBInterface {

	private Connection con;

	private final static String selectQuery = 
			"SELECT " +
				"first_name, last_name, middle_initial, " +
    			"address_line1, address_line2, title ," +
    			"phone, fax, pager, cell, email, " +
    			"city, state, country, zip, zip4, " +
				"modified_by, modified_date " +
			"FROM address " +
			"WHERE pk=";

	private final static String updateQuery = 
			"UPDATE address " +
			"SET ";

	private final static String insertQuery = 
			"INSERT INTO address VALUES ( ";

	private final static String deleteQuery = 
			"DELETE FROM address WHERE pk =";

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 */
	public AddressDB() {
	}

	/**
 	 * construct using a db type
	 * @param dbType the database type 
	 */
	public AddressDB(int dbType) {
		DB_TYPE=dbType;
	}

	/**
	 * a constructor that accepts an existing Connection
	 * to use for future operations
	 *
	 * @param con the Connection to use
	 */
	public AddressDB(Connection con) {
		this.con = con;
	}

	/**
	 * get the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single address row using the passed
	 * primary key
	 * 
	 * @param pk the primary key we are searching with
	 * @return the Address(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		Address address = new Address();
		long pkValue = ((Long)pk).longValue();
		address.setPK(pkValue);
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + pkValue;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			while (rs.next()) {
				i=1;
				address.setFirstName(rs.getString(i)); i++;
				address.setLastName(rs.getString(i)); i++;
				address.setMiddleInitial(rs.getString(i)); i++;
				address.setAddressLine1(rs.getString(i)); i++;
				address.setAddressLine2(rs.getString(i)); i++;
				address.setTitle(rs.getString(i)); i++;
				address.setPhone(rs.getString(i)); i++;
				address.setFax(rs.getString(i)); i++;
				address.setPager(rs.getString(i)); i++;
				address.setCell(rs.getString(i)); i++;
				address.setEmail(rs.getString(i)); i++;
				address.setCity(rs.getString(i)); i++;
				address.setState(rs.getString(i)); i++;
				address.setCountry(rs.getString(i)); i++;
				address.setZip(rs.getString(i)); i++;
				address.setZip4(rs.getString(i)); i++;
				address.setModifiedBy(rs.getString(i)); i++;
				address.setModifiedDate(rs.getDate(i)); 
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return address;
	}


	/**
	 * update a single address row using the passed
	 * Address object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param obj the object we want to update with
	 * @exception java.sql.SQLException 
	 */
	public void updateRow(Object obj) 
		throws SQLException {

		Address address = (Address)obj;
		long pk = address.getPK();

		Statement stmt = null;
		StringBuffer query = new StringBuffer(updateQuery);

		try {
			stmt = con.createStatement();

			query.append("first_name=");
			query.append(JDBC.quoteMore(address.getFirstName()));
			query.append("last_name=");
			query.append(JDBC.quoteMore(address.getLastName()));
			query.append("middle_initial=");
			query.append(JDBC.quoteMore(address.getMiddleInitial()));
			query.append("address_line1=");
			query.append(JDBC.quoteMore(address.getAddressLine1()));
			query.append("address_line2=");
			query.append(JDBC.quoteMore(address.getAddressLine2()));
			query.append("title=");
			query.append(JDBC.quoteMore(address.getTitle()));
			query.append("phone=");
			query.append(JDBC.quoteMore(address.getPhone()));
			query.append("fax=");
			query.append(JDBC.quoteMore(address.getFax()));
			query.append("pager=");
			query.append(JDBC.quoteMore(address.getPager()));
			query.append("cell=");
			query.append(JDBC.quoteMore(address.getCell()));
			query.append("email=");
			query.append(JDBC.quoteMore(address.getEmail()));
			query.append("city=");
			query.append(JDBC.quoteMore(address.getCity()));
			query.append("state=");
			query.append(JDBC.quoteMore(address.getState()));
			query.append("country=");
			query.append(JDBC.quoteMore(address.getCountry()));
			query.append("zip=");
			query.append(JDBC.quoteMore(address.getZip()));
			query.append("zip4=");
			query.append(JDBC.quote(address.getZip4()));
			query.append(" WHERE pk=");
			query.append(address.getPK());	

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int updatedRows = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close(); 
			} catch (SQLException x) { }
		}
	}

	/**
	 * insert a new address row using the passed
	 * Address object as the column values.
	 * 
	 * @param obj the object we are inserting
	 * @param load true if the row is to be loaded, false if the
	 * row is to be added for the first time
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public long insertRow(Object obj, boolean load) 
		throws SQLException {

		Address address = (Address)obj;
		if (!load) {
			address.setPK(DBUtils.generatePK());
			LogWrite.write("generated address pk = " + address.getPK());
			address.setPK(address.getPK() + 1);
		}

		Statement stmt = null;
		StringBuffer query = new StringBuffer(insertQuery);

		try {
			stmt = con.createStatement();

			query.append(address.getPK()).append(",");
			query.append(JDBC.quoteMore(address.getFirstName()));
			query.append(JDBC.quoteMore(address.getLastName()));
			query.append(JDBC.quoteMore(address.getMiddleInitial()));
			query.append(JDBC.quoteMore(address.getAddressLine1()));
			query.append(JDBC.quoteMore(address.getAddressLine2()));
			query.append(JDBC.quoteMore(address.getTitle()));
			query.append(JDBC.quoteMore(address.getPhone()));
			query.append(JDBC.quoteMore(address.getFax()));
			query.append(JDBC.quoteMore(address.getPager()));
			query.append(JDBC.quoteMore(address.getCell()));
			query.append(JDBC.quoteMore(address.getEmail()));
			query.append(JDBC.quoteMore(address.getCity()));
			query.append(JDBC.quoteMore(address.getState()));
			query.append(JDBC.quoteMore(address.getCountry()));
			query.append(JDBC.quoteMore(address.getZip()));
			query.append(JDBC.quoteMore(address.getZip4()));
			query.append(JDBC.quoteMore(address.getModifiedBy()));
			query.append(JDBC.quote(DateUtils.format(DB_TYPE, address.getModifiedDate())));
			query.append(")");

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int rc = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}

		return address.getPK();
	}

	/**
	 * delete a single address row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();
		String query = deleteQuery + pkValue;
	
		Statement stmt = null;
		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }		
		}
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table address";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }


}
