package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.util.*;
import java.sql.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * This  class implements the DBInterface for
 * the Customer class objects which are stored
 * in the customer database table.
 */
public class CustomerDB extends DBType implements DBInterface {

	private Connection con;

	private final static String selectNamesQuery = 
			"SELECT " +
				"name " +
			"FROM customer " +
			"ORDER BY name";

	private final static String selectByNameQuery = 
			"SELECT " +
				"pk, ship_address_pk, bill_address_pk, " +
    			"annual_sales, industry, channel ," +
    			"contact_pk, modified_by, modified_date " +
			"FROM customer " +
			"WHERE name=";

	private final static String selectQuery = 
			"SELECT " +
				"name, ship_address_pk, bill_address_pk, " +
    			"annual_sales, industry, channel ," +
    			"contact_pk, modified_by, modified_date " +
			"FROM customer " +
			"WHERE pk=";

	private final static String selectAllCustsQuery = 
			"SELECT " +
				"pk, name, ship_address_pk, bill_address_pk, " +
    			"annual_sales, industry, channel ," +
    			"contact_pk, modified_by, modified_date " +
			"FROM customer " ;

	private final static String updateQuery = 
			"UPDATE customer " +
			"SET " ;

	private final static String insertQuery = 
			"INSERT INTO customer VALUES ( ";

	private final static String deleteQuery = 
			"DELETE FROM customer WHERE pk =";

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 *
	 */
	public CustomerDB() {
	}

	/**
	 * construct specifying a particular db type
	 * @param dbType db type to assume
	 */
	public CustomerDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a constructor that accepts an existing Connection
	 * to use for future operations
	 *
	 * @param con the Connection to use
	 */
	public CustomerDB(Connection con) {
		this.con = con;
	}

	/**
	 * get the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single address row using the passed
	 * primary key
	 * 
	 * @param pk the primary key to search with
	 * @return the row that was selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		Customer cust = new Customer();
		long pkValue = ((Long)pk).longValue();
		cust.setPK(pkValue);
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + pkValue;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			long shipAddressPK, billAddressPK, contactAddressPK;

			AddressDB addrDB = new AddressDB(DB_TYPE);
			addrDB.setConnection(getConnection());
			Address shipAddress, billAddress, contactAddress;

			while (rs.next()) {
				i=1;
				cust.setName(rs.getString(i)); i++;
				shipAddressPK = rs.getLong(i); i++; //ship address
				shipAddress = (Address)addrDB.selectRow(new Long(shipAddressPK));
				cust.setShipAddress(shipAddress);
				billAddressPK = rs.getLong(i); i++; //bill address
				billAddress = (Address)addrDB.selectRow(new Long(billAddressPK));
				cust.setBillAddress(billAddress);
				cust.setAnnualSales(rs.getInt(i)); i++;
				cust.setIndustry(rs.getString(i)); i++;
				cust.setChannel(rs.getString(i)); i++;
				contactAddressPK = rs.getLong(i); i++; //contact address
				contactAddress = (Address)addrDB.selectRow(new Long(contactAddressPK));
				cust.setContact(contactAddress);
				cust.setModifiedBy(rs.getString(i)); i++;
				cust.setModifiedDate(rs.getDate(i)); 
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return cust;
	}

	/**
	 * select all customer rows 
	 * 
	 * @param lastSyncDate a user's last sync date used to limit
	 * the query or null if no limit
	 * @return the Customer(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectAllRows(java.util.Date lastSyncDate) 
		throws SQLException {

		ArrayList custs = new ArrayList();
		Customer cust = new Customer();
		Statement stmt = null;
		ResultSet rs = null;
		StringBuffer query = new StringBuffer();
		query.append(selectAllCustsQuery);

		if (lastSyncDate != null) {
			query.append(" WHERE customer.modified_date > ");
            query.append(DateUtils.formatDateTime(DB_TYPE, lastSyncDate));
		}

		query.append(" ORDER BY name");

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query.toString());
			rs = stmt.executeQuery(query.toString());

			int i;
			long shipAddressPK, billAddressPK, contactAddressPK;

			AddressDB addrDB = new AddressDB(DB_TYPE);
			addrDB.setConnection(getConnection());
			Address shipAddress, billAddress, contactAddress;

			while (rs.next()) {
				i=1;
				cust = new Customer();
				cust.setPK(rs.getLong(i)); i++;
				cust.setName(rs.getString(i)); i++;
				shipAddressPK = rs.getLong(i); i++; //ship address
				shipAddress = (Address)addrDB.selectRow(new Long(shipAddressPK));
				cust.setShipAddress(shipAddress);
				billAddressPK = rs.getLong(i); i++; //bill address
				billAddress = (Address)addrDB.selectRow(new Long(billAddressPK));
				cust.setBillAddress(billAddress);
				cust.setAnnualSales(rs.getInt(i)); i++;
				cust.setIndustry(rs.getString(i)); i++;
				cust.setChannel(rs.getString(i)); i++;
				contactAddressPK = rs.getLong(i); i++; //contact address
				contactAddress = (Address)addrDB.selectRow(new Long(contactAddressPK));
				cust.setContact(contactAddress);
				cust.setModifiedBy(rs.getString(i)); i++;
				cust.setModifiedDate(rs.getDate(i)); 

				custs.add(cust);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return custs;
	}


	/**
	 * update a single address row using the passed
	 * Address object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param obj the object to update with
	 * @exception java.sql.SQLException 
	 */
	public void updateRow(Object obj) 
		throws SQLException {

		Customer cust = (Customer)obj;

		Statement stmt = null;

		try {

			//update all the children hanging off this customer first
			AddressDB addrDB = new AddressDB(DB_TYPE);
			addrDB.setConnection(getConnection());

			Address contactAddress = cust.getContact();
			if (contactAddress.isModified())
				addrDB.updateRow(contactAddress);
			Address shipAddress = cust.getShipAddress();
			if (shipAddress.isModified())
				addrDB.updateRow(shipAddress);
			Address billAddress = cust.getBillAddress();
			if (billAddress.isModified())
				addrDB.updateRow(billAddress);

			//update the customer's inventory
			CustomerInventoryDB custInventoryDB = new CustomerInventoryDB(DB_TYPE);
			custInventoryDB.setConnection(getConnection());
			ArrayList inventory = cust.getInventory();
			CustomerInventory ci;
			for (int i=0;i<inventory.size();i++) {
				ci = (CustomerInventory)inventory.get(i);
				if (ci.getPK() < 1) {
					custInventoryDB.insertRow(ci, false);	
				} 
			}

			//now preceed to update the customer itself
			StringBuffer query = new StringBuffer(updateQuery);

			stmt = con.createStatement();

			query.append("name=");
			query.append(JDBC.quoteMore(cust.getName()));
			query.append("annual_sales=");
			query.append(cust.getAnnualSales()).append(",");
			query.append("industry=");
			query.append(JDBC.quoteMore(cust.getIndustry()));
			query.append("channel=");
			query.append(JDBC.quote(cust.getChannel()));
			query.append(" WHERE pk=");
			query.append(cust.getPK());	

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int updatedRows = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close(); 
			} catch (SQLException x) { }
		}
	}

	/**
	 * insert a new customer row using the passed
	 * Customer object as the column values.
	 * 
	 * @param obj the object we are going to insert
	 * @param load true if the row is not new, false if the row
	 * is new
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public long insertRow(Object obj, boolean load) 
		throws SQLException {

		Customer cust = (Customer)obj;

		if (!load)
			cust.setPK(DBUtils.generatePK());

		Statement stmt = null;

		try {
			AddressDB addrDB = new AddressDB(DB_TYPE);
			addrDB.setConnection(getConnection());
			cust.getShipAddress().setPK(addrDB.insertRow(cust.getShipAddress(), load));
			LogWrite.write("here pk=" + cust.getShipAddress().getPK());

			cust.getBillAddress().setPK(cust.getShipAddress().getPK() + 1);

			cust.getBillAddress().setPK(addrDB.insertRow(cust.getBillAddress(), true));
			LogWrite.write("here2 pk=" + cust.getBillAddress().getPK());

			cust.getContact().setPK(cust.getBillAddress().getPK() + 1);

			long newkey = addrDB.insertRow(cust.getContact(), true);

			cust.getContact().setPK(newkey);
			LogWrite.write("here3 pk=" + cust.getContact().getPK());

			StringBuffer query = new StringBuffer(insertQuery);
			stmt = con.createStatement();

			query.append(cust.getPK()).append(",");
			query.append(JDBC.quoteMore(cust.getName()));
			query.append(cust.getShipAddress().getPK()).append(",");
			query.append(cust.getBillAddress().getPK()).append(",");
			query.append(cust.getAnnualSales()).append(",");
			query.append(JDBC.quoteMore(cust.getIndustry()));
			query.append(JDBC.quoteMore(cust.getChannel()));
			query.append(cust.getContact().getPK()).append(",");
			query.append(JDBC.quoteMore(cust.getModifiedBy()));
			query.append(JDBC.quote(DateUtils.format(DB_TYPE, cust.getModifiedDate())));
			query.append(")");

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int rc = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}

		return cust.getPK();
	}

	/**
	 * delete a single address row using the passed
	 * primary key value
	 *
	 * @param ojb Customer to delete
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		Customer cust = (Customer)obj;

		String query = deleteQuery + cust.getPK();
	
		Statement stmt = null;
		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);

			//delete the children 
			AddressDB addr = new AddressDB(DB_TYPE);
			addr.setConnection(getConnection());
			addr.deleteRow(new Long(cust.getContact().getPK()));
			addr.deleteRow(new Long(cust.getShipAddress().getPK()));
			addr.deleteRow(new Long(cust.getBillAddress().getPK()));
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }		
		}
	}

	/**
	 * delete a single address row using the passed
	 * name key
	 *
	 * @param name the customer name used as a key
	 * @exception java.sql.SQLException 
	 */
	public final void deleteByNameRow(String name) 
		throws SQLException {

		Statement stmt = null;

		try {
			Customer cust = selectByNameRow(name);

			deleteRow(new Long(cust.getPK()));
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }		
		}
	}

	/**
	 * select a single address row using the passed
	 * name as the key
	 * 
	 * @param name the name to search with
	 * @return the Customer that was selected
	 * @exception java.sql.SQLException 
	 */
	public final Customer selectByNameRow(String name) 
		throws SQLException {

		Customer cust = new Customer();
		cust.setName(name);
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectByNameQuery + "'" + name + "'";

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			long shipAddressPK, billAddressPK, contactAddressPK;

			AddressDB addrDB = new AddressDB(DB_TYPE);
			addrDB.setConnection(getConnection());
			Address shipAddress, billAddress, contactAddress;

			while (rs.next()) {
				i=1;
				cust.setPK(rs.getLong(i)); i++;
				shipAddressPK = rs.getLong(i); i++; //ship address
				shipAddress = (Address)addrDB.selectRow(new Long(shipAddressPK));
				cust.setShipAddress(shipAddress);
				billAddressPK = rs.getLong(i); i++; //bill address
				billAddress = (Address)addrDB.selectRow(new Long(billAddressPK));
				cust.setBillAddress(billAddress);
				cust.setAnnualSales(rs.getInt(i)); i++;
				cust.setIndustry(rs.getString(i)); i++;
				cust.setChannel(rs.getString(i)); i++;
				contactAddressPK = rs.getLong(i); i++; //contact address
				contactAddress = (Address)addrDB.selectRow(new Long(contactAddressPK));
				cust.setContact(contactAddress);
				cust.setModifiedBy(rs.getString(i)); i++;
				cust.setModifiedDate(rs.getDate(i)); 
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return cust;
	}

	/**
	 * select all the customer names
	 * 
	 * @return the names(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final TreeMap selectAllNames() 
		throws SQLException {

		TreeMap names = new TreeMap();
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectNamesQuery;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			while (rs.next()) {
				names.put(rs.getString(1), rs.getString(1)); 
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return names;
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table customer";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

}
