package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;
import java.util.ArrayList;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * This class implements the DBInterface for
 * the Lead class objects which are stored
 * in the lead database table.
 */
public class LeadDB extends DBType implements DBInterface {
	private Connection con;

	private final static String deleteQuery =
		"DELETE FROM lead WHERE pk = ";
	private final static String insertQuery =
		"INSERT INTO lead " +
		"VALUES (";
	private final static String updateQuery = 
			"UPDATE lead " +
			"SET ";
	private final static String selectQuery =
			"SELECT campaign_pk, address_pk, company, " +
			"interest_level, contacted_by_pk, "+
  			"contact_date, converted_opp, "+
  			"modified_by, modified_date " +
			"FROM lead "+
			"WHERE pk = ";

	private final static String selectAllQuery =
			"SELECT pk, campaign_pk, address_pk, company, " +
			"interest_level, contacted_by_pk, "+
  			"contact_date, converted_opp, "+
  			"modified_by, modified_date " +
			"FROM lead ";

	private final static String selectByCampaignQuery =
			"SELECT pk, address_pk, company, " +
			"interest_level, contacted_by_pk, "+
  			"contact_date, converted_opp, "+
  			"modified_by, modified_date " +
			"FROM lead "+
			"WHERE campaign_pk = ";

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 */
	public LeadDB() {
	}

	/**
	 * construct using a db type
	 * @param dbType the db type to assume
	 */
	public LeadDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a version of the constructor when a connection
	 * is already obtained from somewhere else
	 *
	 * @param con the Connection to use 
	 */
	public LeadDB(Connection con) {
		this.con = con;
	}

	/**
	 * return the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public final void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single lead row using the passed
	 * primary key
	 * 
	 * @param name description
	 * @return the Lead row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		Lead lead = new Lead();
		lead.setPK(((Long)pk).longValue());

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + lead.getPK();

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;

			while (rs.next()) {
				i=1;
				lead.setCampaignKey(rs.getLong(i)); i++;

				lead.getAddress().setPK(rs.getLong(i)); i++;
				AddressDB addr = new AddressDB(DB_TYPE);
				addr.setConnection(getConnection());
				Address a = (Address)addr.selectRow(new Long(lead.getAddress().getPK()));
				lead.setAddress(a);

				lead.setCompany(rs.getString(i)); i++;
				lead.setInterestLevel(rs.getString(i)); i++;

				lead.getContactedBy().setPK(rs.getLong(i)); i++;
				SalesPersonDB spd = new SalesPersonDB(DB_TYPE);
				spd.setConnection(getConnection());
				SalesPerson sp = (SalesPerson)spd.selectRow(new Long(lead.getContactedBy().getPK()));
				lead.setContactedBy(sp);

				lead.setContactDate(rs.getDate(i)); i++;
				String x = rs.getString(i); i++;
				Boolean val;	
				if (x.equals("Y")) val = new Boolean(true); else val = new Boolean(false);
				lead.setConverted(val);
				lead.setModifiedBy(rs.getString(i)); i++;
				lead.setModifiedDate(rs.getDate(i)); 
			}

		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return lead;
	}


	/**
	 * update a single lead row using the passed
	 * Lead object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param name description
	 * @exception java.sql.SQLException 
	 */
	public final void updateRow(Object obj) 
		throws SQLException {

		Lead lead = (Lead)obj;

		StringBuffer query = new StringBuffer(updateQuery);
		Statement stmt = con.createStatement();

		String flag;
		Boolean x = lead.getConverted();
		if (x.booleanValue()) flag = "Y"; else flag = "N";
		query.append("converted_opp=");
		query.append("'").append(flag).append("',");
		query.append("modified_by=");
		query.append("'").append(lead.getModifiedBy()).append("',");
		query.append("modified_date=");
		if (DB_TYPE == Prefs.MYSQL)
			query.append("CURRENT_DATE");
		else
			query.append("SYSDATE");
		query.append("  WHERE pk=");
		query.append(lead.getPK()); 

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int updatedRows = stmt.executeUpdate(query.toString());
	}

	/**
	 * insert a new lead row using the passed
	 * Lead object as the column values.
	 * 
	 * @param obj the Lead to insert or load
	 * @param load true if the Lead is to be loaded or false if
	 * its to be added
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public final long insertRow(Object obj, boolean load) 
		throws SQLException {

		Lead lead = (Lead)obj;

		//insert the Address first
		Address address = lead.getAddress();
		AddressDB a= new AddressDB(DB_TYPE);
		a.setConnection(getConnection());
		lead.getAddress().setPK(a.insertRow(address, load));


		//we are assuming that the 'contacted by' sales person
		//will be already in the system when this lead gets added
		//so we won't add it here

		if (!load)
			lead.setPK(DBUtils.generatePK());
		StringBuffer query = new StringBuffer(insertQuery);

		Statement stmt = con.createStatement();

		query.append(lead.getPK()).append(",");
		query.append(lead.getCampaignKey()).append(",");
		query.append(lead.getAddress().getPK()).append(",");
		query.append(JDBC.quoteMore(lead.getCompany()));
		query.append(JDBC.quoteMore(lead.getInterestLevel()));
		query.append(lead.getContactedBy().getPK()).append(",");
		query.append(JDBC.quoteMore(DateUtils.format(DB_TYPE, lead.getContactDate())));
	
		String x;
		if (lead.getConverted().booleanValue()) x = "Y"; else x="N";
		query.append(JDBC.quoteMore(x));
		query.append(JDBC.quoteMore(lead.getModifiedBy()));
		if (DB_TYPE == Prefs.MYSQL)
			query.append("CURRENT_DATE");
		else
			query.append("SYSDATE");
		query.append(")");

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int rc = stmt.executeUpdate(query.toString());

		return lead.getPK();
	}

	/**
	 * delete a single lead row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();

		String query = deleteQuery + pkValue;

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * select all lead rows using the passed
	 * campaign primary key
	 * 
	 * @param name description
	 * @return the Lead row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectByCampaignRows(long camp_pk) 
		throws SQLException {

		ArrayList leads = new ArrayList();

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectByCampaignQuery + camp_pk;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			Lead lead=null;

			while (rs.next()) {
				i=1;
				lead = new Lead();
				lead.setPK(rs.getLong(i)); i++;
				lead.setCampaignKey(camp_pk);

				lead.getAddress().setPK(rs.getLong(i)); i++;
				AddressDB addr = new AddressDB(DB_TYPE);
				addr.setConnection(getConnection());
				Address a = (Address)addr.selectRow(new Long(lead.getAddress().getPK()));
				lead.setAddress(a);

				lead.setCompany(rs.getString(i)); i++;
				lead.setInterestLevel(rs.getString(i)); i++;

				lead.getContactedBy().setPK(rs.getLong(i)); i++;
				SalesPersonDB spd = new SalesPersonDB(DB_TYPE);
				spd.setConnection(getConnection());
				SalesPerson sp = (SalesPerson)spd.selectRow(new Long(lead.getContactedBy().getPK()));
				lead.setContactedBy(sp);

				lead.setContactDate(rs.getDate(i)); i++;
				String x = rs.getString(i); i++;
				Boolean val;	
				if (x.equals("Y")) val = new Boolean(true); else val = new Boolean(false);
				lead.setConverted(val);
				lead.setModifiedBy(rs.getString(i)); i++;
				lead.setModifiedDate(rs.getDate(i)); 

				leads.add(lead);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return leads;
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table lead";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

	/**
	 * select all lead rows
	 *
	 * @param lastSyncDate a user's last sync date used to limit
	 * the query or null if no limit
	 * @return the Lead row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectAllRows(java.util.Date lastSyncDate) 
		throws SQLException {

		Lead lead;

		Statement stmt = null;
		ResultSet rs = null;
		ArrayList rows = new ArrayList();

		StringBuffer query = new StringBuffer();
		query.append(selectAllQuery);
		if (lastSyncDate != null) {
			query.append(" WHERE lead.modified_date > ");
            query.append(DateUtils.formatDateTime(DB_TYPE, lastSyncDate));
		}

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query.toString());
			rs = stmt.executeQuery(query.toString());

			int i;

			while (rs.next()) {
				i=1;
				lead = new Lead();
				lead.setPK(rs.getLong(i)); i++;
				lead.setCampaignKey(rs.getLong(i)); i++;
				lead.getAddress().setPK(rs.getLong(i)); i++;
				AddressDB addr = new AddressDB(DB_TYPE);
				addr.setConnection(getConnection());
				Address a = (Address)addr.selectRow(new Long(lead.getAddress().getPK()));
				lead.setAddress(a);
				lead.setCompany(rs.getString(i)); i++;
				lead.setInterestLevel(rs.getString(i)); i++;
				lead.getContactedBy().setPK(rs.getLong(i)); i++;
				lead.setContactDate(rs.getDate(i)); i++;
				String x = rs.getString(i); i++;
				Boolean val;	
				if (x.equals("Y")) 
					val = new Boolean(true); 
				else 
					val = new Boolean(false);
				lead.setConverted(val);
				lead.setModifiedBy(rs.getString(i)); i++;
				lead.setModifiedDate(rs.getDate(i)); 
				rows.add(lead);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}
		return rows;
	}
}
