package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;
import java.util.ArrayList;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * This class implements the DBInterface for
 * the Opportunity class objects which are stored
 * in the opportunity database table.
 */
public class OpportunityDB extends DBType implements DBInterface {
	private Connection con;

	private final static String selectOppNamesQuery = 
			"SELECT " +
 			"opportunity.name " +
			"FROM opportunity, user_group, user_group_member " +
			"WHERE " +
			"opportunity.group_name = user_group.name AND " +
			"user_group.pk = user_group_member.user_group_pk AND " +
			"user_group_member.user_pk = ";

	private final static String selectOppIndexQuery = 
			"SELECT " +
 			"opportunity.name, opportunity.stage, opportunity.pk " +
			"FROM opportunity, user_group, user_group_member " +
			"WHERE " +
			"opportunity.group_name = user_group.name AND " +
			"user_group.pk = user_group_member.user_group_pk AND " +
			"user_group_member.user_pk = ";

	private final static String selectQuery = 
			"SELECT " +
 			"name, dollar_value, probability, " +
    		"stage, lead_source, lead_type , " +
    		"description, group_name, prime_sp_pk , " +
    		"cust_pk, close_date, lead_pk, " +
    		"modified_by, modified_date " +
			"FROM opportunity " +
			"WHERE pk=";

	private final static String selectByUserQuery = 
			"SELECT " +
 			"opportunity.pk, opportunity.name, dollar_value, probability, " +
    		"stage, lead_source, lead_type , " +
    		"description, group_name, prime_sp_pk , " +
    		"cust_pk, close_date, lead_pk, " +
    		"opportunity.modified_by, opportunity.modified_date " +
			"FROM opportunity, user_group, user_group_member " +
			"WHERE " +
			"opportunity.group_name = user_group.name AND " +
			"user_group.pk = user_group_member.user_group_pk AND " +
			"user_group_member.user_pk = ";

	private final static String updateQuery = 
			"UPDATE opportunity " +
			"SET  " ;
	private final static String insertQuery = 
			"INSERT INTO opportunity VALUES ( ";
	private final static String deleteQuery = 
			"DELETE FROM opportunity WHERE pk =";

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 */
	public OpportunityDB() {
	}

	/** 
	 * construct using a db type
	 * @param dbType the db type to assume
	 */
	public OpportunityDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a constructor that accepts an existing Connection
	 * to use for future operations
	 *
	 * @param con the Connection to use
	 */
	public OpportunityDB(Connection con) {
		this.con = con;
	}

	/**
	 * get the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single opportunity row using the passed
	 * primary key
	 * 
	 * @param name description
	 * @return the row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		Opportunity opportunity = new Opportunity();
		long pkValue = ((Long)pk).longValue();
		opportunity.setPK(pkValue);

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + pkValue;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			while (rs.next()) {
				i=1;
				opportunity.setName(rs.getString(i)); i++;
				opportunity.setDollarValue(rs.getInt(i)); i++;
				opportunity.setProbability(rs.getString(i)); i++;
				opportunity.setStage(rs.getString(i)); i++;
				opportunity.setLeadSource(rs.getString(i)); i++;
				opportunity.setLeadType(rs.getString(i)); i++;
				opportunity.setDesc(rs.getString(i)); i++;
				opportunity.setGroupName(rs.getString(i)); i++;
				opportunity.setPrimeSalesPersonPK(rs.getLong(i)); i++;
				opportunity.setCustomerPK(rs.getLong(i)); i++;
				opportunity.setCloseDate(rs.getDate(i)); i++;
				opportunity.setLeadPK(rs.getLong(i)); i++;
				opportunity.setModifiedBy(rs.getString(i)); i++;
				opportunity.setModifiedDate(rs.getDate(i)); 

				//get the customer for this opp
				CustomerDB customerDB = new CustomerDB(DB_TYPE);
				customerDB.setConnection(getConnection());
				Customer customer=null;
				customer = (Customer)customerDB.selectRow(new Long(opportunity.getCustomerPK()));
				opportunity.setCustomer(customer);

				//get all the activities for this opp and add them
				ActivityDB activityDB = new ActivityDB(DB_TYPE);
				activityDB.setConnection(getConnection());
				activityDB.selectByOppRows(pkValue, opportunity.getActivities());
				//get all the quotes for this opp and add them
				QuoteDB quoteDB = new QuoteDB(DB_TYPE);
				quoteDB.setConnection(getConnection());
				quoteDB.selectByOppRow(pkValue, opportunity.getQuotes());

				//get all the forecasts for this opp and add them
				ForecastDB forecastDB = new ForecastDB(DB_TYPE);
				forecastDB.setConnection(getConnection());
				forecastDB.selectByOppRow(pkValue, opportunity.getForecasts());

				//get all the orders for this opp and add them 
				OrderDB orderDB = new OrderDB(DB_TYPE);
				orderDB.setConnection(getConnection());
				orderDB.selectByOppRow(pkValue, opportunity.getOrders());

				//get all the contacts for this opp and add them
				ContactDB contactDB = new ContactDB(DB_TYPE);
				contactDB.setConnection(getConnection());
				contactDB.selectByOppRow(pkValue, opportunity.getContacts());
			}
		} catch (SQLException e) {
			LogWrite.write(e);
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return opportunity;
	}


	/**
	 * a shallow update of opportunity
	 *
	 * @param name description
	 * @exception java.sql.SQLException 
	 */
	public void updateRow(Object obj) 
		throws SQLException {

		Opportunity opportunity = (Opportunity)obj;
		long pk = opportunity.getPK();

		Statement stmt = null;
		StringBuffer query = new StringBuffer(updateQuery);

		try {
			stmt = con.createStatement();

			query.append("name=");
			query.append(JDBC.quoteMore(opportunity.getName()));
			query.append("dollar_value=");
			query.append(opportunity.getDollarValue()).append(",");
			query.append("probability=");
			query.append(JDBC.quoteMore(opportunity.getProbability()));
			query.append("stage=");
			query.append(JDBC.quoteMore(opportunity.getStage()));
			query.append("lead_source=");
			query.append(JDBC.quoteMore(opportunity.getLeadSource()));
			query.append("lead_type=");
			query.append(JDBC.quoteMore(opportunity.getLeadType()));
			query.append("description=");
			query.append(JDBC.quoteMore(opportunity.getDesc()));
			query.append("group_name=");
			query.append(JDBC.quoteMore(opportunity.getGroupName()));
			query.append("prime_sp_pk=");
			if (opportunity.getPrimeSalesPersonPK() == 0) {
				query.append("null,");
			} else {
				query.append(opportunity.getPrimeSalesPersonPK()).append(",");
			}
			query.append("cust_pk=");
			if (opportunity.getCustomerPK() == 0) {
				query.append("null,");
			} else {
				query.append(opportunity.getCustomerPK()).append(",");
			}

			query.append("close_date=");
			query.append(JDBC.quoteMore(DateUtils.format(DB_TYPE, opportunity.getCloseDate())));

			query.append("lead_pk=");
			if (opportunity.getLeadPK() == 0) {
				query.append("null,");
			} else {
				query.append(opportunity.getLeadPK()).append(",");
			}

			query.append("modified_by=");
			query.append(JDBC.quoteMore(opportunity.getModifiedBy()));
			query.append("modified_date=");
			if (DB_TYPE == Prefs.MYSQL)
				query.append("CURRENT_DATE");
			else
				query.append("SYSDATE");
			query.append(" WHERE pk=");
			query.append(opportunity.getPK());	

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int updatedRows = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close(); 
			} catch (SQLException x) { }
		}
	}

	/**
	 * insert a new opportunity row using the passed
	 * Opportunity object as the column values.
	 * 
	 * @param obj the Opportunity to load or add
	 * @param load true if the Opportunity is to be loaded or false
	 * if its to be added
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public long insertRow(Object obj, boolean load) 
		throws SQLException {

		Opportunity opportunity = (Opportunity)obj;

		if (!load)
			opportunity.setPK(DBUtils.generatePK());

		Statement stmt = null;
		StringBuffer query = new StringBuffer(insertQuery);

		try {
			stmt = con.createStatement();

			query.append(opportunity.getPK()).append(",");
			query.append(JDBC.quoteMore(opportunity.getName()));
			query.append(opportunity.getDollarValue()).append(",");
			query.append(JDBC.quoteMore(opportunity.getProbability()));
			query.append(JDBC.quoteMore(opportunity.getStage()));
			query.append(JDBC.quoteMore(opportunity.getLeadSource()));
			query.append(JDBC.quoteMore(opportunity.getLeadType()));
			query.append(JDBC.quoteMore(opportunity.getDesc()));
			query.append(JDBC.quoteMore(opportunity.getGroupName()));
			query.append(opportunity.getPrimeSalesPerson().getPK()).append(",");
			query.append(opportunity.getCustomer().getPK()).append(",");
			query.append(JDBC.quoteMore(DateUtils.format(DB_TYPE, opportunity.getCloseDate())));
			if (opportunity.getLead() == null) {
				query.append("null,");
			} else {
				query.append(opportunity.getLead().getPK()).append(",");
			}
			query.append(JDBC.quoteMore(opportunity.getModifiedBy()));
			if (DB_TYPE == Prefs.MYSQL)
				query.append("CURRENT_DATE");
			else
				query.append("SYSDATE");
			query.append(")");

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int rc = stmt.executeUpdate(query.toString());
		} catch (NullPointerException x) {
			LogWrite.write(x);
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}

		return opportunity.getPK();
	}

	/**
	 * delete a single opportunity row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();
		String query = deleteQuery + pkValue;

		//delete all the children first 

		//forecasts
		ForecastDB f = new ForecastDB(DB_TYPE);
		f.setConnection(getConnection());
		f.deleteOpportunityRows((Long)obj);

		if (Prefs.DEBUG) LogWrite.write("deleted Forecast rows of Opp");

		//contacts
		ContactDB c = new ContactDB(DB_TYPE);
		c.setConnection(getConnection());
		c.deleteOpportunityRows((Long)obj);

		if (Prefs.DEBUG) LogWrite.write("deleted contact rows of Opp");

		//quotes
		QuoteDB q = new QuoteDB(DB_TYPE);
		q.setConnection(getConnection());
		q.deleteOpportunityRows((Long)obj);

		if (Prefs.DEBUG) LogWrite.write("deleted quote rows of Opp");

		//orders
		OrderDB o = new OrderDB(DB_TYPE);
		o.setConnection(getConnection());
		o.deleteOpportunityRows((Long)obj);

		if (Prefs.DEBUG) LogWrite.write("deleted order rows of Opp");

		//activities
		ActivityDB a = new ActivityDB(DB_TYPE);
		a.setConnection(getConnection());
		a.deleteOpportunityRows((Long)obj);

		if (Prefs.DEBUG) LogWrite.write("deleted activity rows of Opp");

		Statement stmt = null;
		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			LogWrite.write(e);
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) { }		
		}
	}

	/**
	 * select all opportunity "index" rows using the passed
	 * key
	 * 
	 * @param name description
	 * @return the row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectOppIndexRows(SalesPerson sp) 
		throws SQLException {

		ArrayList rows = new ArrayList();
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectOppIndexQuery + sp.getPK() +
			" ORDER BY opportunity.name";

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			OppIndex oi = null;

			while (rs.next()) {
				i=1;
				oi = new OppIndex();
				oi.setName(rs.getString(i)); i++;
				oi.setStage(rs.getString(i)); i++;
				oi.setPK(rs.getLong(i)); i++;
				rows.add(oi);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return rows;
	}

	/**
	 * select all opportunity names using the passed
	 * sales person pk
	 * 
	 * @param name description
	 * @return the row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectNames(long user_pk) 
		throws SQLException {

		ArrayList names = new ArrayList();
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectOppNamesQuery + user_pk + 
		" ORDER BY opportunity.name";

		if (Prefs.DEBUG) LogWrite.write(query);
		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			while (rs.next()) {
				names.add(rs.getString(1));
				if (Prefs.DEBUG) LogWrite.write("found an opp for this sp");
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return names;
	}

	/**
	 * select all opportunity rows using the passed
	 * user primary key
	 * 
	 * @param user_pk the user's primary key to limit the query with
	 * @param lastSyncDate the user's last sync date used to limit the
	 * query or null if no limit
	 * @return an ArrayList of row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectByUserRows(long user_pk, java.util.Date lastSyncDate) 
		throws SQLException {

		ArrayList opps = new ArrayList();
		Opportunity opportunity=null;

		Statement stmt = null;
		ResultSet rs = null;
		StringBuffer query = new StringBuffer();
		query.append(selectByUserQuery);
		query.append(user_pk);

		if (lastSyncDate != null) {
			query.append(" AND opportunity.modified_date > ");
            query.append(DateUtils.formatDateTime(DB_TYPE, lastSyncDate));
		}

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query.toString());
			rs = stmt.executeQuery(query.toString());

			int i;
			Customer cust;
			Lead lead;
			SalesPerson sp;
			LeadDB leadDB = new LeadDB(DB_TYPE);
			leadDB.setConnection(getConnection());
			CustomerDB custDB = new CustomerDB(DB_TYPE);
			custDB.setConnection(getConnection());
			SalesPersonDB spDB = new SalesPersonDB(DB_TYPE);
			spDB.setConnection(getConnection());

			while (rs.next()) {
				i=1;
				opportunity = new Opportunity();
				opportunity.setPK(rs.getLong(i)); i++;
				opportunity.setName(rs.getString(i)); i++;
				opportunity.setDollarValue(rs.getInt(i)); i++;
				opportunity.setProbability(rs.getString(i)); i++;
				opportunity.setStage(rs.getString(i)); i++;
				opportunity.setLeadSource(rs.getString(i)); i++;
				opportunity.setLeadType(rs.getString(i)); i++;
				opportunity.setDesc(rs.getString(i)); i++;
				opportunity.setGroupName(rs.getString(i)); i++;
				opportunity.setPrimeSalesPersonPK(rs.getLong(i)); i++;
				sp = (SalesPerson)spDB.selectRow(new Long(opportunity.getPrimeSalesPersonPK()));
				opportunity.setPrimeSalesPerson(sp);
				opportunity.setCustomerPK(rs.getLong(i)); i++;
				cust = (Customer)custDB.selectRow(new Long(opportunity.getCustomerPK()));
				opportunity.setCustomer(cust);
				opportunity.setCloseDate(rs.getDate(i)); i++;
				opportunity.setLeadPK(rs.getLong(i)); i++;
				lead = (Lead)leadDB.selectRow(new Long(opportunity.getLeadPK()));
				opportunity.setLead(lead);
				opportunity.setModifiedBy(rs.getString(i)); i++;
				opportunity.setModifiedDate(rs.getDate(i)); 
				opps.add(opportunity);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return opps;
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table opportunity";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

}
