package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;
import java.util.ArrayList;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the DBInterface for
 * the Order class objects which are stored
 * in the order database table.
 */
public class OrderDB extends DBType implements DBInterface {
	private Connection con;
		
	private final static String deleteOppQuery =
		"DELETE FROM sellwin_order WHERE opp_pk = ";
	private final static String deleteQuery =
		"DELETE FROM sellwin_order WHERE pk = ";
	private final static String insertQuery =
		"INSERT INTO sellwin_order " +
		"VALUES (";
	private final static String updateQuery =
		"UPDATE sellwin_order SET " ;

	private final static String selectQuery =
			"SELECT opp_pk, "+
			"order_number, quote_pk, ship_address_pk, "+
			"bill_address_pk, status, payment_form, "+
  			"modified_by, modified_date " +
			"FROM sellwin_order "+
			"WHERE pk = ";
	private final static String selectByOppQuery =
			"SELECT pk, "+
			"order_number, quote_pk, ship_address_pk, "+
			"bill_address_pk, status, payment_form, "+
  			"modified_by, modified_date " +
			"FROM sellwin_order "+
			"WHERE opp_pk = ";

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 *
	 */
	public OrderDB() {
	}

	/**
	 * construct, specifying a db type
	 * @param dbType the db type to assume
	 */
	public OrderDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a version of the constructor when a connection
	 * is already obtained from somewhere else
	 *
	 * @param con the Connection to use 
	 */
	public OrderDB(Connection con) {
		this.con = con;
	}

	/**
	 * return the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public final void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single order row using the passed
	 * primary key
	 * 
	 * @param name description
	 * @return the Order row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		Order order = new Order();
		order.setPK(((Long)pk).longValue());

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + order.getPK();

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;

			while (rs.next()) {
				i=1;
				order.setOppKey(rs.getLong(i)); i++;
				order.setOrderNumber(new Long(rs.getLong(i))); i++;
				order.setQuoteKey(rs.getLong(i)); i++;
				AddressDB adb = new AddressDB(DB_TYPE);
				adb.setConnection(getConnection());
				order.setBillAddressKey(rs.getLong(i)); i++;
				order.setBillAddress((Address)adb.selectRow(new Long(order.getBillAddressKey())));
				order.setShipAddressKey(rs.getLong(i)); i++;
				order.setShipAddress((Address)adb.selectRow(new Long(order.getShipAddressKey())));
				order.setStatus(rs.getString(i)); i++;
				order.setPaymentForm(rs.getString(i)); i++;
				order.setModifiedBy(rs.getString(i)); i++;
				order.setModifiedDate(rs.getDate(i)); 

				QuoteDB quoteDB = new QuoteDB(DB_TYPE);
				quoteDB.setConnection(getConnection());
				Quote quote = (Quote)quoteDB.selectRow(new Long(order.getQuoteKey()));
				order.setQuote(quote);
			}

		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return order;
	}


	/**
	 * not yet coded
	 * @param name description
	 * @exception java.sql.SQLException 
	 */
	public final void updateRow(Object obj) 
		throws SQLException {

		Order order = (Order)obj;
		Statement stmt=null;
		StringBuffer query = new StringBuffer(updateQuery);

		try {
			
			stmt = con.createStatement();

			query.append("status=");
			query.append(JDBC.quoteMore(order.getStatus()));
			query.append("payment_form=");
			query.append(JDBC.quoteMore(order.getPaymentForm()));
			query.append("sub_total=");
			query.append(order.getSubTotal()).append(",");
			query.append("tax=");
			query.append(order.getTax()).append(",");
			query.append("shipping=");
			query.append(order.getShipping()).append(",");
			query.append("total=");
			query.append(order.getTotal()).append(",");
			query.append("modified_by=");
			query.append(JDBC.quoteMore(order.getModifiedBy()));
			query.append("modified_date=");
			if (DB_TYPE == Prefs.MYSQL)
				query.append("CURRENT_DATE");
			else
				query.append("SYSDATE");
			query.append(" WHERE pk=");
			query.append(order.getPK());

			if (Prefs.DEBUG) LogWrite.write(query.toString());
			int updatedRows = stmt.executeUpdate(query.toString());
		} catch (SQLException e) {
			throw e;
		} finally {
			try { if (stmt != null) stmt.close();
			} catch (SQLException x) {}
		}
	}

	/**
	 * insert a new order row using the passed
	 * Order object as the column values.
	 * 
	 * @param obj the Order to add or load
	 * @param load true if we are to load or false if we are to add
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public final long insertRow(Object obj, boolean load) 
		throws SQLException {

		Order order = (Order)obj;

		if (!load)
			order.setPK(DBUtils.generatePK());

		StringBuffer query = new StringBuffer(insertQuery);
		Statement stmt = con.createStatement();

		try {
			query.append(order.getPK()).append(",");
			query.append(order.getOppKey()).append(",");
			query.append(order.getOrderNumber().longValue()).append(",");
			query.append(order.getQuoteKey()).append(",");
			query.append(order.getShipAddress().getPK()).append(",");
			query.append(order.getBillAddress().getPK()).append(",");
			query.append(JDBC.quoteMore(order.getStatus()));
			query.append(JDBC.quoteMore(order.getPaymentForm()));
			query.append(order.getSubTotal()).append(",");
			query.append(order.getTax()).append(",");
			query.append(order.getShipping()).append(",");
			query.append(order.getTotal()).append(",");

			query.append(JDBC.quoteMore(order.getModifiedBy()));
			if (DB_TYPE == Prefs.MYSQL)
				query.append("CURRENT_DATE");
			else
				query.append("SYSDATE");
			query.append(")");
		} catch (NullPointerException e) {
			LogWrite.write(e);
		}

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int rc = stmt.executeUpdate(query.toString());

		return order.getPK();
	}

	/**
	 * delete a single order row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();

		String query = deleteQuery + pkValue;

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * delete all order rows using the passed
	 * opportunity primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteOpportunityRows(Object obj) 
		throws SQLException {

		long opp_pk = ((Long)obj).longValue();

		String query = deleteQuery + opp_pk;

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * select all order rows using the passed
	 * opportunity primary key
	 * 
	 * @param name description
	 * @exception java.sql.SQLException 
	 */
	public final void selectByOppRow(long opp_pk, ArrayList orders) 
		throws SQLException {

		Order order = null;

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectByOppQuery + opp_pk;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			QuoteDB quoteDB = new QuoteDB(DB_TYPE);
			quoteDB.setConnection(getConnection());
			Quote quote;

			while (rs.next()) {
				i=1;
				order = new Order();
				order.setPK(rs.getLong(i)); i++;
				order.setOrderNumber(new Long(rs.getLong(i))); i++;
				order.setQuoteKey(rs.getLong(i)); i++;
				AddressDB adb = new AddressDB(DB_TYPE);
				adb.setConnection(getConnection());
				order.setBillAddressKey(rs.getLong(i)); i++;
				order.setBillAddress((Address)adb.selectRow(new Long(order.getBillAddressKey())));
				order.setShipAddressKey(rs.getLong(i)); i++;
				order.setShipAddress((Address)adb.selectRow(new Long(order.getShipAddressKey())));
				order.setStatus(rs.getString(i)); i++;
				order.setPaymentForm(rs.getString(i)); i++;
				order.setModifiedBy(rs.getString(i)); i++;
				order.setModifiedDate(rs.getDate(i)); 

				//go ahead and get the order's quote information
				quote = (Quote)quoteDB.selectRow(new Long(order.getQuoteKey()));
				order.setQuote(quote);

				orders.add(order);
			}

		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table sellwin_order";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

}
