package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;
import java.util.ArrayList;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the DBInterface for
 * the Quote class objects which are stored
 * in the quote database table.
 */
public class QuoteDB extends DBType implements DBInterface {
	private Connection con;

	private final static String deleteOppQuery =
		"DELETE FROM quote WHERE opp_pk=";
	private final static String deleteQuery =
		"DELETE FROM quote WHERE pk = ";
	private final static String insertQuery =
		"INSERT INTO quote " +
		"VALUES (";
	private final static String updateQuery = 
			"UPDATE quote " +
			"SET ";
	private final static String selectQuery =
			"SELECT opp_pk, "+
			"name, "+
  			"modified_by, modified_date " +
			"FROM quote "+
			"WHERE pk = ";
	private final static String selectByOppQuery =
			"SELECT pk, "+
			"name, "+
  			"modified_by, modified_date " +
			"FROM quote "+
			"WHERE opp_pk = ";

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 */
	public QuoteDB() {
	}

	/**
	 * construct, use a particular db type
	 * @param dbType the db type to use
	 */
	public QuoteDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a version of the constructor when a connection
	 * is already obtained from somewhere else
	 *
	 * @param con the Connection to use 
	 */
	public QuoteDB(Connection con) {
		this.con = con;
	}

	/**
	 * return the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public final void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single quote row using the passed
	 * primary key
	 * 
	 * @param name description
	 * @return the Quote row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		Quote quote = new Quote();
		quote.setPK(((Long)pk).longValue());

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + quote.getPK();

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;

			while (rs.next()) {
				i=1;
				quote.setOppPK(rs.getLong(i)); i++;
				quote.setName(rs.getString(i)); i++;
				quote.setModifiedBy(rs.getString(i)); i++;
				quote.setModifiedDate(rs.getDate(i)); 

				QuoteLineDB quoteLineDB = new QuoteLineDB(DB_TYPE);
				quoteLineDB.setConnection(getConnection());
				quoteLineDB.selectByQuoteRow(quote.getPK(), quote.getLines());	
			}

		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return quote;
	}


	/**
	 * update a single quote row using the passed
	 * Quote object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param name description
	 * @exception java.sql.SQLException 
	 */
	public final void updateRow(Object obj) 
		throws SQLException {

		Quote quote = (Quote)obj;

		StringBuffer query = new StringBuffer(updateQuery);
		Statement stmt = con.createStatement();

		query.append("name=");
		query.append(JDBC.quoteMore(quote.getName()));
		query.append("modified_by=");
		query.append(JDBC.quoteMore(quote.getModifiedBy()));
		query.append("modified_date=");
		if (DB_TYPE == Prefs.MYSQL)
			query.append("CURRENT_DATE");
		else
			query.append("SYSDATE");
		query.append(" WHERE pk=");
		query.append(quote.getPK()); 

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int updatedRows = stmt.executeUpdate(query.toString());
	}

	/**
	 * insert a new quote row using the passed
	 * Quote object as the column values.
	 * 
	 * @param obj Quote we are loading or adding
	 * @param load true if we are to load, false if adding
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public final long insertRow(Object obj, boolean load) 
		throws SQLException {

		Quote quote = (Quote)obj;

		if (!load)
			quote.setPK(DBUtils.generatePK());

		StringBuffer query = new StringBuffer(insertQuery);
		Statement stmt = con.createStatement();

		query.append(quote.getPK()).append(","); 
		query.append(quote.getOppPK()).append(","); 
		query.append(JDBC.quoteMore(quote.getName()));
		query.append(JDBC.quoteMore(quote.getModifiedBy()));
		if (DB_TYPE == Prefs.MYSQL)
			query.append("CURRENT_DATE");
		else
			query.append("SYSDATE");
		query.append(")");

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int rc = stmt.executeUpdate(query.toString());

		return quote.getPK();
	}

	/**
	 * delete a single quote row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();

		String query = deleteQuery + pkValue;

		Statement stmt = null;

		try {

			//delete the quote lines for this quote first
			QuoteLineDB qlDB = new QuoteLineDB(DB_TYPE);
			qlDB.setConnection(getConnection());
			qlDB.deleteQuoteRows(pkValue);

			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			LogWrite.write(e);
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * delete all quote rows using the passed
	 * opportunity primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteOpportunityRows(Object obj) 
		throws SQLException {

		long oppPK = ((Long)obj).longValue();

		String query = deleteOppQuery + oppPK;

		Statement stmt = null;

		try {
			//delete the quote lines for this quote first
			QuoteLineDB qlDB = new QuoteLineDB(DB_TYPE);
			qlDB.setConnection(getConnection());
			qlDB.deleteQuoteRowsByOpp(oppPK);

			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			LogWrite.write(e);
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * select all quote rows using the passed
	 * opportunity primary key
	 * 
	 * @param oppPK the opportunity pk we are querying with
	 * @param quoteList the list we are going to add the quotes to
	 * @exception java.sql.SQLException 
	 */
	public final void selectByOppRow(long opp_pk, ArrayList quotes) 
		throws SQLException {

		Quote q = null;

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectByOppQuery + opp_pk;

		if (Prefs.DEBUG) LogWrite.write(query);

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			QuoteLineDB quoteLineDB = new QuoteLineDB(DB_TYPE);
			quoteLineDB.setConnection(getConnection());

			while (rs.next()) {
				i=1;
				q = new Quote();
				q.setPK(rs.getLong(i)); i++;
				q.setOppPK(opp_pk);
				q.setName(rs.getString(i)); i++;
				q.setModifiedBy(rs.getString(i)); i++;
				q.setModifiedDate(rs.getDate(i)); 
				quoteLineDB.selectByQuoteRow(q.getPK(), q.getLines());
				quotes.add(q);
			}
			if (Prefs.DEBUG) LogWrite.write("found " + quotes.size() + " for opppk=" + opp_pk);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table quote";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

}
