package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.util.ArrayList;
import java.sql.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the DBInterface for
 * the QuoteLine class objects which are stored
 * in the quote_line database table.
 */
public class QuoteLineDB extends DBType implements DBInterface {

	private Connection con;
	private final static String deleteQuery =
		"DELETE FROM quote_line WHERE pk = ";
		
	private final static String deleteByOppQuery =
		"DELETE FROM quote_line WHERE quote_pk in "+
		"(select quote_pk from opportunity,quote "+
			"where opportunity.pk = quote.opp_pk AND opportunity.pk=";

	private final static String deleteByQuoteQuery =
		"DELETE FROM quote_line WHERE quote_pk=";

	private final static String insertQuery =
		"INSERT INTO quote_line (" +
		" PK, QUOTE_PK, PRODUCT_PK, PRODUCT_NAME, "+
		" PRODUCT_MODEL_NO, QUANTITY, COST, UNIT_PRICE, "+
		" TOTAL_PRICE, PRODUCT_DESC, MODIFIED_BY, MODIFIED_DATE " +
		") VALUES (";

	private final static String updateQuery = 
			"UPDATE quote_line " +
			"SET ";

	private final static String selectQuery =
			"SELECT quote_pk, product_pk,"+
			"quantity, "+
			"total_price, "+
  			"modified_by, modified_date " +
			"FROM quote_line "+
			"WHERE pk = ";

	private final static String selectByQuoteQuery =
			"SELECT pk, product_pk,"+
			"quantity, "+
			"total_price, "+
  			"modified_by, modified_date " +
			"FROM quote_line "+
			"WHERE quote_pk = ";

	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 */
	public QuoteLineDB() {
	}
	
	/**
	 * construct, use a particular db type
	 * @param dbType the db type to use
	 */
	public QuoteLineDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a version of the constructor when a connection
	 * is already obtained from somewhere else
	 *
	 * @param con the Connection to use 
	 */
	public QuoteLineDB(Connection con) {
		this.con = con;
	}

	/**
	 * return the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public final void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single quote_line row using the passed
	 * primary key
	 * 
	 * @param name description
	 * @return the QuoteLine row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		QuoteLine line = new QuoteLine();
		line.setPK(((Long)pk).longValue());

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + line.getPK();

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;

			ProductDB prodDB = new ProductDB(DB_TYPE);
			prodDB.setConnection(getConnection());

			while (rs.next()) {
				i=1;
				line.setQuotePK(rs.getLong(i)); i++;
				line.getProduct().setPK(rs.getLong(i)); i++;
				line.setProduct((Product)prodDB.selectRow(new Long(line.getProduct().getPK())));
				line.setQuantity(new Integer(rs.getInt(i))); i++;
				line.setTotalPrice(new Double(rs.getDouble(i))); i++;
				line.setModifiedBy(rs.getString(i)); i++;
				line.setModifiedDate(rs.getDate(i)); 
			}

		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return line;
	}


	/**
	 * update a single quote_line row using the passed
	 * QuoteLine object's attributes.  All columns
	 * get updated by this routine regardless of whether
	 * an attribute was modified or not.
	 *
	 * @param name description
	 * @exception java.sql.SQLException 
	 */
	public final void updateRow(Object obj) 
		throws SQLException {

		QuoteLine quote_line = (QuoteLine)obj;

		StringBuffer query = new StringBuffer(updateQuery);
		Statement stmt = con.createStatement();

		query.append("quantity=");
		query.append(quote_line.getQuantity().intValue()).append(","); 
		query.append("modified_by=");
		query.append(JDBC.quoteMore(quote_line.getModifiedBy()));
		query.append("modified_date=");
		if (DB_TYPE == Prefs.MYSQL)
			query.append("CURRENT_DATE");
		else
			query.append("SYSDATE");
		query.append(" WHERE pk=");
		query.append(quote_line.getPK()); 
	
		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int updatedRows = stmt.executeUpdate(query.toString());
	}

	/**
	 * insert a new quote_line row using the passed
	 * QuoteLine object as the column values.
	 * 
	 * @param obj QuoteLine we are loading or adding
	 * @param load true if we are loading, false if adding
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public final long insertRow(Object obj, boolean load) 
		throws SQLException {

		QuoteLine quote_line = (QuoteLine)obj;

		if (!load)
			quote_line.setPK(DBUtils.generatePK());

		StringBuffer query = new StringBuffer(insertQuery);
		Statement stmt = con.createStatement();

		query.append(quote_line.getPK()).append(","); 
		query.append(quote_line.getQuotePK()).append(","); 
		query.append(quote_line.getProduct().getPK()).append(","); 
		query.append(JDBC.quoteMore(quote_line.getProduct().getName()));
		query.append(JDBC.quoteMore(quote_line.getProduct().getModelNo()));
		query.append(quote_line.getQuantity().intValue()).append(","); 
		query.append(quote_line.getProduct().getCost().doubleValue()).append(","); 
		query.append(quote_line.getProduct().getPrice().doubleValue()).append(","); 
		query.append(quote_line.getTotalPrice().doubleValue()).append(","); 
		query.append(JDBC.quoteMore(quote_line.getProduct().getDesc()));
		query.append(JDBC.quoteMore(quote_line.getModifiedBy()));
		if (DB_TYPE == Prefs.MYSQL)
			query.append("CURRENT_DATE");
		else
			query.append("SYSDATE");
		query.append(")");

		if (Prefs.DEBUG) LogWrite.write(query.toString());
		int rc = stmt.executeUpdate(query.toString());

		return quote_line.getPK();
	}

	/**
	 * delete a single quote_line row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();

		String query = deleteQuery + pkValue;

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * select a single quote_line row using the passed
	 * primary key
	 * 
	 * @param name description
	 * @return the QuoteLine row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final void selectByQuoteRow(long quote_pk, ArrayList lines) 
		throws SQLException {

		QuoteLine ql = null;

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectByQuoteQuery + quote_pk;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;
			ProductDB prodDB = new ProductDB(DB_TYPE);
			prodDB.setConnection(getConnection());
			long productPK;

			while (rs.next()) {
				i=1;
				ql = new QuoteLine();
				ql.setPK(rs.getLong(i)); i++;
				ql.setQuotePK(quote_pk);
				productPK = rs.getLong(i); i++; 
				ql.setProduct((Product)prodDB.selectRow(new Long(productPK)));
				ql.getProduct().setPK(productPK);
				ql.setQuantity(new Integer(rs.getInt(i))); i++;
				ql.setTotalPrice(new Double(rs.getDouble(i))); i++;
				ql.setModifiedBy(rs.getString(i)); i++;
				ql.setModifiedDate(rs.getDate(i)); 
				lines.add(ql);
			}

		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}
	}

	/**
	 * delete all quote_line rows using the passed
	 * quote_pk primary key value
	 *
	 * @param ojb quote primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteQuoteRows(long quote_pk) 
		throws SQLException {

		String query = deleteByQuoteQuery + quote_pk;

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			LogWrite.write(e);
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * delete all quote_line rows using the passed
	 * opp_pk primary key value
	 *
	 * @param ojb opportunity primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteQuoteRowsByOpp(long opp_pk) 
		throws SQLException {

		String query = deleteByOppQuery + opp_pk + ")";

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			LogWrite.write(e);
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table quote_line";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

}
