package sellwin.db;

import sellwin.domain.*;
import sellwin.utils.*;

import java.sql.*;
import java.util.ArrayList;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the DBInterface for
 * the UserGroupMember class objects which are stored
 * in the user_group_member database table.
 */
public class UserGroupMemberDB extends DBType implements DBInterface {
	private Connection con;

	private final static String deleteUserQuery =
		"DELETE FROM user_group_member WHERE user_pk = ";

	private final static String deleteQuery =
		"DELETE FROM user_group_member WHERE pk = ";

	private final static String deleteGroupQuery =
		"DELETE FROM user_group_member WHERE user_group_pk=";

	private final static String insertQuery =
		"INSERT INTO user_group_member " +
		"VALUES (";

	private final static String selectQuery =
			"SELECT user_group_pk, user_pk, "+
  			"modified_by, modified_date " +
			"FROM user_group_member "+
			"WHERE pk = ";

	private final static String selectUsersQuery =
			"SELECT user_pk, "+
			"FROM user_group_member "+
			"WHERE user_group_pk = ";

	private final static String selectGroupNamesQuery =
			"SELECT user_group.name " +
			"FROM user_group, user_group_member "+
			"WHERE user_group_member.user_group_pk = user_group.pk AND "+
			"user_group_member.user_pk = ";

	private final static String selectGroupsForUserQuery =
			"SELECT user_group.pk " +
			"FROM user_group, user_group_member "+
			"WHERE user_group_member.user_group_pk = user_group.pk AND "+
			"user_group_member.user_pk = ";

	private final static String selectAllQuery =
			"SELECT pk, user_group_pk, user_pk, "+
			"modified_by, modified_date FROM user_group_member ";


	/**
	 * a do-nothing constructor but necessary to
	 * do the operations offered by this class
	 */
	public UserGroupMemberDB() {
	}

	/**
	 * construct using a db type
	 * @param dbType the db type to assume
	 */
	public UserGroupMemberDB(int dbType) {
		DB_TYPE = dbType;
	}

	/**
	 * a version of the constructor when a connection
	 * is already obtained from somewhere else
	 *
	 * @param con the Connection to use 
	 */
	public UserGroupMemberDB(Connection con) {
		this.con = con;
	}

	/**
	 * return the Connection in use
	 *
	 * @return the Connection in use
	 */
	public Connection getConnection() {
		return this.con;
	}

	/**
	 * set the Connection to use 
	 *
	 * @param con the Connection to use for any future IO's
	 */
	public final void setConnection(Connection con) 
		throws SQLException {

		this.con = con;
	}

	/**
	 * select a single user_group_member row using the passed
	 * primary key
	 * 
	 * @param pk the primary key we are searching with
	 * @return the UserGroupMember row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final Object selectRow(Object pk) 
		throws SQLException {

		UserGroupMember ug = new UserGroupMember();
		ug.setPK(((Long)pk).longValue());

		Statement stmt = null;
		ResultSet rs = null;
		String query = selectQuery + ug.getPK();

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);

			int i;

			while (rs.next()) {
				i=1;
				ug.setUserGroupPK(rs.getLong(i)); i++;
				ug.setUserPK(rs.getLong(i)); i++;
				SalesPersonDB spd = new SalesPersonDB(DB_TYPE);
				spd.setConnection(getConnection());
				SalesPerson sp = (SalesPerson)spd.selectRow(new Long(ug.getUserPK()));
				ug.setUser(sp);
				ug.setModifiedBy(rs.getString(i)); i++;
				ug.setModifiedDate(rs.getDate(i)); 
			}

		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return ug;
	}


	/**
	 * currently not necessary
	 *
	 * @param name description
	 * @exception java.sql.SQLException 
	 */
	public final void updateRow(Object obj) 
		throws SQLException {

		//currently not needed
	}

	/**
	 * insert a new user_group_member row using the passed
	 * UserGroupMember object as the column values.
	 * 
	 * @param obj the object we are inserting
	 * @param load true if loading, false if adding
	 * @return the newly assigned primary key of the new row
	 * @exception java.sql.SQLException 
	 */
	public final long insertRow(Object obj, boolean load) 
		throws SQLException {

		UserGroupMember user_group_member = (UserGroupMember)obj;

		if (!load)
			user_group_member.setPK(DBUtils.generatePK());

		StringBuffer query = new StringBuffer(insertQuery);
		Statement stmt = con.createStatement();

		query.append(user_group_member.getPK()).append(",");
		query.append(user_group_member.getUserGroupPK()).append(",");
		query.append(user_group_member.getUserPK()).append(",");
		query.append(JDBC.quoteMore(user_group_member.getModifiedBy()));
		if (DB_TYPE == Prefs.MYSQL)
			query.append("CURRENT_DATE");
		else
			query.append("SYSDATE");
		query.append(")");

		if (Prefs.DEBUG) LogWrite.write(query.toString());	
		int rc = stmt.executeUpdate(query.toString());
		
		return user_group_member.getPK();
	}

	/**
	 * delete a single user_group_member row using the passed
	 * primary key value
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteRow(Object obj) 
		throws SQLException {

		long pkValue = ((Long)obj).longValue();

		String query = deleteQuery + pkValue;

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * delete a single user_group_member row using the passed
	 * user pk 
	 *
	 * @param ojb primary key stored in a Long
	 * @exception java.sql.SQLException 
	 */
	public final void deleteByUserRow(long u_pk) 
		throws SQLException {

		String query = deleteUserQuery + u_pk;

		Statement stmt = null;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			stmt.executeUpdate(query);
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { }
		}
	}

	/**
	 * select all sales persons in a group using the passed
	 * group key
	 * 
	 * @param ug_pk the UserGroup primary key we are searching with 
	 * @return the UserGroupMember row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectUsersInGroup(long ug_pk) 
		throws SQLException {

		ArrayList users = new ArrayList();
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectUsersQuery + ug_pk;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);
			SalesPersonDB spDB = new SalesPersonDB(DB_TYPE);
			spDB.setConnection(getConnection());
			long sp_pk;
			SalesPerson sp;

			while (rs.next()) {
				sp_pk = rs.getLong(1);
				sp = (SalesPerson)(spDB.selectRow(new Long(sp_pk)));	
				users.add(sp);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return users;
	}

	/**
	 * select all group names for a given user
	 * 
	 * @param user_pk the User primary key we are searching with
	 * @return the UserGroupMember row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectGroupNamesForUser(long user_pk) 
		throws SQLException {

		ArrayList groupNames = new ArrayList();
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectGroupNamesQuery + user_pk + 
		" ORDER BY user_group.name ";

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);
			String groupName;

			while (rs.next()) {
				groupName = rs.getString(1);
				groupNames.add(groupName);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return groupNames;
	}

	/**
	 * select all groups for a given user
	 * 
	 * @param user_pk the User primary key we are searching with
	 * @return the UserGroup row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectGroupsForUser(long user_pk) 
		throws SQLException {

		ArrayList groups = new ArrayList();
		Statement stmt = null;
		ResultSet rs = null;
		String query = selectGroupsForUserQuery + user_pk ;

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query);
			rs = stmt.executeQuery(query);
			UserGroup user_group;
			long group_pk;
			UserGroupDB userGroupDB = new UserGroupDB(DB_TYPE);
			userGroupDB.setConnection(getConnection());

			while (rs.next()) {
				group_pk = rs.getLong(1);
				user_group = (UserGroup)userGroupDB.selectRow(new Long(group_pk));
				groups.add(user_group);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return groups;
	}
    /**
     * truncate the whole table
     *
     * @exception java.sql.SQLException
     */
    public final void truncate()
        throws SQLException {

        String query = "truncate table user_group_member";
   
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            if (Prefs.DEBUG) LogWrite.write(query);
            stmt.executeUpdate(query);
        } catch (SQLException e) {
            throw e;
        } finally {
            try { if (stmt != null) stmt.close();
            } catch (SQLException x) { }
        }
    }

	/**
	 * select all user group member rows
	 * 
	 * @param lastSyncDate a user's last sync date used to limit
	 * the query, or null if no limit is to be applied
	 * @return the UserGroupMember row(s) that were selected
	 * @exception java.sql.SQLException 
	 */
	public final ArrayList selectAllRows(java.util.Date lastSyncDate)
		throws SQLException {

		ArrayList rows = new ArrayList();
		Statement stmt = null;
		ResultSet rs = null;
		StringBuffer query = new StringBuffer();
		query.append(selectAllQuery);
		if (lastSyncDate != null) {
            query.append(" WHERE user_group_member.modified_date > ");
            query.append(DateUtils.formatDateTime(DB_TYPE, lastSyncDate));
		}

		try {
			stmt = con.createStatement();
			if (Prefs.DEBUG) LogWrite.write(query.toString());
			rs = stmt.executeQuery(query.toString());

			UserGroupMember ugm;

			while (rs.next()) {
				ugm = new UserGroupMember();

				ugm.setPK(rs.getLong(1));
				ugm.setUserGroupPK(rs.getLong(2));
				ugm.setUserPK(rs.getLong(3));
				ugm.setModifiedBy(rs.getString(4));
				ugm.setModifiedDate(rs.getDate(5));

				rows.add(ugm);
			}
		} catch (SQLException e) {
			throw e;
		} finally {
			try {
				if (rs != null) rs.close();
			} catch (SQLException x) { throw x; }
			try {
				if (stmt != null) stmt.close();
			} catch (SQLException x) { throw x; }
		}

		return rows;
	}

}
