package sellwin.domain;

import java.util.*;
import java.io.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.


/**
 * This class represents an Opportunity within SellWin, these
 * act as containers for various other entities such as Activity,
 * Order, Contact, etc.  Opportunities are mapped/stored within
 * the database table 'opportunity'.
 */
public class Opportunity implements Serializable {
	private long pk;
	private String name;
	private int dollarValue;
	private String probability;
	private String stage;
	private String leadSource;
	private String leadType;
	private String desc;
	private String groupName;
	private long primeSalesPersonPK;
	private SalesPerson primeSalesPerson;
	private long customerPK;
	private Customer cust;
	private Date modifiedDate;
	private Date closeDate;
	private String modifiedBy;
	private long leadPK;
	private Lead lead;
	private ArrayList quotes = new ArrayList();
	private ArrayList orders = new ArrayList();
	private ArrayList contacts = new ArrayList();
	private ArrayList activities = new ArrayList();
	private ArrayList forecasts = new ArrayList();
	private boolean modified=false;
	private boolean updatedLocally=false;
	private boolean addedLocally=false;

	public final static String PROB[] = {
		"0%", "10%", "20%", "30%", "40%", "50%", "60%", "70%", "80%", "90%", "100%" };

 	public final static String STAGE_IDENTIFIED = "Identified";
    public final static String STAGE_CONTACTED = "Contacted";
    public final static String STAGE_BIDDING = "Bidding";
    public final static String STAGE_CHANCE = "Chance of Winning";
    public final static String STAGE_WINNING = "Winning";
    public final static String STAGE_AWARDED = "Awarded";
    public final static String STAGE_CONTRACT = "Contract";
    public final static String STAGE_BILLED = "Billed";

	public final static String[] ALL_STAGES = {
 	STAGE_IDENTIFIED,
    STAGE_CONTACTED,
    STAGE_BIDDING,
    STAGE_CHANCE,
    STAGE_WINNING,
    STAGE_AWARDED,
    STAGE_CONTRACT,
    STAGE_BILLED 
	};

	public final static String LEAD_TRADE_SHOW = "Trade Show";
	public final static String LEAD_EMAIL = "E-Mail";
	public final static String LEAD_DIRECT = "Direct";
	public final static String LEAD_SURVEY = "Survey";
	public final static String LEAD_TELE = "Telesales";
	public final static String LEAD_WEB = "Web";

	public final static String[] ALL_LEADS = {
	LEAD_TRADE_SHOW,
	LEAD_EMAIL,
	LEAD_DIRECT,
	LEAD_SURVEY,
	LEAD_TELE,
	LEAD_WEB
	};

	public final static String TYPE_EXISTING = "Existing Business";
	public final static String TYPE_NEW = "New Business";
	
	public final static String[] ALL_LEAD_TYPES = {
		TYPE_EXISTING,
		TYPE_NEW };

	public Opportunity() {
		this(null, null);
	}

	public Opportunity(String n, String creator) {
		clear();
		Date d = new java.util.Date();
		pk = d.getTime();
		setName(n);

		leadType = TYPE_NEW;
		leadSource=LEAD_TRADE_SHOW;
		desc=" ";
		stage=STAGE_IDENTIFIED;		
		modifiedDate = new java.util.Date();
		closeDate = new java.util.Date();
		modifiedBy = creator;
	}

	public final void clear() {
		name = "";
		cust = new Customer();
		cust.setName(Customer.UNASSIGNED);
		dollarValue = 0;
		probability = PROB[0];
		stage = STAGE_IDENTIFIED;
		leadSource = LEAD_DIRECT;
		leadType = TYPE_NEW;
		modifiedBy = "SYSTEM";
	}

	public final void setLeadPK(long k) { leadPK = k; }	
	public final long getLeadPK() { return leadPK; }

	public final Lead getLead() { return lead; }
	public final void setLead(Lead lead) { this.lead = lead; }

	public final String getGroupName() { return groupName; }
	public final void setGroupName(String g) { groupName = g; }

	public final ArrayList getOrders() { return orders; }
	public final ArrayList getQuotes() { return quotes; }

	public final void addOrder(Order o) { orders.add(o); setModified(); }

	public final void addQuote(Quote f) { 
		quotes.add(f);
		setModified();
	}

	public final ArrayList getForecasts() { return forecasts; }

	public final void addForecast(Forecast f) { 
		forecasts.add(f);
		setModified();
	}

	public final ArrayList getActivities() { return activities; }

	public final void addActivity(Activity a) { 
		activities.add(a);
		setModified();
	}

	public final ArrayList getContacts() { return contacts; }

	public final void addContact(Contact c) {
		contacts.add(c);
		setModified();
	}

	public final long getPK() { return pk; }

	public final void setPK(long i) { pk = i; }
		
	public final void setName(String name) { this.name = name; }

	public final String getName() { return name; }

	public final void setDollarValue(int d) { dollarValue = d; }

	public final int getDollarValue() { return dollarValue; }

	public final void setProbability(String i) { probability = i; }
	
	public final String getProbability() { return probability; }
	
	public final void setStage(String s) { stage = s; }
	
	public final String getStage() { return stage; }

	public final void setDesc(String s) { desc = s; }
	
	public final String getDesc() { return desc; }

	public final void setPrimeSalesPersonPK(long p) { primeSalesPersonPK = p; }
	public final long getPrimeSalesPersonPK() { return primeSalesPersonPK; }

	public final void setPrimeSalesPerson(SalesPerson s) { primeSalesPerson = s; }

	public final SalesPerson getPrimeSalesPerson() { return primeSalesPerson; }


	public final long getCustomerPK() { return customerPK; }
	public final void setCustomerPK(long k) { customerPK = k; }

	public final Customer getCustomer() { return cust; }

	public final void setCustomer(Customer c) { 
		cust = c; 
		customerPK = c.getPK();
	}

	public final void setCloseDate(Date t) { closeDate = t; }
	public final Date getCloseDate() { return closeDate; }

	public final void setModifiedDate(Date t) { modifiedDate = t; }
	public final Date getModifiedDate() { return modifiedDate; }

	public final void setModifiedBy(String d) { modifiedBy = d; }
	public final String getModifiedBy() { return modifiedBy; }

	public final Contact findContact(String formattedName) {
		Contact c = null;
		for (int i = 0;i<contacts.size();i++) {
			c = (Contact)contacts.get(i);
			if (c.getAddress().getFormattedName().equals(formattedName))
				return c;
		}

		return null; //not found 
	}

	public final void setLeadSource(String l) { leadSource = l; }
	public final String getLeadSource() { return leadSource; }

	public final void setLeadType(String l) { leadType = l; }
	public final String getLeadType() { return leadType; }

	public final void setModified() { modified=true; }
	public final boolean isModified() { return modified; }

	public final void setUpdatedLocally(boolean b) { updatedLocally=b; }
	public final void setAddedLocally(boolean b) { addedLocally=b; }
	public final boolean getUpdatedLocally() { return updatedLocally; }
	public final boolean getAddedLocally() { return addedLocally; }

	public final void print() {
		System.out.print( "Opportunity: " + pk);
		System.out.println( " name="+name);
		System.out.println( "leadPK="+lead.getPK());
		System.out.println( "Group="+groupName);
		System.out.println( "dollarValue="+ dollarValue);
		System.out.println( "probability="+ probability);
		System.out.println( "stage="+ stage);
		System.out.println( "lead source="+ leadSource);
		System.out.println( "lead type="+ leadType);
		System.out.println( " desc="+desc);
		System.out.println( "prime sales person="+ primeSalesPerson.getAddress().getFormattedName());
		System.out.println( "customer=" + cust.getName());
		System.out.println( "modifiedDate="+ modifiedDate);
		System.out.println( "closeDate="+ closeDate);
		System.out.println( "modifiedBy="+ modifiedBy);
		System.out.println( "modified="+ modified);
		ArrayList contacts = getContacts();
		Contact contact = null;
		for (int i=0;i<contacts.size();i++) {
			System.out.println("Contact #"+i);
			contact = (Contact)contacts.get(i);
			contact.print();
		}
	}

	public final void deleteQuote(Quote q) {
		ArrayList quotes = getQuotes();
		Quote quote = null;
		for (int i=0;i<quotes.size();i++) {
			quote = (Quote)quotes.get(i);
			if (quote.getPK() == q.getPK())
				quotes.remove(i);
		}
	}

	//local remove of an activity from this opp
	//not to be confused with remote delete
	public final void deleteActivity(Activity a) {
		ArrayList acts = getActivities();
		Activity act = null;
		for (int i=0;i<acts.size();i++) {
			act = (Activity)acts.get(i);
			if (act.getPK() == a.getPK())
				acts.remove(i);
		}
	}

	public final Opportunity copy() {
		String copyName = new String(name);

		Opportunity copy = new Opportunity(copyName, getModifiedBy());

		copy.setGroupName(new String(groupName));	
		copy.setModifiedBy(new String(modifiedBy));	
		Contact contact, contactCopy;	
		ArrayList contacts = getContacts();
		for (int i=0;i<contacts.size();i++) {
			contact = (Contact)contacts.get(i);
			contactCopy = contact.copy();
			copy.addContact(contactCopy);
		}

		Activity activity, activityCopy;
		ArrayList activities = getActivities();
		for (int j=0;j<activities.size();j++) {
			activity = (Activity)activities.get(j);
			activityCopy = activity.copy();
			copy.addActivity(activityCopy);	
		}

		Forecast forecast, forecastCopy;
		ArrayList forecasts = getForecasts();
		for (int k=0;k<forecasts.size();k++) {
			forecast = (Forecast)forecasts.get(k);
			forecastCopy = forecast.copy();
			copy.addForecast(forecastCopy);
		}

		Quote quote, quoteCopy;
		ArrayList quotes = getQuotes();
		for (int k=0;k<quotes.size();k++) {
			quote = (Quote)quotes.get(k);
			quoteCopy = quote.copy();
			copy.addQuote(quoteCopy);
		}
		Order order, orderCopy;
		ArrayList orders = getOrders();
		for (int k=0;k<orders.size();k++) {
			order = (Order)orders.get(k);
			orderCopy = order.copy();
			copy.addOrder(orderCopy);
		}

		if (primeSalesPerson != null)
			copy.setPrimeSalesPerson(primeSalesPerson.copy());
		if (cust != null)
			copy.setCustomer(cust.copy());

		copy.pk = pk;
		if (lead != null)
			copy.lead = lead.copy();
		copy.dollarValue = dollarValue;
		copy.probability = new String(probability);
		if (stage != null)
			copy.stage = new String(stage);
		if (desc != null)
			copy.desc = new String(desc);
		if (leadSource != null)
			copy.leadSource = new String(leadSource);
		if (leadType != null)
			copy.leadType = new String(leadType);
		if (modifiedDate != null)
			copy.modifiedDate = new Date(modifiedDate.getTime());
		if (closeDate != null)
			copy.closeDate = new Date(closeDate.getTime());

		return copy;
	}
	
	public final void printActivities() {
		Activity a;
		for (int i=0;i<getActivities().size();i++) {
			a = (Activity)getActivities().get(i);
			a.print();
		}
	}

}
