package sellwin.gui;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;

import sellwin.utils.*;
import sellwin.domain.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * chat dialog that shows the chat window, users can view
 * what other users are online, and strike up online chats with
 * them, a simple chat protocol is implemented
 *
 * @author  jmccormi
 */
public class ChatDialog extends javax.swing.JDialog implements GUIChars {

	// the basic widgets that make up the screen's look

	private javax.swing.JTextArea msgArea;
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JList usersList;
	private JCheckBox beepBox;
    private javax.swing.JButton sendButton;
    private javax.swing.JButton closeButton;
    private javax.swing.JTextArea conversationArea;
	private JTextPane conversationTextPane;
    private javax.swing.JScrollPane jScrollPane3;
    private javax.swing.JScrollPane jScrollPane2;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JPanel mainPanel;

	// some text styles used to display the received chat text
	private String newline = "\n";
	private String[] initStyles = { "bold", "regular" };
	private Style bold, regular, regularFrom;
  
	// a thread that runs all the time to update the list of online
	// users 
	private ChatOnlineCheck checker = null;
	private JDialog thisDialog = null;

	// this user's socket information used for sending-receiving chat msgs	
    private ObjectOutputStream oos = null;
	private Socket socket = null;
	private SocketReader socketReader = null;

	private static String currentUser = null;

	private Whiteboard wb = null;

 
    /** Creates new form ChatDialog */
    public ChatDialog(java.awt.Frame parent, boolean modal)
		throws ConnectException, IOException {

        super(parent, modal);
		thisDialog = this;
        initComponents();
		initStylesForTextPane();

		wb = MainWindow.getWhiteboard();

		setLang();
		setColors();
		setFonts();

		createSocket();

		try {
			currentUser = wb.getCurrentUser().getID();
			logonToChat();
            checker = new ChatOnlineCheck(oos, currentUser);
			sendQuery();
            checker.start();
        } catch (Exception e) {
            ErrorHandler.show(thisDialog, e);
        }
    }
  
	/**
	 * setup the text styles for the reply message box
	 */ 
	private void initStylesForTextPane() {
		Style def = StyleContext.getDefaultStyleContext().getStyle(StyleContext.DEFAULT_STYLE);

		regular = conversationTextPane.addStyle("regular", def);
		StyleConstants.setFontFamily(regular, "SansSerif");
		StyleConstants.setForeground(regular, Color.RED);

		regularFrom = conversationTextPane.addStyle("regularFrom", def);
		StyleConstants.setFontFamily(regularFrom, "SansSerif");
		StyleConstants.setForeground(regularFrom, Color.GREEN);

		bold = conversationTextPane.addStyle("bold", regular);
		StyleConstants.setForeground(bold, Color.BLACK);
		StyleConstants.setBold(bold, true);

	}

	/**
	 * create the screen's widgets
	 */ 
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

		setTitle("Chat Dialog for " + currentUser);
        buttonPanel = new javax.swing.JPanel();
		beepBox = new JCheckBox("Beep");
        sendButton = new javax.swing.JButton();
        closeButton = new javax.swing.JButton();
        mainPanel = new javax.swing.JPanel();
        jScrollPane1 = new javax.swing.JScrollPane();
        conversationArea = new javax.swing.JTextArea();
		conversationTextPane = new JTextPane();
        jScrollPane2 = new javax.swing.JScrollPane();
        msgArea = new javax.swing.JTextArea();
        jScrollPane3 = new javax.swing.JScrollPane();
        usersList = new javax.swing.JList();

        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog(evt);
            }
        });

        beepBox.setText("Beep");
        buttonPanel.add(beepBox);
        sendButton.setText("Send");
        buttonPanel.add(sendButton);

        closeButton.setText("Close");
        buttonPanel.add(closeButton);

        getContentPane().add(buttonPanel, java.awt.BorderLayout.SOUTH);

        mainPanel.setLayout(new java.awt.GridBagLayout());

        mainPanel.setBorder(new javax.swing.border.EtchedBorder());
        jScrollPane1.setName("null");
        jScrollPane1.setMinimumSize(new java.awt.Dimension(400, 200));
		conversationTextPane.setEditable(false);
        jScrollPane1.setViewportView(conversationTextPane);
		jScrollPane1.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 0);
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(jScrollPane1, gridBagConstraints);

		int HEIGHT=100;
        jScrollPane2.setName("null");
        jScrollPane2.setMinimumSize(new java.awt.Dimension(300, HEIGHT));
        jScrollPane2.setPreferredSize(new java.awt.Dimension(360, HEIGHT));
		jScrollPane2.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        msgArea.setColumns(30);
		msgArea.setLineWrap(true);
		msgArea.setWrapStyleWord(true);
        msgArea.setRows(6);
        jScrollPane2.setViewportView(msgArea);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 7);
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(jScrollPane2, gridBagConstraints);

        jScrollPane3.setPreferredSize(new java.awt.Dimension(80, HEIGHT));
        jScrollPane3.setName("null");
		jScrollPane3.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        jScrollPane3.setMinimumSize(new java.awt.Dimension(80, HEIGHT));
        jScrollPane3.setMaximumSize(new java.awt.Dimension(80, HEIGHT));
        usersList.setName("null");
		usersList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        jScrollPane3.setViewportView(usersList);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(jScrollPane3, gridBagConstraints);

        getContentPane().add(mainPanel, java.awt.BorderLayout.CENTER);

        pack();

		this.addWindowFocusListener(
			new WindowFocusListener() {
				public void windowGainedFocus(WindowEvent ev) {
				
					if ((checker != null) && (checker.isAlive()))
						; //System.out.println("thread already running");
					else {
						//System.out.println("starting new thread");
            			checker = new ChatOnlineCheck(oos, currentUser);
						checker.start();
	
						try {
							createSocket();
						} catch (Exception x) {
							ErrorHandler.show(thisDialog, x);
						}
					}
				}

				public void windowLostFocus(WindowEvent ev) {
				}
			}
		);

		sendButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent ev) {
					ChatPacket msg = new ChatPacket();
					msg.setAction(ChatPacket.SEND_MSG);
					ArrayList toList = new ArrayList();
					String toUser = (String)usersList.getSelectedValue();			
					toList.add(toUser);
					msg.setToList(toList);
					msg.setMessage(msgArea.getText());
					msg.setSender(currentUser);
					sendMessage(msg);
					msgArea.setText("");
				}
			}
		);

		closeButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent ev) {
					thisDialog.hide();
				}
			}
		);

		setSize(463, 219);
    }
    
    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt) {
        setVisible(false);
        dispose();
    }
    
    
	/**
	 * append the received chat message onto the window's text
	 * area
	 * @param the chat message to display
	 * @param style the text style to use when drawing
	 */
	public void appendConversation(ChatPacket msg, Style style) {
		Document doc = conversationTextPane.getDocument();
		
		try {
			doc.insertString(doc.getLength(), msg.getSender(),
				conversationTextPane.getStyle("bold"));

			doc.insertString(doc.getLength(), " : ",
				conversationTextPane.getStyle("bold"));

			doc.insertString(doc.getLength(), msg.getMessage(), style);

			doc.insertString(doc.getLength(), newline, style);


		} catch (BadLocationException ble) {
			ErrorHandler.show(thisDialog, ble);
		}
	}

	/**
	 * send a chat message to the chat server
	 * @param chatMsg the chat message to send
	 */
	public void sendMessage(ChatPacket chatMsg) {
		try {
            oos.writeObject(chatMsg);
            oos.flush();
			if (chatMsg.getAction() == ChatPacket.SEND_MSG)
				appendConversation(chatMsg, regularFrom);
       	} catch (IOException e){
			ErrorHandler.show(thisDialog, e);
       	}
	}    

	/**
	 * create a socket connection to the chat server
	 */
	public void createSocket() 
		throws IOException {

	  	socket = new Socket("localhost", 7900);
   	 	oos = new ObjectOutputStream(socket.getOutputStream());
		socketReader = new SocketReader(socket, conversationTextPane, usersList);
		socketReader.start();
  	}

	/**
	 * this class is a thread which runs periodically and reads
	 * messages from the chat server
	 */
	public class SocketReader extends Thread {

		/** flag which allows us to shut down this thread gracefully */
		private boolean readingSocket = true;

		/** input socket stream we will read from */
		private ObjectInputStream ois = null;
		private Socket socket = null;

		/** window list that we update with query results */
		private JList usersList = null;

		/** window text pane we update with chat msgs */
		private JTextPane textPane = null;

		/** build the thread */
		public SocketReader(Socket socket, JTextPane textPane, JList usersList) 
			throws IOException {

			this.socket = socket;
			this.textPane = textPane;
			this.usersList = usersList;
			ois = new ObjectInputStream(socket.getInputStream());
		}

		/** 
		 * stop the thread gracefully 
	 	 */
		public void stopRunning() {
			readingSocket = false;
		}

		/**
		 * this thread's run() method which runs all the
	 	 * time to check for chat messages
		 */
		public void run() {
			try {
				while (readingSocket == true) {
					ChatPacket msg = (ChatPacket)ois.readObject();
					//System.out.println("this user just got in a message");
					if (msg.getAction() == ChatPacket.QUERY_USERS) {
						//System.out.println("QUERY USERS received");
						updateUsersList(msg);
					} else {
						//System.out.println("SEND MSG received");
						updateMsgArea(msg);
						if (beepBox.isSelected())
							Toolkit.getDefaultToolkit().beep();
					}
				}
			} catch (java.io.EOFException err) {
				ErrorHandler.show(thisDialog, err, wb.getLang().getString("chatServerDied"));
			} catch (Exception e) {
				ErrorHandler.show(thisDialog, e);
			}
		}

		/**
	 	 * update the window's chat message area with a chat message
		 * @param msg the chat message to draw
		 */
		private void updateMsgArea(ChatPacket msg) {
			//System.out.println(" got a message back will update msg area");
			appendConversation(msg, regular);
		}

		/**
		 * update the list of online users
		 * @param the chat message holding the list of online users	
		 */
		private void updateUsersList(ChatPacket msg) {

			String selectedItem = (String)usersList.getSelectedValue();
			usersList.setListData(msg.getOnlineUsers().toArray());
			if (selectedItem != null) {
				usersList.setSelectedValue(selectedItem, true);
			}
		}
	}

	/**
	 * logon to the chat server by sending a logon message
	 */
	private void logonToChat() {

		ChatPacket msg = new ChatPacket();
		msg.setAction(ChatPacket.LOGON);
		msg.setSender(currentUser);
		sendMessage(msg);
	}

	/**
	 * send a query message to the chat server
	 */
	private void sendQuery() {
		try {
			ChatPacket msg = new ChatPacket();
			msg.setSender(currentUser);
			msg.setAction(ChatPacket.QUERY_USERS);
			
			oos.writeObject(msg);
			oos.flush();
		} catch (Exception e) {
			ErrorHandler.show(thisDialog, e);
		} 
	}

	/**
 	* This class is run in a separate thread in the
 	* Swing GUi to check for online users
 	*/
	class ChatOnlineCheck extends Thread {
		private String currentUser = null;
		private ObjectOutputStream oos = null;
		private Socket socket = null;
		private Runnable runnable;
		private static final int ACT_CHK_TIME = 1000 * 10; //10 seconds
		private ArrayList alarms=null;
		private boolean stopRunning=false;
	
		/**
	 	* this method is called by Swing as it processes
	 	* it's event queue which gets populated by the invokeLater()
	 	* below
	 	*/
		public ChatOnlineCheck(ObjectOutputStream oos, String currentUser)  {
			this.oos = oos;
			this.currentUser = currentUser;

			runnable = new Runnable() {
				public void run() {
					System.out.println("ChatOnlineCheck running ");
				}
			};
		}

		/**
	 	* stop this thread from continuing
	 	*/
		public void stopRunning() {
			stopRunning = true;
		}

		/**
		 * the main entry point for this Thread, an infinite loop that
		 * wakes up every so often (ACT_CHK_TIME), does some checking to
		 * see if we need to process some activities, and then queue a 
		 * request to Swing's internal event queue processing mechanism,
		 * then go back to waiting for the next wakeup.
		*/
		public void run() {
			try {
				ChatPacket msg = new ChatPacket();
				msg.setSender(currentUser);
				msg.setAction(ChatPacket.QUERY_USERS);
			
				while (stopRunning == false) {
					sleep(ACT_CHK_TIME);
				
					oos.writeObject(msg);
					oos.flush();
					SwingUtilities.invokeLater(runnable);
				}
			} catch (Exception e) {
				ErrorHandler.show(thisDialog, e);
			} 
		}
	}
    /**
     * set the screen's text fields to some language
     */
    public final void setLang() {
        setTitle(wb.getLang().getString("chatDialog"));
        sendButton.setText(wb.getLang().getString("send"));
        closeButton.setText(wb.getLang().getString("close"));
		beepBox.setText(wb.getLang().getString("beep"));
	}

    /**
     * set the screen's colors
     */
    public final void setColors() {
    }

    /**
     * set the screen's fonts
     */
    public final void setFonts() {
		beepBox.setFont(MainWindow.LABEL_FONT);
        sendButton.setFont(MainWindow.LABEL_FONT);
        closeButton.setFont(MainWindow.LABEL_FONT);
	}

}
