package sellwin.gui;

import sellwin.domain.*;
import sellwin.utils.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the Contacts panel that users
 * see from the Contacts tab on the MainWindow (GUI).
 */
public class ContactsPanel extends JPanel implements GUIChars {
	private Whiteboard wb = null;
	private Opportunity currentOpp = null;
	private ContactsPanel thisPanel = null;
	private JPopupMenu popup = new JPopupMenu();

    private javax.swing.JPanel buttonPanel;
    private javax.swing.JMenuItem deleteMenuItem;
    private javax.swing.JButton addButton;
    private javax.swing.JButton applyButton;
    private final javax.swing.JPanel mainPanel = new JPanel();
    private javax.swing.JLabel contactsLabel;
	private AddressPanel addressPanel = new AddressPanel();
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JList contactsList;
    private javax.swing.JComboBox stateCombo;
    private javax.swing.JTextField miField;
    private javax.swing.JTextField firstNameField;

	/**
	 * construct the contacts panel, initialize it 
	 */
	public ContactsPanel() {
		super();

		thisPanel = this;	

		wb = MainWindow.getWhiteboard();
		initComponents();

		setFonts();
		setColors();
		setLang();

		applyButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					Contact c = getContactFromScreen();

					Contact existingContact;

                    int index = contactsList.getSelectedIndex();

					if (index >= 0) {
						ArrayList contacts = wb.getCurrentOpportunity().getContacts();
						existingContact = (Contact)(contacts.get(index));
					} else {
						JOptionPane.showMessageDialog(thisPanel,
								"Select a contact first.", "Error",
								JOptionPane.ERROR_MESSAGE);
						return;
					}

					try {
	
						if (existingContact != null) {
							Contact.copyFields(existingContact, c);
							wb.updateContact(existingContact);
							refreshOpportunity();  //useful when the name chgs
							JOptionPane.showMessageDialog(thisPanel,
								"Success", "Contact Updated",
								JOptionPane.INFORMATION_MESSAGE);
						} 
					} catch (Exception x) {
						ErrorHandler.show(thisPanel, x);
					}
				}
			}
		);

		addButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					try {
						Opportunity opp = wb.getCurrentOpportunity();
						if (opp == null) {
							JOptionPane.showMessageDialog(thisPanel,
								wb.getLang().getString("selectOpp"),
								wb.getLang().getString("error"),
								JOptionPane.INFORMATION_MESSAGE);
							return;
						}
						Contact c = getContactFromScreen();
						wb.addContact(c);
						currentOpp.addContact(c);
						refreshOpportunity();
						JOptionPane.showMessageDialog(thisPanel,
								wb.getLang().getString("contactAdded"), 
								wb.getLang().getString("success"),
								JOptionPane.INFORMATION_MESSAGE);
						addressPanel.clear();
					} catch (AngError x) {
						ErrorHandler.show(thisPanel, x);
					}
				}
			}
		);

		deleteMenuItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					try {
                        int index = contactsList.getSelectedIndex();
						if (index >= 0) {
							ArrayList contacts = wb.getCurrentOpportunity().getContacts();
							Contact currentContact = (Contact)(contacts.get(index));
							wb.deleteContact(currentContact.getPK());
							contacts.remove(index);
							refreshOpportunity();
						}
					} catch (Exception f) {
						ErrorHandler.show(thisPanel, f);
					}
				}
			}
		);



		contactsList.addMouseListener(
            new MouseAdapter() {
				public void mousePressed(MouseEvent e) {
					if (e.isPopupTrigger())
						popup.show(contactsList, e.getX(), e.getY());
				}

				public void mouseReleased(MouseEvent e) {
					if (e.isPopupTrigger())
						popup.show(contactsList, e.getX(), e.getY());
				}

                public void mouseClicked(MouseEvent e) {
					if (e.isPopupTrigger()) {
						popup.show(contactsList, e.getX(), e.getY());
						return;
					}
					
				
                    switch (e.getClickCount()) {
                        case 1:
                            int index = contactsList.getSelectedIndex();
							if (index < 0) return;
							ArrayList contacts = wb.getCurrentOpportunity().getContacts();	
							Contact x = (Contact)(contacts.get(index));
							if (x != null)
								displayContact(x);
							else
								System.out.println("invalid contact lookup");
                            break;
                        default:
                            break;
                    }
                }
            }
        );

		createPopup(this);
	}

	/**
	 *  create a popup menu that is displayed over the table
	 * @param c the component we are adding this popup to
	 */
    private final void createPopup(Component c) {
		popup.add(deleteMenuItem);

        c.addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(mainPanel, e.getX(), e.getY());
            }

            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(mainPanel, e.getX(), e.getY());
            }

            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    popup.show(mainPanel, e.getX(), e.getY());
            }
        });
    }

	/**
	 * display a Contact in this panel 
	 * @param x the Contact to display
	 */
	public final void displayContact(Contact x) {
		clearContact();
		addressPanel.setAddress(x.getAddress());
	}

	/**
	 * clear the Contact panel 
	 */
	public final void clearContact() {
		Component[] children = getComponents();
		for (int i=0;i<children.length;i++) {
			if (children[i] instanceof JTextField)
				((JTextField)children[i]).setText("");
		}
	}

	/**
	 * get the Contact from the screen's fields 
	 * @return the Contact 
	 */
	public final Contact getContactFromScreen() {
		Address c = addressPanel.getAddress();
		Contact contact = new Contact();
		contact.setAddress(c);
		return contact;
	}

	/**
	 * refresh the screen with another opportunity's values 
	 */
	public final void refreshOpportunity() {
		currentOpp = wb.getCurrentOpportunity();
		DefaultListModel mod = new DefaultListModel();
		ArrayList contacts = currentOpp.getContacts();

		Contact contact = null;
		for (int i=0;i<contacts.size();i++) {
			contact = (Contact)(contacts.get(i));
			mod.addElement(contact.getAddress().getFormattedName());
		}

		contactsList.setModel(mod);

		if (contacts.size() > 0) {
			contactsList.setSelectedIndex(0);
			displayContact((Contact)contacts.get(0));
		}
	}

	/**
	 * set the screen colors
	 */
	public final void setColors() {
		contactsLabel.setForeground(MainWindow.LETTERS);
	}

	/**
	 * set the screen's fonts 
	 */
	public final void setFonts() {
		contactsLabel.setFont(MainWindow.LABEL_FONT);
		contactsList.setFont(MainWindow.FIELD_FONT);
		addButton.setFont(MainWindow.LABEL_FONT);
		deleteMenuItem.setFont(MainWindow.LABEL_FONT);
        applyButton.setFont(MainWindow.LABEL_FONT);
	}

	/**
	 * initialize or build the screen's fields
     */
    private final void initComponents() {
        buttonPanel = new JPanel();
        addButton = new JButton();
        deleteMenuItem = new JMenuItem();
        applyButton = new JButton();
        contactsLabel = new JLabel();
        jScrollPane1 = new JScrollPane();
        contactsList = new JList();
        
        setLayout(new java.awt.BorderLayout());
        
        
        deleteMenuItem.setText("Delete");
        
        addButton.setText("Add");
        applyButton.setText("Apply");
		buttonPanel.add(addButton);
        buttonPanel.add(applyButton);
        
        add(buttonPanel, java.awt.BorderLayout.SOUTH);
        
        mainPanel.setLayout(new java.awt.GridBagLayout());
        java.awt.GridBagConstraints gridBagConstraints1;
        
        mainPanel.setBorder(new javax.swing.border.EtchedBorder());
       
        jScrollPane1.setVerticalScrollBarPolicy(javax.swing.JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        jScrollPane1.setHorizontalScrollBarPolicy(javax.swing.JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        jScrollPane1.setMinimumSize(new Dimension(300, 100));
        jScrollPane1.setViewportView(contactsList);
        
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridwidth = 3;
        gridBagConstraints1.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
		gridBagConstraints1.gridx = 1;
        gridBagConstraints1.gridy = 0;
        mainPanel.add(jScrollPane1, gridBagConstraints1);

		addressPanel.setBorder(new TitledBorder("Contact Address"));
        gridBagConstraints1 = new java.awt.GridBagConstraints();
		gridBagConstraints1.gridx = 1;
        gridBagConstraints1.gridy = 1;
        mainPanel.add(addressPanel, gridBagConstraints1);
        
        add(mainPanel, java.awt.BorderLayout.CENTER);
        
    }

	/**
	 * see if the contact on the screen already exists, this tells us
	 * whether to do an 'update' or an 'add' of the contact info
	 * @param c the Contact we are checking for
	 * @return the Contact that was found in the current list 
	 */
	private final Contact checkInList(Contact c) {
		ArrayList contacts = wb.getCurrentOpportunity().getContacts();
		Contact contact = null;
		for (int i=0;i<contacts.size();i++) {
			contact = (Contact)(contacts.get(i));
			if (contact.getAddress().getFormattedName().equals( c.getAddress().getFormattedName())) {
				return contact;
			}
		}
		return null;
	}

	/**
	 * clear all the screen components 
	 */
	public final void clearOpportunity() {
		DefaultListModel mod = new DefaultListModel();
		contactsList.setModel(mod);
	}

	/**
	 * set the screen's text to some language
	 */
	public final void setLang() {
		addButton.setText(wb.getLang().getString("add"));
		deleteMenuItem.setText(wb.getLang().getString("delete"));
	    applyButton.setText(wb.getLang().getString("apply"));
		addressPanel.setBorder(
			new TitledBorder(
				null,
				wb.getLang().getString("contactAddress"),
                TitledBorder.DEFAULT_JUSTIFICATION,
                TitledBorder.DEFAULT_POSITION,
                MainWindow.LABEL_FONT,
                MainWindow.LETTERS
			)
		);
	}
}
