package sellwin.gui;

import sellwin.domain.*;
import sellwin.utils.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.border.*;
import java.awt.*;
import java.util.*;
import java.awt.event.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the SellWin Forecast Panel which
 * displays in the Forecast tab in the GUI.  All information
 * about an opportunity's set of Forecasts is displayed
 * in this window.
 */
public class ForecastPanel extends JPanel implements GUIChars {

	private JComboBox scenarios = new JComboBox(Forecast.SCENARIOS);

	private ForecastPanel thisPanel = null;
	private JPopupMenu popup = new JPopupMenu();

    private JPanel jPanel1;
    private JButton applyButton;
    private JMenuItem addMenuItem;
    private JMenuItem copyMenuItem;
    private JMenuItem deleteMenuItem;
    private JMenuItem submitMenuItem;
    private final javax.swing.JPanel mainPanel = new JPanel();
    private JLabel label1;
    private JTextField oppName;
    private JLabel custLabel;
    private JTextField custName;
    private JLabel tableLabel;
	private JPanel forecastsTablePanel = new JPanel();
    private JScrollPane tableScrollPane;
    private final javax.swing.JTable table = new JTable();

	private Whiteboard wb;
	private Opportunity currentOpp;
	private ForecastTableModel model = null;

	/**
	 * construct a forecast panel 
	 */
	public ForecastPanel() {
		super();

		thisPanel = this;

		wb = MainWindow.getWhiteboard();
	
        initComponents();

        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        table.getSelectionModel().addListSelectionListener(
            new ListSelectionListener() {
                public void valueChanged(ListSelectionEvent e) {
                    if (e.getValueIsAdjusting()) return; //ignore these extra msgs
                    ListSelectionModel lsm = (ListSelectionModel)e.getSource();
                }
            }
        );

		//start with a blank table
		model = new ForecastTableModel(new ArrayList());
		table.setModel(model);
		initializeScenarioColumn();
		table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);


		submitMenuItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					int opt = JOptionPane.showConfirmDialog(thisPanel, 
						"Continue with submit?");
					if (opt != JOptionPane.YES_OPTION) return;

					ListSelectionModel m = table.getSelectionModel();
					if (m.getLeadSelectionIndex() >= 0) {
						Forecast row = model.getForecast(m.getLeadSelectionIndex());
						row.setSubmitted(new Boolean(true));
						table.tableChanged(new TableModelEvent(model));
						try {
							wb.updateForecast(row);
						} catch (AngError x) {
							ErrorHandler.show(thisPanel, x);
						}
					} else 
						JOptionPane.showMessageDialog(thisPanel,
							"User Error",
							"Please select a row first.",
							JOptionPane.INFORMATION_MESSAGE);
				}
			}
		);

		copyMenuItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					ListSelectionModel m = table.getSelectionModel();
					if (m.getLeadSelectionIndex() >= 0) {
						Forecast row = model.getForecast(m.getLeadSelectionIndex());
						Forecast copyForecast = row.copy();
						copyForecast.setName(copyForecast.getName() + "_copy");
						try {
							wb.addForecast(wb.getCurrentOpportunity().getPK(),
								copyForecast);
							table.tableChanged(new TableModelEvent(model));
						} catch (Exception g) {
							ErrorHandler.show(thisPanel, g);
						}
					}
				}
			}
		);

		deleteMenuItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					ListSelectionModel m = table.getSelectionModel();
					if (m.getLeadSelectionIndex() >= 0) {
						Forecast row = model.getForecast(m.getLeadSelectionIndex());
						try {
							int option = JOptionPane.showConfirmDialog(
								thisPanel, "Continue with delete?");
							if (option != JOptionPane.YES_OPTION)
								return;
							wb.deleteForecast(
								wb.getCurrentOpportunity().getPK(),
								row.getPK());
							JOptionPane.showMessageDialog( thisPanel,
								"Delete Successful", "Delete Successful",
								JOptionPane.INFORMATION_MESSAGE);
							model.deleteForecast(m.getLeadSelectionIndex());
							table.tableChanged(new TableModelEvent(model));
						} catch (AngError x) {
							ErrorHandler.show(thisPanel, x);
						}
					}
				}
			}
		);

		addMenuItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					Opportunity opp = wb.getCurrentOpportunity();
					if (opp == null) {
						JOptionPane.showMessageDialog(thisPanel,
             				wb.getLang().getString("selectOpp"),
                            wb.getLang().getString("error"),
							JOptionPane.INFORMATION_MESSAGE);
						return;
					}
					String name = JOptionPane.showInputDialog("Please enter a forecast name.");
					Forecast f = new Forecast();
					f.setName(name);
					f.setCloseDate(opp.getCloseDate());
					try {
						wb.addForecast(opp.getPK(), f);
						model = new ForecastTableModel(opp.getForecasts());
						table.setModel(model);
						initializeScenarioColumn();
						table.tableChanged(new TableModelEvent(model));
					} catch (Exception g) {
						ErrorHandler.show(thisPanel, g);
					}
				}
			}
		);

		setFonts();
		setColors();

		createPopup();
		table.addMouseListener(new MouseAdapter() {
			public void mousePressed(MouseEvent e) {
				if (e.isPopupTrigger())
					popup.show(table, e.getX(), e.getY());
			}

			public void mouseReleased(MouseEvent e) {
				if (e.isPopupTrigger())
					popup.show(table, e.getX(), e.getY());
			}

			public void mouseClicked(MouseEvent e) {
				if (e.isPopupTrigger())
					popup.show(table, e.getX(), e.getY());
			}
		});

		table.getTableHeader().addMouseListener(new MouseAdapter() {
			public void mousePressed(MouseEvent e) {
				if (e.isPopupTrigger())
					popup.show(table, e.getX(), e.getY());
			}

			public void mouseReleased(MouseEvent e) {
				if (e.isPopupTrigger())
					popup.show(table, e.getX(), e.getY());
			}

			public void mouseClicked(MouseEvent e) {
				if (e.isPopupTrigger())
					popup.show(table, e.getX(), e.getY());
			}
		});

        tableScrollPane.addMouseListener(new MouseAdapter() {
			public void mousePressed(MouseEvent e) {
				if (e.isPopupTrigger())
					popup.show(table, e.getX(), e.getY());
			}

			public void mouseReleased(MouseEvent e) {
				if (e.isPopupTrigger())
					popup.show(table, e.getX(), e.getY());
			}

			public void mouseClicked(MouseEvent e) {
				if (e.isPopupTrigger())
					popup.show(table, e.getX(), e.getY());
			}
		});

		applyButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					int opt = JOptionPane.showConfirmDialog(thisPanel, 
						"Continue with update?");
					if (opt != JOptionPane.YES_OPTION) return;

					ListSelectionModel m = table.getSelectionModel();
					if (m.getLeadSelectionIndex() >= 0) {
						Forecast row = model.getForecast(m.getLeadSelectionIndex());
						table.tableChanged(new TableModelEvent(model));
						try {
							wb.updateForecast(row);
						} catch (AngError x) {
							ErrorHandler.show(thisPanel, x);
						}
					} else 
						JOptionPane.showMessageDialog(thisPanel,
							"User Error",
							"Please select a row first.",
							JOptionPane.INFORMATION_MESSAGE);
				}
			}
		);

		oppName.setEditable(false);
		custName.setEditable(false);

		scenarios.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					ListSelectionModel m = table.getSelectionModel();
					if (m.getLeadSelectionIndex() >= 0) {
						Forecast row = model.getForecast(m.getLeadSelectionIndex());
						row.factorChange();
					}
				}
			}
		);

		initTableColumnSize();
		setLang();
	}

	/**
	 * create a menu popup that will hover over the table
	 */
	private final void createPopup() {
    	popup.add(addMenuItem);
    	popup.add(copyMenuItem);
    	popup.add(deleteMenuItem);
    	popup.add(submitMenuItem);
	}

	/**
	 * refresh the data on the screen using the current
	 * opportunity
	 */
	public final void refreshOpportunity() {
		currentOpp = wb.getCurrentOpportunity();

		setSecurity();

		Utility.strSet(custName, currentOpp.getCustomer().getName());
		Utility.strSet(oppName, currentOpp.getName());
		ArrayList rows=null;
		if (currentOpp.getForecasts().size() == 0)
			rows = new ArrayList();
		else
			rows = currentOpp.getForecasts();

		model = new ForecastTableModel(rows);
		table.setModel(model);
		initializeScenarioColumn();
		initializeDateColumn();
		table.setDefaultRenderer(Double.class, new MoneyRenderer());
		table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
	}

	/**
	 * set the screen's colors 
	 */
	public final void setColors() {
		label1.setForeground(MainWindow.LETTERS);
		custLabel.setForeground(MainWindow.LETTERS);
		tableLabel.setForeground(MainWindow.LETTERS);
		table.setBackground(Color.white);
	}


	/**
	 * set the screen's fonts 
	 */
	public final void setFonts() {
    	applyButton.setFont(MainWindow.LABEL_FONT);
    	addMenuItem.setFont(MainWindow.LABEL_FONT);
	    copyMenuItem.setFont(MainWindow.LABEL_FONT);
   	 	deleteMenuItem.setFont(MainWindow.LABEL_FONT);
    	submitMenuItem.setFont(MainWindow.LABEL_FONT);
    	label1.setFont(MainWindow.LABEL_FONT);
    	oppName.setFont(MainWindow.FIELD_FONT);
    	custLabel.setFont(MainWindow.LABEL_FONT);
    	custName.setFont(MainWindow.FIELD_FONT);
    	tableLabel.setFont(MainWindow.LABEL_FONT);
    	table.setFont(MainWindow.FIELD_FONT);
	}

	/**
	 * build and initialize the screen's components 
     */
    private final void initComponents() {
        jPanel1 = new JPanel();
		applyButton = new JButton("Apply");
        addMenuItem = new JMenuItem();
        copyMenuItem = new JMenuItem();
        deleteMenuItem = new JMenuItem();
        submitMenuItem = new JMenuItem();
        label1 = new JLabel();
        oppName = new JTextField();
        custLabel = new JLabel();
        custName = new JTextField();
        tableLabel = new JLabel();
		forecastsTablePanel.setMinimumSize(new Dimension(700,220));
		forecastsTablePanel.setPreferredSize(new Dimension(700,220));
		forecastsTablePanel.setMaximumSize(new Dimension(700,220));
        tableScrollPane = new JScrollPane();
		tableScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
		tableScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
		tableScrollPane.setMinimumSize(new Dimension(650,180));
		tableScrollPane.setPreferredSize(new Dimension(650,180));
		tableScrollPane.setMaximumSize(new Dimension(650,180));


        setLayout(new java.awt.BorderLayout());

        addMenuItem.setText("Add");
        copyMenuItem.setText("Copy");
        deleteMenuItem.setText("Delete");
        submitMenuItem.setText("Submit");

		jPanel1.add(applyButton);
        add(jPanel1, java.awt.BorderLayout.SOUTH);

        mainPanel.setLayout(new java.awt.GridBagLayout());
        java.awt.GridBagConstraints gridBagConstraints1;

        mainPanel.setBorder(new javax.swing.border.EtchedBorder());
        label1.setText("Opportunity");
        label1.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 0;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(label1, gridBagConstraints1);

        oppName.setText("");
		oppName.setMinimumSize(new Dimension(180,Prefs.FIELD_HEIGHT));
		oppName.setPreferredSize(new Dimension(180,Prefs.FIELD_HEIGHT));
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 1;
        gridBagConstraints1.gridy = 0;
        gridBagConstraints1.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        mainPanel.add(oppName, gridBagConstraints1);

        custLabel.setText("Customer");
        custLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 1;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(custLabel, gridBagConstraints1);

        custName.setText("");
		custName.setMinimumSize(new Dimension(180,Prefs.FIELD_HEIGHT));
		custName.setPreferredSize(new Dimension(180,Prefs.FIELD_HEIGHT));
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 1;
        gridBagConstraints1.gridy = 1;
        gridBagConstraints1.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        mainPanel.add(custName, gridBagConstraints1);


        tableScrollPane.setViewportView(table);

        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 4;
        gridBagConstraints1.gridwidth = 3;
        gridBagConstraints1.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        mainPanel.add(tableScrollPane, gridBagConstraints1);

        add(mainPanel, java.awt.BorderLayout.CENTER);

    }

	/**
	 * initialize the scenario column's renderers and
	 * editors of the table 
	 */
	private void initializeScenarioColumn() {
		TableColumnModel colModel = table.getColumnModel();
		TableColumn col = colModel.getColumn(1);
		col.setCellEditor(new DefaultCellEditor(scenarios));
		DefaultTableCellRenderer renderer = new DefaultTableCellRenderer();
		col.setCellRenderer(renderer);
	}

	/**
	 * initialize the date column - close date
	 */
	private final void initializeDateColumn() {
        final JButton button = new JButton();

        final DateEditor editor = new DateEditor(button, table);
        table.setDefaultEditor(Date.class, editor);

        //pass in the editor so that we can pass back the entered date
        final CalendarDialog editDialog = new CalendarDialog(MainWindow.getMainParent(), editor, true);

        //code that brings up the date editor dialog
        button.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                editDialog.setDate(editor.getDate());
                editDialog.show();
                button.setText(Prefs.dateTimeFormat.format(editor.getDate()));
            }
        });

        table.setDefaultRenderer(Date.class, new DateTimeRenderer());
    }


	/**
	 * enable or disable fields based upon the security
	 * settings
	 */
	private final void setSecurity() {
		try {
			if (wb.getLogin().forecastWritable()) {
    			applyButton.setEnabled(true);
	    		addMenuItem.setEnabled(true);
   	 			copyMenuItem.setEnabled(true);
   	 			deleteMenuItem.setEnabled(true);
    			submitMenuItem.setEnabled(true);
			} else {
    			applyButton.setEnabled(false);
	    		addMenuItem.setEnabled(false);
   		 		copyMenuItem.setEnabled(false);
   		 		deleteMenuItem.setEnabled(false);
   		 		submitMenuItem.setEnabled(false);
			}
		} catch (AngError e) {
			ErrorHandler.show(thisPanel, e);
		}
	}

	/**
	 * clear the screen's components 
	 */
	public final void clearOpportunity() {
		Utility.strSet(custName, null);
		Utility.strSet(oppName, null);
		ArrayList rows = new ArrayList();
		model = new ForecastTableModel(rows);
		table.setModel(model);
		initializeScenarioColumn();
	}

	/**
	 * set the screen's language
	 */
	public final void setLang() {
		model.setLang();
        addMenuItem.setText(wb.getLang().getString("add"));
        copyMenuItem.setText(wb.getLang().getString("copy"));
        deleteMenuItem.setText(wb.getLang().getString("delete"));
        submitMenuItem.setText(wb.getLang().getString("submit"));
        label1.setText(wb.getLang().getString("opportunity"));
        custLabel.setText(wb.getLang().getString("customer"));
		applyButton.setText(wb.getLang().getString("apply"));
	}

	private final void initTableColumnSize() {
        TableColumn col;

		for (int i=0;i<ForecastTableModel.columnNames.length;i++) {
        	col = table.getColumnModel().getColumn(i);
        	col.setPreferredWidth(100);
		}
	}

}
