package sellwin.gui;

import sellwin.domain.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the dialog that shows the
 * sales leads and lets users convert leads to 
 * opportunities.  This dialog is visible when a
 * user selects the leads menu item from the Admin
 * menu pull down from the main window.
 */
public class LeadsDialog extends JDialog implements GUIChars {
	private int currentLeadIndex, currentCampaignIndex;
	private Whiteboard wb;
	private JScrollPane scrollPane, scroller;
	
	private ArrayList camps; //campaigns
	private ArrayList campaignLeads; //leads for the currently selected campaign

	private JDialog thisDialog;
	private final static int STRUT_LEN = 3;

	private JLabel label1 = new JLabel("Campaigns:");
	private JTable campaigns = new JTable();
	private JButton detailsButton = new JButton("Show Leads");
	private JButton closeButton = new JButton("Close");
	private JLabel label2 = new JLabel("Leads:");
	private JTable leads = new JTable();
	private JButton makeButton = new JButton("Convert to Opportunity");
	private CampaignTableModel campaignModel;
	private LeadTableModel leadsModel;

	/**
	 * construct a leads dialog 
	 */
	public LeadsDialog() {
		super();

		thisDialog = this;
		wb = MainWindow.getWhiteboard();
		setTitle(wb.getLang().getString("salesLeads"));
		setFonts();
		setColors();


		getContentPane().setLayout(new BorderLayout());

		GridBagLayout gbl = new GridBagLayout();
		GridBagConstraints gbc = new GridBagConstraints();
		JPanel mainPanel = new JPanel();
		mainPanel.setBorder(new EtchedBorder());
		mainPanel.setLayout(gbl);

		mainPanel.add(Box.createHorizontalStrut(STRUT_LEN));
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbc.anchor = GridBagConstraints.WEST;
		scrollPane = new JScrollPane(campaigns);
		scrollPane.setBorder(new TitledBorder("Campaigns"));
		scrollPane.setPreferredSize(new Dimension(600,125));
		scrollPane.setMinimumSize(new Dimension(600,125));
		mainPanel.add(scrollPane, gbc);

		mainPanel.add(Box.createHorizontalStrut(STRUT_LEN));
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbc.anchor = GridBagConstraints.WEST;
		scroller = new JScrollPane(leads);
		scroller.setPreferredSize(new Dimension(600,125));
		scroller.setMinimumSize(new Dimension(600,125));
		scroller.setBorder(new TitledBorder("Leads"));
		mainPanel.add(scroller, gbc);

		JPanel buttonPanel = new JPanel();
		buttonPanel.add(makeButton);
		buttonPanel.add(detailsButton);
		buttonPanel.add(closeButton);

		getContentPane().add(mainPanel, BorderLayout.CENTER);
		getContentPane().add(buttonPanel, BorderLayout.SOUTH);

		makeButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (campaigns.getSelectedRow() < 0) {
						JOptionPane.showMessageDialog(
							thisDialog, "Please select a campaign first.",
							"Error", JOptionPane.INFORMATION_MESSAGE);
							return;
					}
					if (leads.getSelectedRow() < 0) {
						JOptionPane.showMessageDialog(
							thisDialog, "Please select a lead first.",
							"Error", JOptionPane.INFORMATION_MESSAGE);
							return;
					}
					convert();

				}
			}
		);

		detailsButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					loadLeads(currentCampaignIndex);
				}
			}
		);

		closeButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					setVisible(false);
        			dispose();
				}
			}
		);

		campaigns.setPreferredScrollableViewportSize(
			new Dimension(500,70));
		campaigns.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		campaigns.getSelectionModel().addListSelectionListener(
			new ListSelectionListener() {
				public void valueChanged(ListSelectionEvent e) {
					if (e.getValueIsAdjusting()) return; //ignore these
                    ListSelectionModel lsm = (ListSelectionModel)e.getSource();

                    if (!lsm.isSelectionEmpty())
                         currentCampaignIndex = lsm.getMinSelectionIndex();
				}
			}
		);

		leads.setPreferredScrollableViewportSize(
			new Dimension(500,70));
		leads.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		leads.getSelectionModel().addListSelectionListener(
			new ListSelectionListener() {
				public void valueChanged(ListSelectionEvent e) {
	  				if (e.getValueIsAdjusting()) return; //ignore these
                    ListSelectionModel lsm = (ListSelectionModel)e.getSource();

                    if (!lsm.isSelectionEmpty())
                        currentLeadIndex = lsm.getMinSelectionIndex();
				}
			}
		);

		getRootPane().setDefaultButton(makeButton);
		campaignModel = new CampaignTableModel(new ArrayList());
		campaigns.setModel(campaignModel);
		leadsModel = new LeadTableModel(new ArrayList());
		leads.setModel(leadsModel);
		setLang();
		setSize(661,320);
	}

	/**
	 * load the campaigns into this dialog 
	 */
	public final void loadCampaigns() {
		try {
			camps = wb.getCampaigns();
			if ((camps == null) || (camps.size() <= 0)) return;
			campaignModel = new CampaignTableModel(camps);
			campaigns.setModel(campaignModel);
			if (camps.size() > 0)
				campaigns.setRowSelectionInterval(0,0);
		} catch (AngError e) {
			ErrorHandler.show(thisDialog, e);
		}
	}

	/**
	 * load the leads for a given campaign into
	 * this dialog
	 * @param campIndex the index of the campaign to
	 * find leads for
	 */
	public final void loadLeads(int campIndex) {
		Campaign c = (Campaign)camps.get(campIndex);
		try {
			campaignLeads = wb.getCampaignLeads(c.getPK());
			if ((campaignLeads == null) || (campaignLeads.size() <= 0)) return;
			leadsModel = new LeadTableModel(campaignLeads);
			leads.setModel(leadsModel);
		} catch (AngError e) {
			ErrorHandler.show(thisDialog, e);
		}
	}

	/**
	 * disable or enable all the components on the screen  
	 * @param true if enable
	 */
	public final void setEnabled(boolean f) {
        Component[] children = getComponents();
        for (int i=0;i<children.length;i++)
            children[i].setEnabled(f);
	}

	/**
	 * convert a lead into an opportunity 
	 */
	public final void convert() {
		CampaignTableModel model = (CampaignTableModel)(campaigns.getModel());	
		Campaign camp =  model.getCampaign(campaigns.getSelectedRow());
		Lead l = (Lead)campaignLeads.get(currentLeadIndex);
		if (l.getConverted().booleanValue()) {	
			JOptionPane.showMessageDialog(
				thisDialog, 
				"Lead already converted, please select another campaign.",
				"Error", JOptionPane.INFORMATION_MESSAGE);
			return;
		}

		String val = JOptionPane.showInputDialog(
			wb.getLang().getString("enterOpp"));

		Opportunity newOpp = new Opportunity(val, wb.getCurrentUser().getID());
		newOpp.setLeadPK(l.getPK());
		newOpp.setStage(Opportunity.STAGE_IDENTIFIED);
		newOpp.setPrimeSalesPerson(l.getContactedBy());
		Customer cust = new Customer();
		cust.setName(l.getCompany());
		Address contact = l.getAddress();
		cust.setContact(contact);
		newOpp.setCustomer(cust);	
		try {
			ArrayList groups = wb.getGroupsForUser();
			UserGroup ug;
			ArrayList groupNames = new ArrayList();
			for (int x=0;x<groups.size();x++) {
				ug = (UserGroup)groups.get(x);
				groupNames.add(ug.getName());
			}
			Object[] groupNamesList = groupNames.toArray();
			String gName = (String)JOptionPane.showInputDialog(
				thisDialog,
				wb.getLang().getString("chooseGroup"),
				wb.getLang().getString("selectGroup"),
				JOptionPane.PLAIN_MESSAGE,
				null,
				groupNamesList,
				groupNamesList[0]);
			newOpp.setGroupName(gName);
				
			newOpp.setPK(wb.addOpportunity(newOpp));
			l.setConverted(new Boolean(true));
			LeadTableModel leadsModel = (LeadTableModel)(leads.getModel());
			leads.tableChanged(new TableModelEvent(leadsModel));
			wb.updateLead(camp.getPK(), l);
			wb.setCurrentOpportunity(newOpp); //make the new opp show
			MainWindow.refreshOpportunity();
		} catch (Exception e) {
			e.printStackTrace();
		}
 		JOptionPane.showMessageDialog(
        	thisDialog,
            "Lead successfully converted to Opportunity!",
            "Success",
            JOptionPane.INFORMATION_MESSAGE);
	}

	/**
	 * set the dialog's colors 
	 */
	public final void setColors() {
		label1.setForeground(Color.black);
		label2.setForeground(Color.black);
	}

	/**
	 * set the screen's fonts 
	 */
	public final void setFonts() {
		label1.setFont(MainWindow.LABEL_FONT);
		campaigns.setFont(MainWindow.FIELD_FONT);
		detailsButton.setFont(MainWindow.LABEL_FONT);
		closeButton.setFont(MainWindow.LABEL_FONT);
		label2.setFont(MainWindow.LABEL_FONT);
		leads.setFont(MainWindow.FIELD_FONT);
		makeButton.setFont(MainWindow.LABEL_FONT);
	}

	/**
	 * set the screen's language
	 */
	public final void setLang() {
		campaignModel.setLang();
		leadsModel.setLang();
		setTitle(wb.getLang().getString("salesLeads"));
		scrollPane.setBorder(
			new TitledBorder(
				null,
				wb.getLang().getString("campaigns"),
				TitledBorder.DEFAULT_JUSTIFICATION,
				TitledBorder.DEFAULT_POSITION,
				MainWindow.LABEL_FONT,
				MainWindow.LETTERS
			)
		);
		closeButton.setText(wb.getLang().getString("close"));
		detailsButton.setText(wb.getLang().getString("showLeads"));
		makeButton.setText(wb.getLang().getString("convert"));
		scroller.setBorder(
			new TitledBorder(
				null,
				wb.getLang().getString("leads"),
				TitledBorder.DEFAULT_JUSTIFICATION,
				TitledBorder.DEFAULT_POSITION,
				MainWindow.LABEL_FONT,
				MainWindow.LETTERS
			)
		);
	}
}
