package sellwin.gui;

import sellwin.domain.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.text.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class is the viewable part of the sales
 * funnel, polygons are used to draw the funnel, and
 * each polygon is clickable.
 */
public class PolyPanel extends JPanel implements MouseListener {
	
	private NumberFormat numFormat = new DecimalFormat("##0.00");
	private OppListDialog oppListDialog = new OppListDialog();

	private Whiteboard wb=null;

	private int currentSelection = 0;

	private static final int MAX=8;

	private static final int X_OFF = 10, Y_OFF=-50;
	private static final int TEXT_OFFSET_Y = 14, TEXT_OFFSET_X=7;

	private ArrayList oppIndex = null;
	private int totalOpps = 0;
	private int[] oppCounts = new int[MAX];

	private String[] str = {
        Opportunity.STAGE_IDENTIFIED,
        Opportunity.STAGE_CONTACTED,
        Opportunity.STAGE_BIDDING,
        Opportunity.STAGE_CHANCE,
        Opportunity.STAGE_WINNING,
        Opportunity.STAGE_AWARDED,
        Opportunity.STAGE_CONTRACT,
        Opportunity.STAGE_BILLED };

	private Color[] colors = {
			Color.blue,
			Color.cyan,
			Color.lightGray,
			Color.green,
			Color.magenta,
			Color.orange,
			Color.pink,
			Color.red
		};

	private MyPolygon[] polys = {
			new MyPolygon(
				new Point(121 + X_OFF,164 + Y_OFF),	
				new Point(489 + X_OFF,164 + Y_OFF),
				new Point(478 + X_OFF,214 + Y_OFF),
				new Point(134 + X_OFF,214 + Y_OFF)),
			new MyPolygon(
				new Point(134 + X_OFF,214 + Y_OFF),	
				new Point(478 + X_OFF,214 + Y_OFF),
				new Point(465 + X_OFF,263 + Y_OFF),
				new Point(147 + X_OFF,263 + Y_OFF)),
			new MyPolygon(
				new Point(147 + X_OFF,263 + Y_OFF),	
				new Point(465 + X_OFF,263 + Y_OFF),
				new Point(454 + X_OFF,312 + Y_OFF),
				new Point(160 + X_OFF,312 + Y_OFF)),
			new MyPolygon(
				new Point(160 + X_OFF,312 + Y_OFF),	
				new Point(454 + X_OFF,312 + Y_OFF),
				new Point(442 + X_OFF,360 + Y_OFF),
				new Point(172 + X_OFF,360 + Y_OFF)),
			new MyPolygon(
				new Point(172 + X_OFF,360 + Y_OFF),	
				new Point(442 + X_OFF,360 + Y_OFF),
				new Point(429 + X_OFF,409 + Y_OFF),
				new Point(185 + X_OFF,409 + Y_OFF)),
			new MyPolygon(
				new Point(185 + X_OFF,409 + Y_OFF),	
				new Point(429 + X_OFF,409 + Y_OFF),
				new Point(418 + X_OFF,457 + Y_OFF),
				new Point(197 + X_OFF,457 + Y_OFF)),
			new MyPolygon(
				new Point(197 + X_OFF,457 + Y_OFF),	
				new Point(418 + X_OFF,457 + Y_OFF),
				new Point(406 + X_OFF,505 + Y_OFF),
				new Point(210 + X_OFF,505 + Y_OFF)),
			new MyPolygon(
				new Point(210 + X_OFF,505 + Y_OFF),	
				new Point(406 + X_OFF,505 + Y_OFF),
				new Point(393 + X_OFF,553 + Y_OFF),
				new Point(223 + X_OFF,553 + Y_OFF))
		};

	
		
	/**
	 * construct the polygon panel
	 */
	public PolyPanel() 
		throws AngError {

		this.addMouseListener(this);

		refresh();

		setSize(400,400);

	}

	/**
	 * refresh the opps
	 */
	public final void refresh() {
		wb = MainWindow.getWhiteboard();
		try {
			oppIndex = wb.getOpportunityIndex();	
			totalOpps = oppIndex.size();
			countOpps();
			this.repaint();
		} catch (AngError e) {
			ErrorHandler.show(this, e);
		}
	}

	/** part of the MouseListener interface */
	public void mouseEntered(MouseEvent e) { }

	/** part of the MouseListener interface */
	public void mouseExited(MouseEvent e) { }

	/** part of the MouseListener interface */
	public void mouseReleased(MouseEvent e) { }

	/** part of the MouseListener interface */
	public void mouseClicked(MouseEvent e) { }

	/** part of the MouseListener interface */
	public final void mousePressed(MouseEvent e) {

		for (int i=0;i<8;i++) {
			if (polys[i].contains(e.getX(), e.getY())) {
				currentSelection = i;
				showList(i);
				this.repaint();
				break;
			}
		}
	
	}

	/**
	 * show the opportunities using the OppListDialog's
	 * contents 
	 * @param i description
	 */
	private final void showList(int i) {
		ArrayList opps = new ArrayList();
		OppIndex oi;
		for (int j=0;j<oppIndex.size();j++) {
			oi = (OppIndex)(oppIndex.get(j));
			if (oi.getStage().equals(str[i]))
				opps.add(oi.getName());	
		}
	
		oppListDialog.setData(opps.toArray());
		oppListDialog.show();
	}

	/**
	 * redraw the funnel polygons 
	 * @param g the graphics context to draw with
	 */
	public final void paintComponent(Graphics g) {
		super.paintComponent(g);
	
		String str = null;

		g.setFont(MainWindow.LABEL_FONT);
	
		//draw polys
		for (int i=0;i<8;i++) {
			g.drawPolygon(polys[i]);
			if (i == currentSelection)
				g.setColor(Color.white);
			else
				g.setColor(colors[i]);
			g.fillPolygon(polys[i]);
			g.setColor(Color.black);
			str = buildStat(i);
			g.drawString(str, polys[i].p1.x + TEXT_OFFSET_X, polys[i].p1.y + TEXT_OFFSET_Y);
		}
	}

	/**
	 * figure the statistics for each polygon and
	 * construct a string to be drawn in 
	 * @param i the number of status to 
	 * @return the formatted stat string
	 */
	private final String buildStat(int i) {
		double ax, bx;


		if (totalOpps <= 0) return str[i] + " 0 - 0%";

		ax = oppCounts[i] * 1.00;
		bx = totalOpps * 1.00;

		double pct = (ax/bx) * 100.00;
		return str[i] + " (" + oppCounts[i] + ")" +  " " + numFormat.format(pct) + "%";
	}

	/**
	 * count all the opportunities for all stages	
	 */
	private final void countOpps() {
		String stage;
		for (int i=0;i<MAX;i++) {
			stage = str[i];
			oppCounts[i] = countStage(stage);
		}
	}

	/**
	 * count all the opportunities for a given stage
	 * @param stage the stage to count
	 * @return the number of stages
	 */
	private final int countStage(String stage) {
		int cnt=0;
		OppIndex opp;
		for (int i=0;i<oppIndex.size();i++) {
			opp = (OppIndex)(oppIndex.get(i));
			if (opp.getStage().equals(stage))
				cnt++;
		}
		return cnt;
	}
}

/**
 * This class represents each polygon that makes up the 
 * sales funnel
 * 
 * @author Jeff McCormick
 */
class MyPolygon extends Polygon {
	public Point p1, p2, p3, p4;

	/** construct the polygon using 4 points 
	* @param p1 point 1
	* @param p2 point 2
	* @param p3 point 3
	* @param p4 point 4
	*/
	public MyPolygon(Point p1, Point p2, Point p3, Point p4) {
		addPoint(p1.x, p1.y);
		addPoint(p2.x, p2.y);
		addPoint(p3.x, p3.y);
		addPoint(p4.x, p4.y);
	
		this.p1 = p1;
		this.p2 = p2;
		this.p3 = p3;
		this.p4 = p4;
	}
}
