package sellwin.gui;

import sellwin.domain.*;
import sellwin.utils.*;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import javax.swing.table.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class is the panel that is shown when
 * the user selects the Quotes tab from the MainWindow
 * This panel shows all the quotes for a given Opportunity
 * and the quote details.
 */
public class QuotesPanel extends JPanel 
	implements InventoryListener,TableModelListener, GUIChars {

	private QuotesPanel thisPanel = null;
	private JPopupMenu quotePopup = new JPopupMenu();
	private JPopupMenu linePopup = new JPopupMenu();

	private ArrayList quotes; //reference to current opp's quotes

	private ProductSelectionDialog addProductSelectionDialog;
	private Whiteboard wb;
	private Opportunity currentOpp;
	
	private QuoteTableModel quoteModel = null;
	private QuoteLineTableModel lineModel = null;

    private JPanel buttonPanel;
    private JMenuItem addQuoteMenuItem;
    private JMenuItem deleteQuoteMenuItem;
    private JMenuItem addLineMenuItem;
    private JMenuItem deleteLineMenuItem;
    private JButton orderButton;
    private JButton applyButton;

    private JPanel mainPanel;
    private JScrollPane quotesScrollPane;
    private final javax.swing.JTable quotesTable = new JTable();
    private JLabel lineItemsLabel;
	private TitledBorder lineItemBorder;
    private JLabel quotesLabel;

	private JPanel linesPanel = new JPanel();
    private JScrollPane linesScrollPane;
    private final javax.swing.JTable linesTable = new JTable();

    private JLabel totalLabel;
    private JTextField totalField;

    private JLabel profitLabel;
    private JTextField profitField;
	
	/**
	 * construct a quote panel
	 */
	public QuotesPanel() {
		super();

		thisPanel = this;
		wb = MainWindow.getWhiteboard();
        initComponents();

		setFonts();
		setColors();

   		quoteModel = new QuoteTableModel(new ArrayList());
        quotesTable.setModel(quoteModel);
        lineModel = new QuoteLineTableModel(new ArrayList());
        linesTable.setModel(lineModel);

		setLang();

		linesTable.setPreferredScrollableViewportSize(new Dimension(500,150));
		linesTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		linesTable.getSelectionModel().addListSelectionListener(
			new ListSelectionListener() {
				public void valueChanged(ListSelectionEvent e) {
					if (e.getValueIsAdjusting()) return; //ignore these
					ListSelectionModel lsm = (ListSelectionModel)e.getSource();
				}
			}
		);

		quotesTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		quotesTable.getSelectionModel().addListSelectionListener(
			new ListSelectionListener() {
				public void valueChanged(ListSelectionEvent e) {

					if (e.getValueIsAdjusting()) return; //ignore these

					ListSelectionModel lsm = (ListSelectionModel)e.getSource();

					if (lsm.isSelectionEmpty())
						;
					else {
                    	Quote quote = quoteModel.getQuote(quotesTable.getSelectedRow());
						setLines(quote.getLines());
						
					}
				}
			}
		);

		quotesTable.setDefaultRenderer(Date.class, new DateRenderer());

      	addQuoteMenuItem.addActionListener(
            new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					Opportunity opp = wb.getCurrentOpportunity();
					if (opp == null) {
						JOptionPane.showMessageDialog( thisPanel,
             				wb.getLang().getString("selectOpp"),
                            wb.getLang().getString("error"),
                       		JOptionPane.INFORMATION_MESSAGE);
						return;
					}
   					String val=JOptionPane.showInputDialog("Please enter a Quote name");
                    Quote quote = new Quote();
					quote.setName(val);
                    try {
						quote.setModifiedBy(wb.getCurrentUser().getAddress().getFormattedName());
                        quote.setPK(wb.addQuote(opp.getPK(), quote));
						opp.addQuote(quote);
						quotesTable.tableChanged(new TableModelEvent(quoteModel));
						int quoteCt = opp.getQuotes().size();
						if (quoteCt > 0)
							quotesTable.setRowSelectionInterval(quoteCt-1,quoteCt-1);
                    } catch (Exception f) {
						ErrorHandler.show(thisPanel, f);
                    }
                }
            }
        );

      	deleteQuoteMenuItem.addActionListener(
            new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    Quote quote = quoteModel.getQuote(quotesTable.getSelectedRow());
                    try {
						Opportunity opp = wb.getCurrentOpportunity();
                        wb.deleteQuote(opp.getPK(), quote.getPK());
						opp.deleteQuote(quote);
						quotesTable.tableChanged(new TableModelEvent(quoteModel));
                    } catch (Exception f) {
						ErrorHandler.show(thisPanel, f);
                    }
                }
            }
        );


      	addLineMenuItem.addActionListener(
            new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					if (addProductSelectionDialog == null)
						addProductSelectionDialog = new ProductSelectionDialog(thisPanel);
					addProductSelectionDialog.show();
                }
            }
        );

      	deleteLineMenuItem.addActionListener(
            new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    Quote quote = quoteModel.getQuote(quotesTable.getSelectedRow());
                    QuoteLine line = lineModel.getQuoteLine(linesTable.getSelectedRow());
                    try {
						Opportunity opp = wb.getCurrentOpportunity();
                        wb.deleteQuoteLine(opp.getPK(), quote.getPK(), line.getPK());
						quote.deleteLine(line.getPK());
						linesTable.tableChanged(new TableModelEvent(lineModel));
						calcTotal();
						//double total = Double.parseDouble(totalField.getText());
						//total -= line.getTotalPrice().doubleValue();
						//totalField.setText(Prefs.money.format(total));
                    } catch (Exception f) {
						ErrorHandler.show(thisPanel, f);
                    }
                }
            }
        );

		createPopup();

   		quotesTable.addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    quotePopup.show(quotesTable, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    quotePopup.show(quotesTable, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    quotePopup.show(quotesTable, e.getX(), e.getY());
            }
        });

   		quotesTable.getTableHeader().addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    quotePopup.show(quotesTable, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    quotePopup.show(quotesTable, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    quotePopup.show(quotesTable, e.getX(), e.getY());
            }
        });
   		quotesScrollPane.addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    quotePopup.show(quotesTable, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    quotePopup.show(quotesTable, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    quotePopup.show(quotesTable, e.getX(), e.getY());
            }
        });


   		linesScrollPane.addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    linePopup.show(linesTable, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    linePopup.show(linesTable, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    linePopup.show(linesTable, e.getX(), e.getY());
            }
        });
   		linesTable.addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    linePopup.show(linesTable, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    linePopup.show(linesTable, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    linePopup.show(linesTable, e.getX(), e.getY());
            }
        });

   		linesTable.getTableHeader().addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    linePopup.show(linesTable, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    linePopup.show(linesTable, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    linePopup.show(linesTable, e.getX(), e.getY());
            }
        });

      	orderButton.addActionListener(
            new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					int opt = JOptionPane.showConfirmDialog(thisPanel, "Select 'Yes' to confirm order.", "Confirm Order Creation", JOptionPane.YES_NO_OPTION);
					if (opt == JOptionPane.YES_OPTION)
						createOrder();	
                }
            }
        );

      	applyButton.addActionListener(
            new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					ArrayList quotes = wb.getCurrentOpportunity().getQuotes();
					Quote quote;
					for (int i=0;i<quotes.size();i++) {
						quote = (Quote)quotes.get(i);
						if (quote.getModified() == true) {
							try {
								wb.updateQuote(quote);
							} catch (AngError r) {
								ErrorHandler.show(thisPanel, r);
							}
						}
					}
					JOptionPane.showMessageDialog( thisPanel,
						"Update Successful", "Update Successful",
                       	JOptionPane.INFORMATION_MESSAGE);
                }
            }
        );


	}

	/**
	 * create an order when the user presses the
	 * create order button
	 */
	private final void createOrder() {
		Opportunity opp = wb.getCurrentOpportunity();
		Order order = new Order();

		// 1) get the current selected quote and make an order with it
		Quote quote = quoteModel.getQuote(quotesTable.getSelectedRow());
		order.setQuoteKey(quote.getPK());
		order.setQuote(quote);
		//for now, just make the order number some big unique number
		Date temp = new java.util.Date();
		order.setOrderNumber(new Long(temp.getTime()));
		//order.setSubmitDate(temp);

		// 2) get the ship & bill-to addresses from the current customer
		//    these can be later changed by the user if they wish
		Customer currCust = opp.getCustomer();
		order.setShipAddress(currCust.getShipAddress());
		System.out.println("ship addr");
		currCust.getShipAddress().print();
		order.setBillAddress(currCust.getBillAddress());
		System.out.println("bill addr");
		currCust.getBillAddress().print();

		try {

			// 3) add it to database
			order.setPK(
				wb.addOrder(order));

			// 4) add it to memory
			opp.addOrder(order); 

			// 5) refresh the Orders in the Orders Panel
			OrderFulfillPanel p = MainWindow.getOrdersPanel();
			p.refreshOpportunity();

			// 6) make the order the one that is initially selected
			p.setSelectedOrder(order);

			// 7) go to the Orders tab
			MainWindow.selectOrderTab();

		} catch (AngError e) {
			ErrorHandler.show(thisPanel, e);
		}
	}

	/**
	callback when the ProductSelectionDialog user selects a Product to
	be added as a QuoteLine
	 * @param q the QuoteLine to add
	 */
	public final void addQuoteLine(QuoteLine q) {
		Quote quote = quoteModel.getQuote(quotesTable.getSelectedRow());
		int lineCt = quote.getLines().size();
		try {
			wb.addQuoteLine(
				wb.getCurrentOpportunity().getPK(),
				quote.getPK(),
				q);
			quote.addLine(q);
			linesTable.tableChanged(new TableModelEvent(lineModel));
			linesTable.setRowSelectionInterval(lineCt, lineCt);
			calcTotal();
			//double total = Double.parseDouble(totalField.getText());
			//total +=q.getTotalPrice().doubleValue();
			//totalField.setText(Prefs.money.format(total));
		} catch (AngError e) {
			ErrorHandler.show(thisPanel, e);
		}
	}

	/**
	 *  create the popup menu
	 */
	private final void createPopup() {
		linePopup.add(addLineMenuItem);
		linePopup.add(deleteLineMenuItem);
		quotePopup.add(addQuoteMenuItem);
		quotePopup.add(deleteQuoteMenuItem);
	}

	/**
	 * 
	 * @param name description
	 * @return description
	 * @exception class-name description
	 */
	public final void setEnabled(boolean t) {
	}

	/**
	 * refresh the screen with the current
	 * opportunity
	 */
	public final void refreshOpportunity() {

		setSecurity();

		currentOpp = wb.getCurrentOpportunity();

		Quote quote=null;
		ArrayList quotes = currentOpp.getQuotes();
		for (int i=0;i<quotes.size();i++) {
			quote = (Quote)(quotes.get(i));
		}
   		quoteModel = new QuoteTableModel(currentOpp.getQuotes());
        quotesTable.setModel(quoteModel);
        ArrayList lines = null;
        Quote firstQuote = null;
        if (currentOpp.getQuotes().size() > 0) {
            firstQuote = (Quote)(currentOpp.getQuotes().get(0));
            lines = firstQuote.getLines();
			quotesTable.setRowSelectionInterval(0,0);
        } else
            lines = new ArrayList();

		setLines(lines);

		linesTable.setDefaultRenderer(Double.class, new MoneyRenderer());

	}

	/**
	 * set the product lines on the screen
	 * @param lines the product liens to display
	 */
	private final void setLines(ArrayList lines) {
        lineModel = new QuoteLineTableModel(lines);
		lineModel.addTableModelListener(this);
        linesTable.setModel(lineModel);
		setLinesColumnWidths();
		if (lines.size() > 0)
			linesTable.setRowSelectionInterval(0,0);

		calcTotal();
		//double total = 0.00;
		//QuoteLine line;
		//for (int i=0;i<lines.size();i++) {
			//line = (QuoteLine)(lines.get(i));
			//total += line.getTotalPrice().doubleValue();
		//}

    	//totalField.setText(Prefs.money.format(total));
		
	}

	/**
	 *  set the quote lines on the screen
	 * @param name description
	 * @return description
	 * @exception class-name description
	 */
	public final void setDisplayedQuoteLines(int index) {
		lineModel = new QuoteLineTableModel(((Quote)(quotes.get(index))).getLines());
		lineModel.addTableModelListener(this);
		linesTable.setModel(lineModel);
		setLinesColumnWidths();
	}

	/**
	 * set the screen color 
	 */
	public final void setColors() {
		quotesLabel.setForeground(MainWindow.LETTERS);
		lineItemsLabel.setForeground(MainWindow.LETTERS);
        totalLabel.setForeground(MainWindow.LETTERS);
        profitLabel.setForeground(MainWindow.LETTERS);
	}

	/**
	 * set the screen fonts 
	 */
	public final void setFonts() {
		quotesLabel.setFont(MainWindow.LABEL_FONT);
		lineItemsLabel.setFont(MainWindow.LABEL_FONT);
		linesTable.setFont(MainWindow.FIELD_FONT);
		quotesTable.setFont(MainWindow.FIELD_FONT);
		addQuoteMenuItem.setFont(MainWindow.LABEL_FONT);
		deleteQuoteMenuItem.setFont(MainWindow.LABEL_FONT);
		addLineMenuItem.setFont(MainWindow.LABEL_FONT);
		deleteLineMenuItem.setFont(MainWindow.LABEL_FONT);
        totalLabel.setFont(MainWindow.LABEL_FONT);
        profitLabel.setFont(MainWindow.LABEL_FONT);
        orderButton.setFont(MainWindow.LABEL_FONT);
        applyButton.setFont(MainWindow.LABEL_FONT);
	}

    /** This method is called from within the constructor to
     * initialize the form.
	 */
    private final void initComponents() {
        buttonPanel = new JPanel();
        addQuoteMenuItem = new JMenuItem();
        deleteQuoteMenuItem = new JMenuItem();
        addLineMenuItem = new JMenuItem();
        deleteLineMenuItem = new JMenuItem();
        orderButton = new JButton();
        applyButton = new JButton();
        mainPanel = new JPanel();
        quotesScrollPane = new JScrollPane();
        lineItemsLabel = new JLabel();
        quotesLabel = new JLabel();
        linesScrollPane = new JScrollPane();
        totalLabel = new JLabel();
        totalField = new JTextField();
		totalField.setEditable(false);
        profitLabel = new JLabel();
        profitField = new JTextField();
		profitField.setEditable(false);
        
        setLayout(new java.awt.BorderLayout());
        
        addQuoteMenuItem.setText("Add");
        deleteQuoteMenuItem.setText("Delete");
        addLineMenuItem.setText("Add");
        deleteLineMenuItem.setText("Delete");
        
        orderButton.setText("Create Order");
        buttonPanel.add(orderButton);
        applyButton.setText("Apply");
        buttonPanel.add(applyButton);
        
        add(buttonPanel, java.awt.BorderLayout.SOUTH);
        
        mainPanel.setLayout(new java.awt.GridBagLayout());
        java.awt.GridBagConstraints gridBagConstraints1;
        
        mainPanel.setBorder(new javax.swing.border.EtchedBorder());

		//quotesScrollPane.setBorder(new TitledBorder("Quotes"));
        quotesScrollPane.setVerticalScrollBarPolicy(javax.swing.JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        quotesScrollPane.setHorizontalScrollBarPolicy(javax.swing.JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        quotesScrollPane.setMinimumSize(new Dimension(500, 100));
        quotesScrollPane.setMaximumSize(new Dimension(500, 100));
        quotesScrollPane.setPreferredSize(new Dimension(500, 100));
        quotesTable.setPreferredSize(new Dimension(500, 100));
        quotesTable.setMaximumSize(new Dimension(500, 100));
        quotesTable.setMinimumSize(new Dimension(500, 100));
        quotesTable.setAutoResizeMode(javax.swing.JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
        quotesScrollPane.setViewportView(quotesTable);
        
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 1;
        gridBagConstraints1.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints1.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints1.ipadx = 9;
        gridBagConstraints1.ipady = 9;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(quotesScrollPane, gridBagConstraints1);
        
       
	   	//lineItemBorder = new TitledBorder(null, "Line Items", TitledBorder.DEFAULT_JUSTIFICATION, TitledBorder.DEFAULT_POSITION, MainWindow.LABEL_FONT, MainWindow.LETTERS);
       	//linesScrollPane.setBorder(lineItemBorder);
        linesScrollPane.setVerticalScrollBarPolicy(javax.swing.JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        linesScrollPane.setHorizontalScrollBarPolicy(javax.swing.JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        linesScrollPane.setMinimumSize(new Dimension(500, 100));
        linesScrollPane.setMaximumSize(new Dimension(500, 100));
        linesScrollPane.setPreferredSize(new Dimension(500, 100));
        linesTable.setPreferredSize(new Dimension(500, 100));
        linesTable.setMinimumSize(new Dimension(500, 100));
        linesTable.setMaximumSize(new Dimension(500, 100));
        linesTable.setAutoResizeMode(javax.swing.JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
        linesScrollPane.setViewportView(linesTable);
        
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 3;
        gridBagConstraints1.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints1.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints1.ipadx = 9;
        gridBagConstraints1.ipady = 9;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        //mainPanel.add(linesScrollPane, gridBagConstraints1);
		linesPanel.add(linesScrollPane);
        mainPanel.add(linesPanel, gridBagConstraints1);
        
        totalLabel.setText("Total");
        totalLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 4;
        gridBagConstraints1.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints1.ipadx = 9;
        gridBagConstraints1.ipady = 9;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.EAST;
        mainPanel.add(totalLabel, gridBagConstraints1);
        
        totalField.setText("0.00");
        totalField.setMinimumSize(new Dimension(100, Prefs.FIELD_HEIGHT));
        totalField.setPreferredSize(new Dimension(100, Prefs.FIELD_HEIGHT));
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 1;
        gridBagConstraints1.gridy = 4;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(totalField, gridBagConstraints1);
        
        profitLabel.setText("Profit");
        profitLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 0;
        gridBagConstraints1.gridy = 5;
        gridBagConstraints1.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints1.ipadx = 9;
        gridBagConstraints1.ipady = 9;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.EAST;
        mainPanel.add(profitLabel, gridBagConstraints1);
        
        profitField.setText("0.00");
        profitField.setMinimumSize(new Dimension(100, Prefs.FIELD_HEIGHT));
        profitField.setPreferredSize(new Dimension(100, Prefs.FIELD_HEIGHT));
        gridBagConstraints1 = new java.awt.GridBagConstraints();
        gridBagConstraints1.gridx = 1;
        gridBagConstraints1.gridy = 5;
        gridBagConstraints1.insets = new Insets(Prefs.I_TOP,Prefs.I_LF,Prefs.I_BOTT,Prefs.I_RT);
        gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
        mainPanel.add(profitField, gridBagConstraints1);
        
        add(mainPanel, java.awt.BorderLayout.CENTER);
    }

	/**
	 * recalculate the total when the table has changed 
	 */
	public final void tableChanged(TableModelEvent e) {
		calcTotal();
	}

	/**
	 * calculate the total for this quote 
	 */
	public final void calcTotal() {
		//System.out.println("calcTotal called");
		double total=0.00;
		double profit=0.00;

		if (quotesTable.getSelectedRow() < 0)
			return;

		Quote quote = quoteModel.getQuote(quotesTable.getSelectedRow());
		quote.setModified(true);
		ArrayList lines = quote.getLines();
		QuoteLine line;
		for (int i=0;i<lines.size();i++) {
			line = (QuoteLine)lines.get(i);
			total = total + 
				(line.getProduct().getPrice().doubleValue() 
				* line.getQuantity().intValue()); 
			profit = profit +
				((line.getProduct().getPrice().doubleValue() -
				 line.getProduct().getCost().doubleValue()) 
				* line.getQuantity().doubleValue());
			//System.out.println("profit="+ profit);
		}
		totalField.setText(Prefs.money.format(total));
		profitField.setText(Prefs.money.format(profit));
		//System.out.println("calcTotal ends");
	}

	/** 
	 * enable fields according to security 
	 */
	private final void setSecurity() {
		try {
			if (wb.getLogin().quoteWritable()) {
    			addQuoteMenuItem.setEnabled(true);
    			deleteQuoteMenuItem.setEnabled(true);
    			addLineMenuItem.setEnabled(true);
    			deleteLineMenuItem.setEnabled(true);
			} else {
    			addQuoteMenuItem.setEnabled(false);
    			deleteQuoteMenuItem.setEnabled(false);
    			addLineMenuItem.setEnabled(false);
    			deleteLineMenuItem.setEnabled(false);
			}
		} catch (AngError e) {
			ErrorHandler.show(thisPanel, e);
		}
	}

	/**
	 * clear the screen components 
	 */
	public final void clearOpportunity() {

   		quoteModel = new QuoteTableModel(new ArrayList());
        quotesTable.setModel(quoteModel);
        ArrayList lines = new ArrayList();
		setLines(lines);
	}

	/**
	 * set the screen language
	 */
	public final void setLang() {
        addQuoteMenuItem.setText(wb.getLang().getString("add"));
        deleteQuoteMenuItem.setText(wb.getLang().getString("delete"));
        addLineMenuItem.setText(wb.getLang().getString("add"));
        deleteLineMenuItem.setText(wb.getLang().getString("delete"));
        orderButton.setText(wb.getLang().getString("createOrder"));
        applyButton.setText(wb.getLang().getString("apply"));
	   	linesPanel.setBorder(
			new TitledBorder(
				null, 
				wb.getLang().getString("lineItems"), 
				TitledBorder.DEFAULT_JUSTIFICATION, 
				TitledBorder.DEFAULT_POSITION, 
				MainWindow.LABEL_FONT, 
				MainWindow.LETTERS
			)
		);
        totalLabel.setText(wb.getLang().getString("total"));
        profitLabel.setText(wb.getLang().getString("profit"));
		quoteModel.setLang();
		lineModel.setLang();
	}

	private void setLinesColumnWidths() {
		TableColumn col = linesTable.getColumnModel().getColumn(0); //name
		col.setPreferredWidth(170);
		col = linesTable.getColumnModel().getColumn(1); //qty
		col.setPreferredWidth(80);
		col = linesTable.getColumnModel().getColumn(2); //model
		col.setPreferredWidth(80);
		col = linesTable.getColumnModel().getColumn(3); //cost
		col.setPreferredWidth(80);
		col = linesTable.getColumnModel().getColumn(4); //unit cost
		col.setPreferredWidth(80);
		col = linesTable.getColumnModel().getColumn(5); //total
		col.setPreferredWidth(80);
		col = linesTable.getColumnModel().getColumn(6); //desc
		col.setPreferredWidth(260);
	}
}
