package sellwin.gui;

import sellwin.domain.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class implements the Sales Group Admin
 * dialog.  This screen pops up when a user
 * selects the 'Group Admin' menu option from
 * the MainWindow.
 */
public class SalesGroupDialog extends JDialog implements GUIChars {
	private final static int TABLE_WIDTH = 300;
	private final static int TABLE_HEIGHT = 120;

	private TreeMap members=null;
	private TreeMap currentUsers=null;
	private JDialog thisDialog = null;

	private JPanel groupPanel = new JPanel();
	private JScrollPane groupListScrollPane = null;
	private JList groupList = new JList();

	private JPanel userGroupPanel = new JPanel();
	private JScrollPane userGroupListScrollPane = null;
	private JList userGroupList = new JList();

	private JPanel userSystemPanel = new JPanel();
	private JScrollPane userSystemListScrollPane = null;
	private JList userSystemList = new JList();

	private JMenuItem addGroupMenuItem = new JMenuItem("Add");
	private JMenuItem addUserMenuItem = new JMenuItem("Add");
	private JMenuItem deleteUserMenuItem = new JMenuItem("Delete");
	private JMenuItem deleteGroupMenuItem = new JMenuItem("Delete");
	private JButton closeButton = new JButton("Close");
	private JPopupMenu groupListPopup = new JPopupMenu();
	private JPopupMenu userListPopup = new JPopupMenu();
	
	private Whiteboard wb;

	private TreeMap allGroups = null;

	private DefaultListModel groupListModel = null;
	private DefaultListModel usersInSystemListModel = null;
	private DefaultListModel usersInGroupListModel = null;
	

	/**
	 * construct a Sales Group Dialog
	 */
	public SalesGroupDialog() {
		super();
		thisDialog = this;
		wb = MainWindow.getWhiteboard();

		setFonts();
		setColors();

		groupListPopup.add(addGroupMenuItem);
		groupListPopup.add(deleteGroupMenuItem);

		userListPopup.add(addUserMenuItem);
		userListPopup.add(deleteUserMenuItem);


		setTitle(wb.getLang().getString("salesGroupDialog"));
		setSize(350, 575);
		getContentPane().setLayout(new BorderLayout());

 		GridBagLayout gbl = new GridBagLayout();
        GridBagConstraints gbc = new GridBagConstraints();
		JPanel mainPanel = new JPanel();
		mainPanel.setBorder(new EtchedBorder());
        mainPanel.setLayout(gbl);
 
        gbc.gridwidth = GridBagConstraints.REMAINDER;
		groupList.setVisibleRowCount(3);
		groupList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		groupList.setPrototypeCellValue("XXXXXXXXXXXXXXXX");
		groupList.addMouseListener(
			new MouseAdapter() {
				public void mouseClicked(MouseEvent e) {
					if (e.getClickCount() == 1) {
						String key = (String)groupList.getSelectedValue();
						UserGroup g = null;
						try {
							allGroups = wb.getAllGroups();
							g = (UserGroup)(allGroups.get(key));
							wb.setCurrentGroup(g);
							setUsersInGroup(g);
						} catch (AngError f) {
							ErrorHandler.show(thisDialog, f);
						}
					}
				}
			}
		);
		groupListScrollPane = new JScrollPane(groupList);
		groupListScrollPane.setMinimumSize(new Dimension(TABLE_WIDTH,TABLE_HEIGHT));
		groupListScrollPane.setPreferredSize(new Dimension(TABLE_WIDTH,TABLE_HEIGHT));
		groupListScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
		groupPanel.add(groupListScrollPane);
		mainPanel.add(groupPanel, gbc);

        gbc.gridwidth = GridBagConstraints.REMAINDER;
		userGroupList.setVisibleRowCount(3);
		userGroupList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		userGroupList.setPrototypeCellValue("XXXXXXXXXXXXXXXX");
		userGroupListScrollPane = new JScrollPane(userGroupList);
		userGroupListScrollPane.setMinimumSize(new Dimension(TABLE_WIDTH,TABLE_HEIGHT));
		userGroupListScrollPane.setPreferredSize(new Dimension(TABLE_WIDTH,TABLE_HEIGHT));
		userGroupListScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
		userGroupPanel.add(userGroupListScrollPane);
		mainPanel.add(userGroupPanel, gbc);

        gbc.gridwidth = GridBagConstraints.REMAINDER;
		userSystemList.setVisibleRowCount(3);
		userSystemList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		userSystemList.setPrototypeCellValue("XXXXXXXXXXXXXXXX");
		userSystemListScrollPane = new JScrollPane(userSystemList);
		userSystemListScrollPane.setMinimumSize(new Dimension(TABLE_WIDTH,TABLE_HEIGHT));
		userSystemListScrollPane.setPreferredSize(new Dimension(TABLE_WIDTH,TABLE_HEIGHT));
		userSystemListScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
		userSystemPanel.add(userSystemListScrollPane);
		mainPanel.add(userSystemPanel, gbc);

		getContentPane().add(mainPanel, BorderLayout.CENTER);

 		WindowListener l = new WindowAdapter() {
			public void windowClosed(WindowEvent e) {
			}

            public void windowClosing(WindowEvent e) {
                hide();
            }
        };
 
        addWindowListener(l);

		JPanel buttonPanel = new JPanel();
		buttonPanel.add(closeButton);
		getContentPane().add(buttonPanel, BorderLayout.SOUTH);


		addUserMenuItem.addActionListener(
           new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					String selectedUserName = (String)userSystemList.getSelectedValue();
					String selectedGroup = (String)groupList.getSelectedValue();
					int option = JOptionPane.showConfirmDialog(thisDialog, "Add " + selectedUserName + " to " + selectedGroup + " ?");
					if (option != JOptionPane.YES_OPTION)
						return;

					UserGroup g = null;
					try {
						allGroups = wb.getAllGroups();
						String key = (String)groupList.getSelectedValue();
						g = (UserGroup)(allGroups.get(key));
						SalesPerson selectedUser = (SalesPerson)(currentUsers.get(userSystemList.getSelectedValue()));
						int i = UserGroup.findUser(g, selectedUser.getPK());
						if (i == -1) {
							wb.addUserToGroup(selectedUser, g);
							g.addUser(new Long(selectedUser.getPK()));
							setUsersInGroup(g);
						} else {
							JOptionPane.showMessageDialog(
								thisDialog,
								"The selected user was already in the group.",
								"Error",
								JOptionPane.ERROR_MESSAGE);
						}

					} catch (AngError f) {
						ErrorHandler.show(thisDialog, f);
					}
                }
            }
		);

		deleteUserMenuItem.addActionListener(
           new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					deleteUserFromGroup((String)userGroupList.getSelectedValue());
				}
			}
		);

		closeButton.addActionListener(
           new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    thisDialog.hide();
                }
            }
		);

		addGroupMenuItem.addActionListener(
           new ActionListener() {
                public void actionPerformed(ActionEvent e) {
					String newGroup = JOptionPane.showInputDialog("Please enter a Group name.");
                    if ((newGroup == null) || (newGroup.length() == 0)) return;

					UserGroup g = new UserGroup(newGroup);
					try {
						wb.addUserGroup(g);
						updateGroups();	
						JOptionPane.showMessageDialog(
                                thisDialog,
                                "Group Add Successful",
                                "Information",
                                JOptionPane.INFORMATION_MESSAGE);

					} catch (AngError f) {
						ErrorHandler.show(thisDialog, f);
					}
                }
            }
		);

		deleteGroupMenuItem.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
			
					String selected = (String)(groupList.getSelectedValue());
					int option = JOptionPane.showConfirmDialog(thisDialog, "Continue with Delete?");
					if (option == JOptionPane.YES_OPTION) {
						try {
							wb.deleteUserGroup(selected);
							DefaultListModel model = (DefaultListModel)(groupList.getModel());
							model.removeElement(selected);
						} catch (AngError f) {
							ErrorHandler.show(thisDialog, f);
						}
					} 
				}
			}
		);

		groupList.addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    groupListPopup.show(groupList, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    groupListPopup.show(groupList, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    groupListPopup.show(groupList, e.getX(), e.getY());
            }

		});

		userGroupList.addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger())
                    userListPopup.show(userGroupList, e.getX(), e.getY());
            }
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger())
                    userListPopup.show(userGroupList, e.getX(), e.getY());
            }
            public void mouseClicked(MouseEvent e) {
                if (e.isPopupTrigger())
                    userListPopup.show(userGroupList, e.getX(), e.getY());
            }

		});

		setLang();
		loadData();
	}

	/**
	 *  load some data into this dialog
	 */
	public final void loadData() {
		setSecurity();
		try {
			setUsersInSystem(wb.getSalesPersons());
			updateGroups();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 *  set the list of group members in the dialog list
	 * @param g the UserGroup to load the list with
	 */
	public final void setUsersInGroup(UserGroup g) {
		try {
			members = wb.getGroupMembers(g.getName());
			Object[] keys = members.keySet().toArray();
			userGroupList.setListData(keys);
		} catch (AngError e) {
			ErrorHandler.show(thisDialog, e);
		}
	}

	/**
	 * set the list of users in the system user's list 
	 * @param users a list of SalesPersons to load with
	 */
	public final void setUsersInSystem(TreeMap users) {

		DefaultListModel model = new DefaultListModel();
		String userName = null;
		SalesPerson user=null;
		currentUsers = users;

		Object[] keys = users.keySet().toArray();

		for (int i=0;i<keys.length;i++) {
			model.addElement((String)keys[i]);
		}
		userSystemList.setModel(model);
		if (keys.length > 0)
			userSystemList.setSelectedIndex(0);
	}
	
	/**
	 * update the list of user groups 
	 */
	public final void updateGroups() {

		setGroupList();

		String firstKey = (String)allGroups.firstKey();
		if (firstKey != null) {
			groupList.setSelectedIndex(0);
			UserGroup ug = (UserGroup)allGroups.get(firstKey);
			setUsersInGroup(ug);
		}
	}

	private void setGroupList() {
		try {
			allGroups = wb.getAllGroups();
			groupList.setListData(allGroups.keySet().toArray());
		} catch (AngError e) {
			ErrorHandler.show(thisDialog, e);
		}
	}
	
	/**
	 * delete a user from the group members list
	 * @param selectedUserIndex the index into the user list to 
	 * delete 
	 */
	public final void deleteUserFromGroup(String selectedUserId) {
		//remove it from the current selected UserGroup object
		SalesPerson selectedUser = (SalesPerson)(currentUsers.get(selectedUserId));
		ArrayList usersInGroup = wb.getCurrentGroup().getUsers();		
		Long nextPK = null;
		for (int i=0;i<usersInGroup.size();i++) {
			nextPK = (Long)usersInGroup.get(i);
			if (selectedUser.getPK() == nextPK.longValue()) {
				usersInGroup.remove(i);
				break;
			}
		}

		try {
			setUsersInGroup(wb.getCurrentGroup());
			wb.deleteUserInGroup(wb.getCurrentGroup(), selectedUser);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * set the dialog's fonts 
	 */
	public final void setFonts() {
	
		groupList.setFont(MainWindow.FIELD_FONT);
		userGroupList.setFont(MainWindow.FIELD_FONT);
		userSystemList.setFont(MainWindow.FIELD_FONT);

		addGroupMenuItem.setFont(MainWindow.LABEL_FONT);
		addUserMenuItem.setFont(MainWindow.LABEL_FONT);
		deleteGroupMenuItem.setFont(MainWindow.LABEL_FONT);
		deleteUserMenuItem.setFont(MainWindow.LABEL_FONT);
	}

	/**
	 * set the dialog's colors 
	 */
	public void setColors() {
	}

	/**
	 * set the dialog's fields visibility based
	 * upon the user's security creditials
	 */
	private final void setSecurity() {
		try {
			if (wb.getLogin().userAdminWritable()) {
				addGroupMenuItem.setEnabled(true);
				addUserMenuItem.setEnabled(true);
				deleteUserMenuItem.setEnabled(true);
				deleteGroupMenuItem.setEnabled(true);
			} else {
				addGroupMenuItem.setEnabled(false);
				addUserMenuItem.setEnabled(false);
				deleteUserMenuItem.setEnabled(false);
				deleteGroupMenuItem.setEnabled(false);
			}
		} catch (AngError e) {
			ErrorHandler.show(thisDialog, e);
		}
	}

	/** 
	 * set the dialog's language
	 */
	public final void setLang() {
		setTitle(wb.getLang().getString("salesGroupDialog"));
		addGroupMenuItem.setText(wb.getLang().getString("add"));
		addUserMenuItem.setText(wb.getLang().getString("add"));
		deleteUserMenuItem.setText(wb.getLang().getString("delete"));
		deleteGroupMenuItem.setText(wb.getLang().getString("delete"));
		closeButton.setText(wb.getLang().getString("close"));
		userGroupPanel.setBorder(
			new TitledBorder(
				null,
				wb.getLang().getString("groupMembers"),
				TitledBorder.DEFAULT_JUSTIFICATION,
				TitledBorder.DEFAULT_POSITION,
				MainWindow.LABEL_FONT,
				MainWindow.LETTERS
			)
		);
		groupPanel.setBorder(
			new TitledBorder(
				null,
				wb.getLang().getString("groups"),
				TitledBorder.DEFAULT_JUSTIFICATION,
				TitledBorder.DEFAULT_POSITION,
				MainWindow.LABEL_FONT,
				MainWindow.LETTERS
			)
		);
		userSystemPanel.setBorder(
			new TitledBorder(
				null,
				wb.getLang().getString("usersInSystem"),
				TitledBorder.DEFAULT_JUSTIFICATION,
				TitledBorder.DEFAULT_POSITION,
				MainWindow.LABEL_FONT,
				MainWindow.LETTERS
			)
		);
	}
}
