package sellwin.gui;

import sellwin.domain.*;
import sellwin.server.*;
import sellwin.utils.*;

import java.util.*;
import java.util.ArrayList;
import java.rmi.*;

import javax.swing.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class represents a global 'whiteboard' or 'clipboard'
 * for the GUI to read and write values to.  
 * There is only one whiteboard for the GUI.
 */
public class Whiteboard {
	private MainWindow parent;
	private Login login=null;
	private String speed = null;
	private String dataDir = null;
	private ResourceBundle langBundle;
	private SellwinSession remIF = null;
	private UserGroup currentGroup = null;
	private TreeMap custNamesList = null;
	private ArrayList opportunityNames = null;
	private ArrayList roleList = null;
	private TreeMap salesPersons = null;
	private ArrayList taxList = null;
	private TreeMap allGroups = null;
	private ArrayList allGroupsForUser = null;
	private ArrayList opportunities = new ArrayList();
	private boolean loggedOn = false;
	private SalesPerson currentUser = null;
	private Opportunity currentOpportunity = null;
	private Customer cust = new Customer();
	private ArrayList oppIndex;
	private Properties props;

	/**
	 * construct the Whiteboard and set the default language 
	 * and user properties
	 */
	public Whiteboard(MainWindow p) {
		try {
			parent = p;
			setProps(Prefs.getApplProps());
			String lang = (String)(props.getProperty(Prefs.DEF_LANG));
			setLang(lang);
		} catch (Exception e) {
			ErrorHandler.show(MainWindow.getMainParent(), e);
		}
	}


	/**
	 * set the user's default Properties
	 * @param p the new Properties we are setting with
	 */
	public final void setProps(Properties p) {
		props = p;
	}

	/**
	 * get the user's default Properties
	 * @return the current user Properties
	 */
	public final Properties getProps() {
		return props;
	}

	/**
	 * set the connection speed (not currently implemented) 
	 * @param name description
	 * @return description
	 * @exception AngError thrown for all application errors
	 */
	public final void setSpeed(String s) {
		this.speed = s;
	}

	/**
	 *  not currently implemented
	 * @param name description
	 * @return description
	 * @exception AngError thrown for all application errors
	 */
	public final String getSpeed() { return speed; }

	/**
	 * set the remote interface 
	 * by remote interface, we are referring to the 
	 * means of talking to the server, this can be
	 * anything that implements the SellwinSession
	 *
	 * @param r the remote interface we will be using
	 */
	public final void setRemIF(SellwinSession r) {
		remIF = r;
	}

	/**
	 * get the remote interface 
	 * @return the remote interface
	 */
	public final SellwinSession getRemIF() {
		return remIF;
	}

	/**
	 * get the current Opportunity which should
	 * also be the Opportunity on the screen at
	 * the moment
	 * @return the current Opportunity
	 */
	public final Opportunity getCurrentOpportunity() {
		return currentOpportunity;
	}

	/**
	 * set the current Opportunity 
	 * @param o the Opportunity we are setting with
	 */
	public final void setCurrentOpportunity(Opportunity o) {
		currentOpportunity = o;
	}

	/**
	 * add a customer name to the current cache of
	 * customer names
	 * @param name the Customer name we are adding
	 * @exception AngError thrown when an application error occurs
	 */
	public final void addCustName(String name) 
		throws AngError {

		if (custNamesList == null)
			custNamesList = getAllCustNames(true);
		custNamesList.put(name, name);

		//update the opp panel's customer combo box to reflect changes
		OppPanel oppPanel = MainWindow.getOppPanel();
		if (oppPanel != null) 
			oppPanel.updateCustomerList();
	}


	/**
	 * get the current list of customer names
	 * @param refresh if true, then force database query 
	 * @return the list of customer names
	 * @exception AngError thrown for all application errors
	 */
	public final TreeMap getAllCustNames(boolean refresh) 
		throws AngError {

		if ((custNamesList == null)  || (refresh)) {
			try {
				custNamesList = new TreeMap();
				Object[] values = getRemIF().getAllCustomerNames();
				for (int i=0;i<values.length;i++) 
					custNamesList.put((String)values[i], (String)values[i]);	
			} catch (RemoteException e) {
				throw new AngError(e.getMessage());
			}
		}

		return custNamesList;
	}

	/**
	 * get a list of all current user roles
	 * @return the list of user roles
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getAllUserRoles() 
		throws AngError {

		if (roleList == null) {
			try {
				roleList = getRemIF().getAllUserRoles(null);
			} catch (RemoteException e) {
				throw new AngError(e.getMessage());
			}
		}

		return roleList;
	}

	/**
	 * get the UserGroupMembers for a given group 
	 * @param groupName the group we are using as a search
	 * criteria
	 * @return the TreeMap of members in that group
	 * @exception AngError thrown for all application errors
	 */
	public final TreeMap getGroupMembers(String groupName) 
		throws AngError {

		TreeMap members=null;
		try {
			Object[] values = getRemIF().getUsersInGroup(groupName);
			members = new TreeMap();
			SalesPerson sp;
			for (int i=0;i<values.length;i++) {
				sp = (SalesPerson)values[i];
				members.put(sp.getID(), sp);
			}
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
		return members;
	}

	/**
	 * delete a user within a user group
	 * @param group the UserGroup we are to delete from
	 * @param user the SalesPerson we are to delete from the group 
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteUserInGroup(UserGroup group, SalesPerson user)
		throws AngError {

		try {
			getRemIF().deleteUserInGroup(group, user.getPK());
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * get the current UserGroup 
	 * @return the UserGroup
	 * @exception AngError thrown for all application errors
	 */
	public final UserGroup getCurrentGroup() {
		return currentGroup;
	}

	/**
	 * set the current UserGroup
	 * @param g the UserGroup
	 * @exception AngError thrown for all application errors
	 */
	public final void setCurrentGroup(UserGroup g) {
		currentGroup = g;
	}

	/**
	 * get the current list of users in a group 
	 * @param groupName the user group name we are using
	 * as a search criteria
	 * @return the list of users in the group
	 * @exception AngError thrown for all application errors
	 */
	public final TreeMap getUsersInGroup(String groupName) 
		throws AngError {

		try {
			Object[] values = getRemIF().getUsersInGroup(groupName);
			TreeMap userNames = new TreeMap();
			SalesPerson sp;
			for (int i=0;i<values.length;i++) {
				sp = (SalesPerson)values[i];
				userNames.put(sp.getID(), sp);
			}
			return userNames;
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * get a list of the current set of user groups
	 * @return the TreeMap of user groups
	 * @exception AngError thrown for all application errors
	 */
	public final TreeMap getAllGroups() 
		throws AngError {

		try {
			Object[] values = getRemIF().getUserGroups(null);
			allGroups = new TreeMap();
			UserGroup ug;
			for (int i=0;i<values.length;i++) {
				ug = (UserGroup)values[i];
				allGroups.put(ug.getName(), ug);
			}
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}
		return allGroups;
	}

	/**
	 * get the current list of groups that a user belongs
	 * to
	 * @return an ArrayList of UserGroup Strings
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getGroupsForUser() 
		throws AngError {

		if (allGroupsForUser != null)
			return allGroupsForUser;

		try {
			allGroupsForUser = getRemIF().getGroupsForUser(currentUser.getPK());
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}
		return allGroupsForUser;
	}

	/**
	 * get the current list of user groups
	 * @return the list of user groups
	 * @exception AngError thrown for all application errors
	 */
	public final TreeMap getUserGroups()
		throws AngError {

		try {
			Object[] values = getRemIF().getUserGroups(null);
			TreeMap map = new TreeMap();
			UserGroup ug;
			for (int i=0;i<values.length;i++) {
				ug = (UserGroup)values[i];
				map.put(ug.getName(), ug);
			}
			return map;
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * add a user to a user group 
	 * @param user the user we are adding
	 * @param group the group we are adding to
	 * @exception AngError thrown for all application errors
	 */
	public final void addUserToGroup(SalesPerson user, UserGroup group)
		throws AngError {

		try {
			//add to local cache
			if (user.getPK() == currentUser.getPK()) {
				if (allGroupsForUser == null)
					allGroupsForUser = new ArrayList();
				allGroupsForUser.add(group);
			}

			group.setModifiedBy(currentUser.getID());
			getRemIF().addUserToGroup(user.getPK(), group);
			user.addUserGroup(group);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * add a new user group 
	 * @param group the user group to add
	 * @exception AngError thrown for all application errors
	 */
	public final void addUserGroup(UserGroup group) 
		throws AngError {
		
		try {
			group.setModifiedBy(currentUser.getID());
			getRemIF().addUserGroup(group); //add to persistent store
			allGroups.put(group.getName(), group); //add to cache
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * delete a user group 
	 * @param groupName the name of the user group we delete
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteUserGroup(String groupName)
		throws AngError {

		try {
			getRemIF().deleteUserGroup(groupName);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}


	/**
	 * set the current list of Opportunities 
	 * @param o the list of Opportunities we set with
	 * @exception AngError thrown for all application errors
	 */
	public final void setOpportunities(ArrayList o) {
		opportunities = o;
	}

	/**
	 * get the Opportunity index.  This index is a quick
	 * reference for the application to do Opportunity
	 * lookups using the Opportunity name, especially important
	 * when operating in 'disconnected' mode
	 * @return the Opportunity Index list
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getOpportunityIndex() 
		throws AngError {
		
		ArrayList oppIndex = null;
		try {
			oppIndex = getRemIF().getOpportunityIndex(getCurrentUser());
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
		return oppIndex;
	}

	/**
	 * lookup an Opportunity by it's name 
	 * @param name the Opportunity name we look up with
	 * @return the Opportunity if found
	 * @exception AngError thrown for all application errors
	 */
	public final Opportunity getOpportunityByName(String name) 
		throws AngError {

		Opportunity opp=null;

		//check the cache first
		for (int i=0;i<opportunities.size();i++) {
			opp = (Opportunity)opportunities.get(i);
			if (opp.getName().equals(name))
				return opp;
		}

		//at this point the opp was not in the cache so look else where
		if (oppIndex == null) {
			try {
				oppIndex = getRemIF().getOpportunityIndex(getCurrentUser());
			} catch (RemoteException r) {
				throw new AngError(r.getMessage());
			}
		}

		long pk=-1;
		OppIndex oi = null;
		for (int k=0;k<oppIndex.size();k++) {
			oi = (OppIndex)oppIndex.get(k);
			if (oi.getName().equals(name))
				pk = oi.getPK();
		}

		if (pk == -1)
			throw new AngError("getOppByName lookup failed [name]="+name);
		try {
			opp =  getRemIF().getOpportunity(pk);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
		if (opp == null)
			throw new AngError("getOppByName opp not found [name]="+name);
		return opp;
	}

	/**
	 * get the current list of Alarms
	 * @return the list of Alarms
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getAlarms() 
		throws AngError {
		
		try {
			//this 'if' handles the case when a person doesn't log in for
			//a period longer than the activity checker time (5 mins)
		
			ArrayList acts;	
			if ((getRemIF() != null) && (getCurrentUser() != null)) {
				acts = getRemIF().getAlarms(getCurrentUser().getPK());
				System.out.println("Whiteboard acts="+ acts.size());
			} else 
				acts = new ArrayList();
			return acts;
		} catch (RemoteException r) {
			r.printStackTrace();
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * add a new Activity 
	 * @param oppPK the current Opportunity primary key of
	 * the Opportunity that will get the Activity added to
	 * @param a the Activity we are adding
	 * @return the primary key of the newly added Activity
	 * @exception AngError thrown for all application errors
	 */
	public final long addActivity(long oppPK, Activity a) 
		throws AngError {
	
		try {
			a.setModifiedBy(currentUser.getID());
			return getRemIF().addActivity(oppPK, a);
		} catch (RemoteException r) {
			r.printStackTrace();
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * delete an Activity from a given Opportunity
	 * @param oppPK the Opportunity primary key we are
	 * to delete from
	 * @param actPK the Activity primary key we are to
	 * delete 
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteActivity(long oppPK, long actPK)
		throws AngError {

		try {
			getRemIF().deleteActivity(oppPK, actPK);
		} catch (RemoteException r) {
			r.printStackTrace();
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * update an Activity
	 * @param oppPK the Opportunity primary key the Activity
	 * belongs to
	 * @param a the Activity we are updating
	 * @exception AngError thrown for all application errors
	 */
	public final void updateActivity(long oppPK, Activity a)
		throws AngError {
		
		try {
			a.setModifiedBy(currentUser.getID());
			getRemIF().updateActivity(oppPK, a);
		} catch (RemoteException r) {
			r.printStackTrace();
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * udpate an Opportunity
	 * @param o the Opportunity we are updating
	 * @exception AngError thrown for all application errors
	 */
	public final void updateOpportunity(Opportunity o)
		throws AngError {

		try {
			o.setModifiedBy(currentUser.getID());
			getRemIF().updateOpportunity(o);
		} catch (RemoteException r) {
			r.printStackTrace();
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * get a list of Opportunity names 
	 * @return the list of names
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getOpportunityNames() 
		throws AngError {

		try {
			opportunityNames = getRemIF().getOpportunityNames(getCurrentUser());
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
		return opportunityNames;
	}
		
			
	/**
	 * get all the Opportunities for this current user
	 * @return the list of Opportunities
	 */
	public final ArrayList getOpportunities() {
		return opportunities;
	}

	/**
	 * check to see if the current user is logged on 
	 * @return true if logged on or false otherwise
	 */
	public final boolean isLoggedOn() {
		return loggedOn;
	}

	/**
	 * set the logon status
	 * @param t true if the user is logged on, false otherwize
	 */
	public final void setLoggedOn(boolean t) {
		loggedOn = t;
	}

	/**
	 * set the current Customer 
	 * @param c the Customer we are to set as current
	 */
	public final void setCustomer(Customer c) { 
		cust = c;
	}

	/**
	 * get the current Customer 
	 * @return the Customer
	 */
	public final Customer getCustomer() {
		return cust;
	}

	/**
	 * get the Customer that has a given name 
	 * @param custName the Customer name we are searching with
	 * @return the found Customer
	 * @exception AngError thrown for all application errors
	 */
	public final Customer getCustomer(String custName)
		throws AngError {

		try {
			return getRemIF().getCustomer(custName);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * add a Customer to the system's list of Customers
	 * @param c the Customer to add
	 * @return the primary key of the new Customer
	 * @exception AngError thrown for all application errors
	 */
	public final long addCustomer(Customer c) 
		throws AngError {

		try {
			c.setModifiedBy(currentUser.getID());
			return getRemIF().addCustomer(c);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * delete a SalesPerson 
	 * @param pk the primary key of the SalesPerson to delete
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteSalesPerson(SalesPerson person)
		throws AngError {
		
		try {
			getRemIF().deleteSalesPerson(person); //delete the perm copy
			Object sp = salesPersons.remove(person.getID()); //delete mem copy
			if (sp == null)
				throw new AngError("sp pk not found");
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}
			
	/**
	 * add a SalesPerson 
	 * @param s the SalesPerson to add
	 * @exception AngError thrown for all application errors
	 */
	public final void addSalesPerson(SalesPerson s) 
		throws AngError {
		
		try {
			s.setModifiedBy(currentUser.getID());
			System.out.println(s.getModifiedBy() + " is the modified value in whiteboard");
			s.setPK(getRemIF().addSalesPerson(s)); //add to db
			salesPersons.put(s.getID(), s); //add to memory copy
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * get a SalesPerson using a formatted name
	 * @param formatted the SalesPerson's formatted name to
	 * search with
	 * @return the found SalesPerson
	 * @exception AngError thrown for all application errors
	 */
//	public final SalesPerson getSalesPerson(String formatted)
//		throws AngError {
//
//		try {
//			return getRemIF().getSalesPerson(formatted);
//		} catch (RemoteException r) {
//			throw new AngError(r.getMessage());
//		}
//	}

	/**
	 * add an Opportunity 
	 * @param o the Opportunity we are adding
	 * @return the primary key of the new Opportunity
	 * @exception AngError thrown for all application errors
	 */
	public final long addOpportunity(Opportunity o) 
		throws AngError {
	
		try {
			o.setModifiedBy(currentUser.getID());
			long pk = getRemIF().addOpportunity(o);
			o.setPK(pk);
			opportunities.add(o);
			return pk;	
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * delete an Opportunity 
	 * @param pk the primary key we find the Opportunity with
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteOpportunity(long pk) 
		throws AngError {
		try {
			getRemIF().deleteOpportunity(pk);
			Opportunity opp = null;
			for (int i = 0;i<opportunities.size();i++) {	
				opp = (Opportunity)opportunities.get(i);
				if (opp.getPK() == pk)
					opportunities.remove(i);
			}
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * get a list of SalesPersons
	 * @return the list of SalesPersons
	 * @exception AngError thrown for all application errors
	 */
	public final TreeMap getSalesPersons() 
		throws AngError {

		try {
			
			Object[] values =  getRemIF().getSalesPersons(null);
			salesPersons =  new TreeMap();
			SalesPerson sp;
			for (int i=0;i<values.length;i++) {
				sp = (SalesPerson)values[i];
				salesPersons.put(sp.getID(), sp);
			}

			return salesPersons;
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * get a list of SalesPerson names
	 * @return the list of names
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getSalesPersonNames() 
		throws AngError {

		try {
			return getRemIF().getSalesPersonNames();
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * update a SalesPerson
	 * @param u the SalesPerson to update
	 * @exception AngError thrown for all application errors
	 */
	public final void updateSalesPerson(SalesPerson u) 
		throws AngError {

		try {
			u.setModifiedBy(currentUser.getID());
			getRemIF().updateSalesPerson(u); //update the db copy

			if (salesPersons == null)
				salesPersons = getSalesPersons();

			Object sp = salesPersons.remove(u.getID());
			if (sp == null)
				throw new AngError("sp not found in memory");
			salesPersons.put(u.getID(), u);
		} catch (RemoteException r) {
			r.printStackTrace();
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * set the current Login information, this is done
	 * after a person logs into the system successfully
	 * @param l the Login creditials
	 */
 	public final void setCurrentLogin(Login l) {
		System.out.println("setting current login to " + l.getSalesPerson().getID());
		setCurrentUser(l.getSalesPerson());
		login = l;
	}

	/**
	 * get the current Login creditials
	 * @return the current Login
	 */
	public final Login getLogin()
		throws AngError { 

		if (login == null)
			throw new AngError("User not logged in.  Please log in.");
		return login; 
	}
	
	/**
	 * set the current user or SalesPerson 
	 * @param s the SalesPerson we set with (yuck)
	 */
	public final void setCurrentUser(SalesPerson s) { 
		currentUser = s;
	}

	/**
	 * get the current SalesPerson or logged on user 
	 * @return the current SalesPerson
	 */
	public final SalesPerson getCurrentUser() {
		return currentUser;
	}

	/**
	 * delete a Forecast from an Opportunity 
	 * @param oppPK the primary key of the Opportunity that this
	 * Forecast belongs to
	 * @param forePK the primary key of the Forecast we are deleting 
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteForecast(long oppPK, long forePK)
		throws AngError {

		try {
			getRemIF().deleteForecast(oppPK, forePK);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * delete a Quote from a given Opportunity 
	 * @param oppPK the primary key of the Opportunity we delete this
	 * Quote from
	 * @param quotePK the Quote primary key we delete with
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteQuote(long oppPK, long quotePK)
		throws AngError {
		try {
			getRemIF().deleteQuote(oppPK, quotePK);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * update a Quote 
	 * @param q the Quote we update
	 * @exception AngError thrown for all application errors
	 */
	public final void updateQuote(Quote q)
		throws AngError {
		try {
			Opportunity opp = getCurrentOpportunity();
			q.setModifiedBy(currentUser.getID());
			getRemIF().updateQuote(opp.getPK(), q);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}
	/**
	 * add a Quote 
	 * @param oppPK the primary key of the Opportunity we are adding to
	 * @param q the Quote we are adding
	 * @return the primary key of the newly added Quote
	 * @exception AngError thrown for all application errors
	 */
	public final long addQuote(long oppPK, Quote q)
		throws AngError {
		try {
			q.setModifiedBy(currentUser.getID());
			return getRemIF().addQuote(oppPK, q);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

		
	/**
	 * delete a QuoteLine 
	 * @param oppPK the primary key of the containing Opportunity
	 * @param quotePK the primary key of the containing Quote
	 * @param linePK the primary key of the QuoteLine we want to delete
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteQuoteLine(long oppPK, long quotePK, long linePK) 
		throws AngError {
		
		try {
			getRemIF().deleteQuoteLine(oppPK, quotePK, linePK);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * add a new QuoteLine
	 * @param oppPK the primary key of the containing Opportunity
	 * @param quotePK the primary key of the containing Quote
	 * @param q the QuoteLine to add
	 * @return the primary key of the newly added QuoteLine
	 * @exception AngError thrown for all application errors
	 */
	public final long addQuoteLine(long oppPK, long quotePK, QuoteLine q) 
		throws AngError {
		
		try {
			q.setModifiedBy(getCurrentUser().getID());
			return getRemIF().addQuoteLine(oppPK, quotePK, q);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * delete a Contact
	 * @param pk the primary key of the Contact we wish to delete
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteContact(long pk) 
		throws AngError {

		Opportunity opp = getCurrentOpportunity();
		try {
			getRemIF().deleteContact(opp.getPK(), pk);
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add a Contact to the current Opportunity
	 * @param c the Contact we want to add
	 * @return the primay key of the new Contact
	 * @exception AngError thrown for all application errors
	 */
	public final long addContact(Contact c) 
		throws AngError {

		Opportunity opp = getCurrentOpportunity();
		try {
			c.setModifiedBy(currentUser.getID());
			return getRemIF().addContact(opp.getPK(), c);
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * update a Contact of the current Opportunity 
	 * @param c the Contact to update
	 * @exception AngError thrown for all application errors
	 */
	public final void updateContact(Contact c) 
		throws AngError {

		Opportunity opp = getCurrentOpportunity();
		try {
			c.setModifiedBy(currentUser.getID());
			getRemIF().updateContact(opp.getPK(), c);
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * add a Forecast to a given Opportunity 
	 * @param oppPK the primary key of the containing Opportunity
	 * @param f the Forecast to add
	 * @exception AngError thrown for all application errors
	 */
	public final void addForecast(long oppPK, Forecast f) 
		throws AngError {

		try {
			f.setModifiedBy(currentUser.getID());
			long pk = getRemIF().addForecast(oppPK, f);
			f.setPK(pk);
			getCurrentOpportunity().addForecast(f);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * update a Forecast of the current Opportunity 
	 * @param f the Forecast to update
	 * @exception AngError thrown for all application errors
	 */
	public final void updateForecast(Forecast f)
		throws AngError {

		try {
			long oppPK = getCurrentOpportunity().getPK();
			f.setModifiedBy(currentUser.getID());
			getRemIF().updateForecast(oppPK, f);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * update an Order of the current Opportunity 
	 * @param o the Order to update
	 * @exception AngError thrown for all application errors
	 */
	public final void updateOrder(Order o)
		throws AngError {
		try {
			long oppPK = getCurrentOpportunity().getPK();
			o.setModifiedBy(currentUser.getID());
			getRemIF().updateOrder(oppPK, o);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * add an Order to the current Opportunity 
	 * @param o the Order to add
	 * @return the primary key of the new Order
	 * @exception AngError thrown for all application errors
	 */
	public final long addOrder(Order o)
		throws AngError {
		try {
			o.setModifiedBy(getCurrentUser().getID());
			long oppPK = getCurrentOpportunity().getPK();
			return getRemIF().addOrder(oppPK, o);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * update a Customer 
	 * @param c the Customer to update
	 * @exception AngError thrown for all application errors
	 */
	public final void updateCustomer(Customer c) 
		throws AngError {

		try {
			c.setModifiedBy(currentUser.getID());
			getRemIF().updateCustomer(c);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * delete a Customer  using a name as search criteria
	 * @param custName  the customer's name we search with
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteCustomer(String custName)
		throws AngError {

		try {
			getRemIF().deleteCustomer(custName);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * udpate a Lead  
	 * @param campPK the Campaign this lead is contained in
	 * @param l the Lead we wish to update
	 * @exception AngError thrown for all application errors
	 */
	public final void updateLead(long campPK, Lead l) 
		throws AngError {
		try {
			l.setModifiedBy(currentUser.getID());
			getRemIF().updateLead(campPK, l);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * get the list of Campaigns 
	 * @return the list of Campaigns
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getCampaigns() 
		throws AngError {

		ArrayList camps = null;
		try { 
			camps = getRemIF().getCampaigns(null);
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}
		return camps;
	}

	/**
	 *  get a list of Leads for a given Campaign
	 * @param campPK the Campaign primary key to find with
	 * @return the list of Leads
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getCampaignLeads(long campPK) 
		throws AngError {

		ArrayList leads = null;

		try {	
			leads = getRemIF().getCampaignLeads(campPK);			
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}
		return leads;
	}

	/**
	 * get a Product using some search criteria
	 * @param group the product group to search
	 * @param line the product line to search
	 * @param name the product name to search
	 * @return the found Product
	 * @exception AngError thrown for all application errors
	 */
	public final Product getProduct(String group, String line, String name)
		throws AngError {

		Product prod = null;

		try {	
			prod = getRemIF().getProduct(group, line, name);			
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}
		return prod;
	}

	/**
	 * get the ArrayList of products for a given group and line
	 * @param group the product group to search
	 * @param line the product line to search
	 * @return an ArrayList of product matrix data
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getProductsForLine(String group, String line)
		throws AngError {

		try {
			return getRemIF().getProductsForLine(group, line);
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}

	}

	/**
	 * get the product matrix 
	 * @return an Arraylist of product matrix data
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getProductMatrix() 
		throws AngError {

		try {
			return getRemIF().getProductMatrix();
		} catch (RemoteException e) {
			throw new AngError(e.getMessage());
		}
	}

	/**
	 * gets the resource bundle that is currently set
	 * for the user's language preference
	 * @return the resource bundle for a specific language
	 */
	public final ResourceBundle getLang() {
		return langBundle;
	}

	/**
	 * sets the resource bundle that is currently set
	 * for the user's language preference
	 * @param r resource bundle for a specific language
	 */
	public final void setLang(String lang) {
        try {

			if (lang.equals("French")) {
				System.out.println("French selected");
				langBundle = ResourceBundle.getBundle("French", new Locale("fr", "FR"));
			}
			else
			if (lang.equals("German")) {
				System.out.println("German selected");
				langBundle = ResourceBundle.getBundle("German", new Locale("de", "DE"));
			} 
			else
			if (lang.equals("Spanish")) {
				System.out.println("Spanish selected");
				langBundle = ResourceBundle.getBundle("Spanish", new Locale("es", "MX"));
			}
			else  {
				//default to English
				System.out.println("English selected");
				langBundle = ResourceBundle.getBundle("English", new Locale("en", "US"));
			}
		} catch (MissingResourceException e) {
			ErrorHandler.show(MainWindow.getMainParent(), e);
		}
	}

	/**
	 * get the current list of state tax codes
	 * @param refresh if true, then force database query 
	 * @return the list of state tax codes
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getStateTax(boolean refresh) 
		throws AngError {

		if ((taxList == null)  || (refresh)) {
			try {
				taxList = getRemIF().getStateTax(null);
			} catch (RemoteException e) {
				throw new AngError(e.getMessage());
			}
		}

		return taxList;
	}

	/**
	 * add a CustomerInventory 
	 * @param c the Customer to add
	 * @return the primary key of the new Customer
	 * @exception AngError thrown for all application errors
	 */
	public final long addCustomerInventory(CustomerInventory c) 
		throws AngError {

		try {
			c.setModifiedBy(currentUser.getID());
			return getRemIF().addCustomerInventory(c);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * delete a CustomerInventory 
	 * @param custName  the customer's name we search with
	 * @exception AngError thrown for all application errors
	 */
	public final void deleteCustomerInventory(CustomerInventory ci)
		throws AngError {

		try {
			getRemIF().deleteCustomerInventory(ci);
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * get CustomerInventory  for current opp's customer
	 * @return the list of CustomerInventory
	 * @exception AngError thrown for all application errors
	 */
	public final ArrayList getCustomerInventory() 
		throws AngError {

		try {
			return getRemIF().getCustomerInventory(getCurrentOpportunity().getCustomer().getPK());
		} catch (RemoteException r) {
			throw new AngError(r.getMessage());
		}
	}

	/**
	 * get the main window or mother parent of the whole gui
	 * @return a reference to the MainWindow
	 */
	public final MainWindow getMainWindow() {
		return parent;
	}

}
