package sellwin.server;

import sellwin.domain.*;
import java.util.*;
import java.io.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This class runs on the user's client machine when
 * operating SellWin in 'disconnected' mode.  This class
 * specifically simulates a simple database that holds
 * all downloaded SellWin data.  
 */
public class LocalDB {
	private String BASE;
	public final static String EXTENSION = ".opp";
	public final static String F_INDEX = "index.objs";
	public final static String F_ROLES = "role.objs";
	public final static String F_GROPS = "group.objs";
	public final static String F_SPERS = "salesPerson.objs";
	public final static String F_PRODS = "product.objs";
	public final static String F_CAMPS = "campaign.objs";
	public final static String F_CUSTS = "customer.objs";
	public final static String F_TAX = "tax.objs";
	public final static String F_DELETE = "deleted.objs";

	/**
	 * construct a local database object 
	 * @param base the base pathname used to find local
	 * data store files
	 */
	public LocalDB(String base) {
		BASE = base;
	}

	/**
	 * set the base path to the local store files 
	 * @param b the base path name
	 */
	public final void setBASE(String b) {
		BASE = b;
	}

	/**
	 * get the base pathname to the local files 
	 * @return the base path name
	 */
	public final String getBASE() {
		return BASE;
	}

	/**
	 * get the opportunity index list 
	 * @return the ArrayList of OpportunityIndex structure
	 * @exception AngError thrown when an app error occurs
	 */
	public final ArrayList getOppIndex() 
		throws AngError {

		FileInputStream istream = null;
		ArrayList oppIndex = null;

		try {
			istream = new FileInputStream(BASE + F_INDEX);
			ObjectInputStream p = new ObjectInputStream(istream);
			oppIndex = (ArrayList)p.readObject();
		} catch (FileNotFoundException f) {
			//if not found, then create an emtpy file to start with
			System.out.println("creating " + BASE + F_INDEX);
			oppIndex = new ArrayList();
			saveOppIndex(oppIndex);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}

		return oppIndex;

	}

	/**
	 * save the opportunity index to local disk
	 * @param oppIndex the ArrayList we are to save
	 * @exception AngError thrown when an app error occurs
	 */
	public final void saveOppIndex(ArrayList oppIndex) 
		throws AngError {

		FileOutputStream ostream = null;

		try {
			System.out.println("saving " + BASE + F_INDEX);
			ostream = new FileOutputStream(BASE + F_INDEX);
			ObjectOutputStream p = new ObjectOutputStream(ostream);
			p.writeObject(oppIndex);
			p.flush();
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { ostream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * lookup a single Opportunity
	 * @param oppName the opportunity name as search key
	 * @return the found Opportunity or null if not found
	 * @exception AngError thrown when an app error occurs
	 */
	public final Opportunity getOpportunity(String oppName) 
		throws AngError {

		FileInputStream istream = null;

		try {
			istream = new FileInputStream(generateOppFileName(oppName));
			ObjectInputStream p = new ObjectInputStream(istream);
			Opportunity o = (Opportunity)p.readObject();
			return o;
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * find an Opportunity by primary key 
	 * @param pk the primary key of the Opportunity to search with
	 * @return the found Opportunity or null if not found
	 * @exception AngError thrown when an app error occurs
	 */
	public final Opportunity getOpportunity(long pk)
		throws AngError {

		FileInputStream istream = null;

		try {
			istream = new FileInputStream(generateOppFileName(pk));
			ObjectInputStream p = new ObjectInputStream(istream);
			Opportunity o = (Opportunity)p.readObject();
			return o;
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * lookup an Opportunity by primary key and by using
	 * the opportunity index
	 * @param index the opportunity index structure to search in
	 * @param pk the primary key to search with
	 * @return the index into the opportunity index that matches 
	 * @exception AngError thrown when an app error occurs
	 */
	public int findOpp(ArrayList index, long pk) {

		OppIndex oi = null;
		for (int i=0;i<index.size();i++) {
			oi = (OppIndex)index.get(i);
			if (oi.getPK() == pk)
				return i;
		}

		return -1; //not found
	}

	/**
	 * save an Opportunity to local disk
	 * @param o the Opportunity to save
	 * @exception AngError thrown when an app error occurs
	 */
	public final void saveOpportunity(Opportunity o) 
		throws AngError {

		FileOutputStream ostream = null;

		try {
			System.out.println("saving "+ generateOppFileName(o.getPK()));
			ostream = new FileOutputStream(generateOppFileName(o.getPK()));
			ObjectOutputStream p = new ObjectOutputStream(ostream);
			p.writeObject(o);
			p.flush();

			//save the new opp to the index file for future reads
			ArrayList index = getOppIndex();
			int i = findOpp(index, o.getPK());
			OppIndex oi = null;
			if (i >= 0) { //already exists so update index
				oi = (OppIndex)index.get(i);
				oi.setName(o.getName());
			} else { //doesn't exist so add to index
				oi = new OppIndex(o.getName(), o.getStage(), o.getPK());
				index.add(oi);
			}
			saveOppIndex(index);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { ostream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * get the Opportunity Index using an Opportunity name 
	 * @param oppName search with an Opportunity name
	 * @return the matching OppIndex 
	 * @exception AngError thrown when an app error occurs
	 */
	private OppIndex getOppIndex(String oppName) 
		throws AngError {
		OppIndex oi = null;
		ArrayList index = getOppIndex();
		for (int i=0;i<index.size();i++) {
			oi = (OppIndex)index.get(i);
			if (oi.getName().equals(oppName))
				return oi;
		}

		throw new AngError("not found not correct");
	}

	/**
	 * generate a file name based upon an opportunity name 
	 * @param oppName the Opportunity name to generate for
	 * @return a String containing the generated file name
	 * @exception AngError thrown when an app error occurs
	 */
	public final String generateOppFileName(String oppName) 
		throws AngError {

		ArrayList index = getOppIndex();
		OppIndex oi = getOppIndex(oppName);
		return generateOppFileName(oi.getPK());
	}

	/**
	 * generate an opportunity file name using a primary
	 * key value
	 * @param pk the primary key used to generate the file name
	 * @return a String containing the generated file name
	 * @exception AngError thrown when an app error occurs
	 */
	public final String generateOppFileName(long pk) {
		return new String(BASE + pk + EXTENSION);
	}

	/**
	 * get the list of UserRoles from local disk 
	 * @return the ArrayList of UserRoles
	 * @exception AngError thrown when an app error occurs
	 */
	public final ArrayList getRoleList() 
		throws AngError {

		FileInputStream istream=null;
		ArrayList roleList = null;

		try {
			istream = new FileInputStream(BASE + F_ROLES);
			ObjectInputStream p = new ObjectInputStream(istream);
			roleList = (ArrayList)p.readObject();
		} catch (FileNotFoundException f) {
			System.out.println("creating " + BASE + F_ROLES);
			roleList = new ArrayList();
			saveRoleList(roleList);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}
		return roleList;
	}

	/**
	 * save to local disk the list of UserRoles
	 * @param roles the ArrayList of UserRoles to svae
	 * @exception AngError thrown when an app error occurs
	 */
	public final void saveRoleList(ArrayList roles) 
		throws AngError {

		FileOutputStream ostream = null;

		try {
			System.out.println("saving " + BASE + F_ROLES);
			ostream = new FileOutputStream(BASE + F_ROLES);
			ObjectOutputStream p = new ObjectOutputStream(ostream);
			p.writeObject(roles);
			p.flush();
		} catch (Exception e) {
			throw new AngError(e.getMessage());
		} finally {
			try { ostream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * get the ArrayList of UserGroups from local disk
	 * @return the ArrayList that was retrieved
	 * @exception AngError thrown when an app error occurs
	 */
	public final ArrayList getGroupList() 
		throws AngError {

		FileInputStream istream = null;
		ArrayList groupList = null;

		try {
			istream = new FileInputStream(BASE + F_GROPS);
			ObjectInputStream p = new ObjectInputStream(istream);
			groupList = (ArrayList)p.readObject();
		} catch (FileNotFoundException f) {
			System.out.println("creating " + BASE + F_GROPS);
			groupList = new ArrayList();
			saveGroupList(groupList);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }	
		}
		return groupList;
	}

	/**
	 * save the user group list to local disk
	 * @param groups the ArrayList of UserGroups to save
	 * @exception AngError thrown when an app error occurs
	 */
	public final void saveGroupList(ArrayList groups) 
		throws AngError {

		FileOutputStream ostream = null;

		try {
			System.out.println("saving " + BASE + F_GROPS);
			ostream = new FileOutputStream(BASE + F_GROPS);
			ObjectOutputStream p = new ObjectOutputStream(ostream);
			p.writeObject(groups);
			p.flush();
		} catch (Exception e) {
			throw new AngError(e.getMessage());
		} finally {
			try { ostream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * get the ArrayList of Products from local disk
	 * @return the Product list
	 * @exception AngError thrown when an app error occurs
	 */
	public final ArrayList getProducts() 
		throws AngError {

		FileInputStream istream = null;
		ArrayList prodList = null;

		try {
			istream = new FileInputStream(BASE + F_PRODS);
			ObjectInputStream p = new ObjectInputStream(istream);
			prodList = (ArrayList)p.readObject();
		} catch (FileNotFoundException f) {
			System.out.println("creating " + BASE + F_PRODS);
			prodList = new ArrayList();
			saveProducts(prodList);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}
		return prodList;
	}

	/**
	 * save to local disk the list of Products
	 * @param prods the ArrayList of Products to save
	 * @exception AngError thrown when an app error occurs
	 */
	public final void saveProducts(ArrayList prods) 
		throws AngError {

		FileOutputStream ostream = null;

		try {
			ostream = new FileOutputStream(BASE + F_PRODS);
			ObjectOutputStream p = new ObjectOutputStream(ostream);
			p.writeObject(prods);
			p.flush();
		} catch (Exception e) {
			throw new AngError(e.getMessage());
		} finally {
			try { ostream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * get the ArrayList of SalesPersons on local disk
	 * @return the ArrayList of SalesPersons
	 * @exception AngError thrown when an app error occurs
	 */
	public final TreeMap getSalesPersons() 
		throws AngError {

		FileInputStream istream = null;
		TreeMap salesPersonList = null;

		try {
			istream = new FileInputStream(BASE + F_SPERS);
			ObjectInputStream p = new ObjectInputStream(istream);
			salesPersonList = (TreeMap)p.readObject();
		} catch (FileNotFoundException f) {
			System.out.println("creating " + BASE + F_SPERS);
			salesPersonList = new TreeMap();
			saveSalesPersons(salesPersonList);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}
		return salesPersonList;
	}

	/**
	 * save the SalesPersons to local disk
	 * @param persons the SalesPersons we are saving (funny eh?)
	 * @exception AngError thrown when an app error occurs
	 */
	public final void saveSalesPersons(TreeMap persons) 
		throws AngError {

		FileOutputStream ostream = null;

		try {
			ostream = new FileOutputStream(BASE + F_SPERS);
			ObjectOutputStream p = new ObjectOutputStream(ostream);
			p.writeObject(persons);
			p.flush();
		} catch (Exception e) {
			throw new AngError(e.getMessage());
		} finally {
			try { ostream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * get the local ArrayList of Campaigns from disk
	 * @return the ArrayList of Campaigns found on disk
	 * @exception AngError thrown when an app error occurs
	 */
	public final ArrayList getCampaigns() 
		throws AngError {

		FileInputStream istream = null;
		ArrayList campaigns = null;

		try {
			istream = new FileInputStream(BASE + F_CAMPS);
			ObjectInputStream p = new ObjectInputStream(istream);
			campaigns = (ArrayList)p.readObject();
		} catch (FileNotFoundException f) {
			campaigns = new ArrayList();
			saveCampaigns(campaigns);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}
		return campaigns;
	}

	/**
	 * save to local disk the list of Campaigns 
	 * @param campaigns the ArrayList we save
	 * @exception AngError thrown when an app error occurs
	 */
	public final void saveCampaigns(ArrayList campaigns) 
		throws AngError {

		FileOutputStream ostream = null;

		try {
			ostream = new FileOutputStream(BASE + F_CAMPS);
			ObjectOutputStream p = new ObjectOutputStream(ostream);
			p.writeObject(campaigns);
			p.flush();
		} catch (Exception e) {
			throw new AngError(e.getMessage());
		} finally {
			try { ostream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * get the local List of Customers from disk
	 * @return the ArrayList of Customers we found
	 * @exception AngError thrown when an app error occurs
	 */
	public final ArrayList getCustomers() 
		throws AngError {

		FileInputStream istream = null;
		ArrayList custList = null;

		try {
			istream = new FileInputStream(BASE + F_CUSTS);
			ObjectInputStream p = new ObjectInputStream(istream);
			custList = (ArrayList)p.readObject();
		} catch (FileNotFoundException f) {
			System.out.println("creating " + BASE + F_CUSTS);
			custList = new ArrayList();
			saveCustomers(custList);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}
		return custList;
	}

	/**
	 * sava a list of Customers to local disk
	 * @param custs the ArrayList we are saving
	 * @exception AngError thrown when an app error occurs
	 */
	public final void saveCustomers(ArrayList custs) 
		throws AngError {

		FileOutputStream ostream = null;

		try {
			ostream = new FileOutputStream(BASE + F_CUSTS);
			ObjectOutputStream p = new ObjectOutputStream(ostream);
			p.writeObject(custs);
			p.flush();
		} catch (Exception e) {
			throw new AngError(e.getMessage());
		} finally {
			try { ostream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * get the ArrayList of 'deletes' from local disk, add the
	 * delete info, and persist to disk the 'deletes'
	 * @param objType class name of object that got deleted
	 * @param pk primary key of object that got deleted
	 * @exception AngError thrown when an app error occurs
	 */
	public final void writeDelete(DeleteInfo deleteInfo)
		throws AngError {

		FileInputStream istream = null;
		ArrayList deleteList = null;

		try {
			//open file
			istream = new FileInputStream(BASE + F_DELETE);
			ObjectInputStream p = new ObjectInputStream(istream);
			deleteList = (ArrayList)p.readObject();

			//add delete info to list
			deleteList.add(deleteInfo);

			//save deletes back to disk
			saveDeletes(deleteList);

		} catch (FileNotFoundException f) {
			System.out.println("creating " + BASE + F_DELETE);
			deleteList = new ArrayList();
			deleteList.add(deleteInfo);
			saveDeletes(deleteList);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}
	}

	/**
	 * save back to disk the list of deletes
	 * @param deletes the list of deletes we want to save
	 */
	public void saveDeletes(ArrayList deletes) 
		throws AngError {

		FileOutputStream ostream = null;
		try {
			ostream = new FileOutputStream(BASE + F_DELETE);
			ObjectOutputStream p = new ObjectOutputStream(ostream);
			p.writeObject(deletes);
			p.flush();
		} catch (Exception e) {
			throw new AngError(e.getMessage());
		} finally {
			try { ostream.close(); } catch (Exception x) { }
		}
	}

	public final ArrayList getDeletes() 
		throws AngError {

		FileInputStream istream = null;
		ArrayList deletes = null;

		try {
			istream = new FileInputStream(BASE + F_DELETE);
			ObjectInputStream p = new ObjectInputStream(istream);
			deletes = (ArrayList)p.readObject();
		} catch (FileNotFoundException f) {
			deletes = new ArrayList();
			saveDeletes(deletes);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}
		return deletes;
	}

	/**
	 * get the local List of State tax codes from disk
	 * @return the ArrayList of StateTax we found
	 * @exception AngError thrown when an app error occurs
	 */
	public final ArrayList getStateTax() 
		throws AngError {

		FileInputStream istream = null;
		ArrayList taxList = null;

		try {
			istream = new FileInputStream(BASE + F_TAX);
			ObjectInputStream p = new ObjectInputStream(istream);
			taxList = (ArrayList)p.readObject();
		} catch (FileNotFoundException f) {
			System.out.println("creating " + BASE + F_TAX);
			taxList = new ArrayList();
			saveStateTax(taxList);
		} catch (Exception e) {
			e.printStackTrace();
			throw new AngError(e.getMessage());
		} finally {
			try { istream.close(); } catch (Exception x) { }
		}
		return taxList;
	}

	/**
	 * sava a list of StateTax to local disk
	 * @param tax the ArrayList we are saving
	 * @exception AngError thrown when an app error occurs
	 */
	public final void saveStateTax(ArrayList tax) 
		throws AngError {

		FileOutputStream ostream = null;

		try {
			ostream = new FileOutputStream(BASE + F_TAX);
			ObjectOutputStream p = new ObjectOutputStream(ostream);
			p.writeObject(tax);
			p.flush();
		} catch (Exception e) {
			throw new AngError(e.getMessage());
		} finally {
			try { ostream.close(); } catch (Exception x) { }
		}
	}
}
