package sellwin.server;

import sellwin.domain.*;
import java.rmi.*;
import java.util.*;

// SellWin http://sourceforge.net/projects/sellwincrm
//Contact support@open-app.com for commercial help with SellWin
//This software is provided "AS IS", without a warranty of any kind.

/**
 * This interface is implemented by the BizServices class for
 * remote services offered to clients.  The LocalServerImpl class
 * implements this class to offer the same services on the
 * local user's laptop when operating in 'disconnected' mode.
 * 
 * By having this shared interface, we can view services as
 * remote or local.
 */
public interface ServerInterface extends java.rmi.Remote {

	/**
	 * get US tax table 
	 * @param lastSyncDate a user's last sync date which limits
	 * the query, null means there is no limit
	 * @return the ArrayList of StateTax objects
	 */
	 public ArrayList getStateTax(java.util.Date lastSyncDate)
	 	throws RemoteException, AngError;

	/**
	 * load a StateTax row
	 * @param row the StateTax to load
	 */
	public void loadTax(StateTax row) 
		throws RemoteException, AngError;

	/**
	 * update a StateTax row
	 * @param row the StateTax to update
	 */
	public void updateTax(StateTax row)
		throws RemoteException, AngError;

	/**
	 * get a StateTax row
	 * @param stateCode the state code
	 */
	public StateTax getTax(String stateCode)
		throws RemoteException, AngError;

	/**
	 * delete all StateTax rows
	 */
	public void deleteAllTax() 
		throws RemoteException, AngError;


	/**
	 * get an index of Opportunities for a given user
	 * @param u the SalesPerson we are searching for
	 * @return the ArrayList of OppIndex objects that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getOpportunityIndex(SalesPerson u)
		throws RemoteException, AngError;

	/**
	 * get the list of Products that are offered 
	 * @return the ArrayList of Products found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductMatrix()
		throws RemoteException, AngError;

	/**
	 * get the list of Product Groups
	 * @return the ArrayList of product groups that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductGroups()
		throws RemoteException, AngError;

	/**
	 * get the list of product lines 
	 * @param group search using this passed product group as key
	 * @return the ArrayList of product lines that was found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductLines(String group)
		throws RemoteException, AngError;

	/**
	 * get the products for a product line and group
	 * @param group the product group to search in
	 * @param line the product line to search in
	 * @return the ArrayList of products found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProductsForLine(String group, String line)
		throws RemoteException, AngError;

	/**
	 * get a product 
	 * @param group product group to search in
	 * @param line product line to search in
	 * @param name product name to search for
	 * @return the Product that was found
	 * @exception AngError thrown when an app error occurs
	 */
	public Product getProduct(String group, String line, String name)
		throws RemoteException, AngError;

	/**
	 * get a product by its primary key
	 * @param pk the primary key
	 * @return the Product if found, null if not found
	 */
	public Product getProduct(long pk)
		throws RemoteException, AngError;

	/**
	 * update a product
	 * @param product the Product to update
	 */
	public void updateProduct(Product product) 
		throws RemoteException, AngError;

	/**
	 * add a forecast
	 * @param opportunityPK the containing Opportunity primary key
	 * @param a the Forecast we are adding
	 * @return the primary key of the new Forecast
	 * @exception AngError thrown when an app error occurs
	 */
	public long addForecast(long opportunityPK, Forecast a)
		throws RemoteException, AngError;

	/**
	 * update a Forecast 
	 * @param oppPK the containing Opportunity primary key
	 * @param a the Forecast we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateForecast(long oppPK, Forecast a)
		throws RemoteException, AngError;

	/**
	 * delete a Forecast 
	 * @param opportunityPK the containing Opportunity primary key
	 * @param forecastPK the Forecast primary key we delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteForecast(long opportunityPK, long forecastPK)
		throws RemoteException, AngError;

	/**
	 * add an Order 
	 * @param oppPK the containing Opportunity primary key
	 * @param o the Order we are adding
	 * @return the primary key of the new Order
	 * @exception AngError thrown when an app error occurs
	 */
	public long addOrder(long oppPK, Order o)
		throws RemoteException, AngError;

	/**
	 * update an Order
	 * @param oppPK the containing Opportunity primary key
	 * @param o the Order we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateOrder(long oppPK, Order o)
		throws RemoteException, AngError;

	/**
	 * add a Quote 
	 * @param oppPK the containing Opportunity primary key
	 * @param q the Quote being added
	 * @return the primary key of the new Quote
	 * @exception AngError thrown when an app error occurs
	 */
	public long addQuote(long oppPK, Quote q)
		throws RemoteException, AngError;

	/**
	 * update a Quote
	 * @param oppPK the containing Opportunity primary key
	 * @param q the Quote being updated
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateQuote(long oppPK, Quote q)
		throws RemoteException, AngError;

	/**
	 * 
	 * @param name description
	 * @return description
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteQuote(long opportuntityPK, long quotePK)
		throws RemoteException, AngError;

	/**
	 * add a QuoteLine 
	 * @param oppPK the containing Opportunity primary key
	 * @param quotePK the containing Quote primary key
	 * @param a the QuoteLine being added
	 * @return the primary key of the newly added QuoteLine
	 * @exception AngError thrown when an app error occurs
	 */
	public long addQuoteLine(long oppPK, long quotePK, QuoteLine a) 
		throws RemoteException, AngError;

	/**
	 * 
	 * @param name description
	 * @return description
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteQuoteLine(long opportuntityPK, long quotePK, long linePK)
		throws RemoteException, AngError;

	/**
	 * get the Alarms for a user 
	 * @param salesPersonPK the primary key of the user to search for
	 * @return the ArrayList of Alarms found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getAlarms(long salesPersonPK)
		throws RemoteException, AngError;
		
	/**
	 * add an Activity
	 * @param opportunityPK the containing Opportunity primary key
	 * @param a the Activity we are adding
	 * @return the newly added Activity primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addActivity(long opportunityPK, Activity a)
		throws RemoteException, AngError;

	/**
	 * update an Actvitity 
	 * @param opportunityPK the containing Opportunity primary key
	 * @param a the Activity being updated
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateActivity(long opportunityPK, Activity a)
		throws RemoteException, AngError;

	/**
	 * delete an Activity 
	 * @param opportunityPK the containing Opportunity primary key
	 * @param activityPK the Activity primary key we delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteActivity(long opportunityPK, long activityPK)
		throws RemoteException, AngError;


	/**
	 * get all the user roles 
	 * @param a date to compare against in the query
	 * @return the ArrayList of found user roles
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getAllUserRoles(java.util.Date lastSyncDate) 
		throws RemoteException, AngError;

	/**
	 * add a user role
	 * @param role the UserRole we are adding to the system
	 * @exception AngError thrown when an app error occurs
	 */
	public void addUserRole(UserRole role)
		throws RemoteException, AngError;
	/**
	 * load a user role
	 * @param role the UserRole we are loading to the system
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadUserRole(UserRole role)
		throws RemoteException, AngError;

	/**
	 * delete a user role 
	 * @param pk the pk of the user role we are deleting
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteUserRole(long pk) 
		throws RemoteException, AngError;

	/**
	 * delete all user roles
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllUserRole() 
		throws RemoteException, AngError;

	/**
	 * udpate a user role 
	 * @param role the UserRole we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateUserRole(UserRole role)
		throws RemoteException, AngError;
	
	/**
	 * get a user role
	 * @param name the name of the user role we search for
	 * @return the UserRole that was found
	 * @exception AngError thrown when an app error occurs
	 */
	public UserRole getUserRole(String name)
		throws RemoteException, AngError;

	/**
	 * logon to SellWin
	 * @param id the user ID to try
	 * @param psw the password to try
	 * @return a Login object if successful
	 * @exception AngError thrown when an app error occurs
	 */
	public Login logon(String id, String psw) 
		throws RemoteException, AngError;

	/**
	 * delete a Customer  
	 * @param pk the pk of the Customer we are to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteCustomer(Customer cust)
		throws RemoteException, AngError;

	/**
	 * delete a Customer 
	 * @param name the name of the Customer we are to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteCustomer(String name)
		throws RemoteException, AngError;

	/**
	 * update a Customer 
	 * @param c the Customer we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateCustomer(Customer c)
		throws RemoteException, AngError;

	/**
	 * load a Customer 
	 * @param c the Customer we are loading
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadCustomer(Customer c)
		throws RemoteException, AngError;

	/**
	 * add a Customer 
	 * @param c the Customer we are adding
	 * @return the new Customer's primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addCustomer(Customer c)
		throws RemoteException, AngError;

	/**
	 * get a Customer 
	 * @param name the name of the Customer we are to get
	 * @return the Customer that was found
	 * @exception AngError thrown when an app error occurs
	 */
	public Customer getCustomer(String name) 
		throws RemoteException, AngError;

	/**
	 * get all the customer names in the system
	 * @return an array of the customer names
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getAllCustomerNames() 
		throws RemoteException, AngError;

	/**
	 * get all the customers in the system
	 * @param lastSyncDate a user's last sync date used to limit the
	 * query or null if no limit
	 * @return an ArrayList of Customers
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCustomers(java.util.Date lastSyncDate) 
		throws RemoteException, AngError;

	/**
	 * add a Product  
	 * @param p the Product we are adding
	 * @exception AngError thrown when an app error occurs
	 */
	public void addProduct(Product p)
		throws RemoteException, AngError;

	/**
	 * load a Product  
	 * @param p the Product we are loading
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadProduct(Product p)
		throws RemoteException, AngError;
	
	/**
	 * get all Products 
	 * @param a date to compare rows against in the query
	 * @return an ArrayList of products that was found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getProducts(java.util.Date afterDate) 
		throws RemoteException, AngError;

	/**
	 * delete product
	 * @param prod Product to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteProduct(Product prod) 
		throws RemoteException, AngError;

	/**
	 * get all the Opportunity names for a user
	 * @param u the user we are searching for
	 * @return an ArrayList of opportunity names
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getOpportunityNames(SalesPerson u)
		throws RemoteException, AngError;

	/**
	 * get all the Opportunities for a user
	 * @param u the user we are searching for
	 * @param lastSyncDate a user's last sync date to limit the query or
	 * null if no limit 
	 * @return an ArrayList of Opportunities
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getOpportunities(SalesPerson u, java.util.Date lastSyncDate)
		throws RemoteException, AngError;

	/**
	 * get an Opportunity
	 * @param pk an Opportunity primary key we are searching for
	 * @return an Opportunity
	 * @exception AngError thrown when an app error occurs
	 */
	public Opportunity getOpportunity(long pk)
		throws RemoteException, AngError;

	/**
	 * load an Opportunity 
	 * @param o the Opportunity we are loading
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadOpportunity(Opportunity o)
		throws RemoteException, AngError;

	/**
	 * add an Opportunity 
	 * @param o the Opportunity we are adding
	 * @return the primary key of the newly added object
	 * @exception AngError thrown when an app error occurs
	 */
	public long addOpportunity(Opportunity o)
		throws RemoteException, AngError;

	/**
	 * update an Opportunity
	 * @param o the Opportunity we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateOpportunity(Opportunity o)
		throws RemoteException, AngError;

	/**
	 * delete an Opportunity 
	 * @param pk the primary key of the Opportunity to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteOpportunity(long pk)
		throws RemoteException, AngError;

	/**
	 * delete a contact 
	 * @param oppPK the containing Opportunity's primary key
	 * @param contPK the Contact's primary key we delete with
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteContact(long oppPk, long contPk)
		throws RemoteException, AngError;

	/**
	 * add a contact 
	 * @param oppPK the containing Opportunity's primary key
	 * @param c the Contact we are adding
	 * @return the primary key of the new Contact
	 * @exception AngError thrown when an app error occurs
	 */
	public long addContact(long oppPk, Contact c)
		throws RemoteException, AngError;

	/**
	 * update a contact
	 * @param oppPK the containing Opportunity primary key
	 * @param c the contact we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateContact(long oppPK, Contact c)
		throws RemoteException, AngError;

	/**
	 * get a sales person 
	 * @param pk the primary key to search with
	 * @return the SalesPerson that was found
	 * @exception AngError thrown when an app error occurs
	 */
	public SalesPerson getSalesPerson(long pk) 
		throws RemoteException, AngError;

	/**
	 * update a sales person
	 * @param  s the sales person we are updating
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateSalesPerson(SalesPerson s)
		throws RemoteException, AngError;

	/**
	 * delete all sales persons
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllSalesPerson()
		throws RemoteException, AngError;

	/**
	 * delete a sales person 
	 * @param per the SalesPerson to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteSalesPerson(SalesPerson per)
		throws RemoteException, AngError;

	/**
	 * add a sales person 
	 * @param s the sales person we are adding
	 * @return the primary key of the newly added person
	 * @exception AngError thrown when an app error occurs
	 */
	public long addSalesPerson(SalesPerson s) 
		throws RemoteException, AngError;

	/**
	 * load a sales person 
	 * @param s the sales person we are loading
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadSalesPerson(SalesPerson s) 
		throws RemoteException, AngError;

	/**
	 * get all sales person logon IDs
	 * @return an ArrayList of user IDs
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getSalesPersonIDs() 
		throws RemoteException, AngError;

	/**
	 * get all sales person names 
	 * @return an ArrayList of sales person names
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getSalesPersonNames() 
		throws RemoteException, AngError;

	/**
	 *  get all sales persons
	 * @param lastSyncDate a user's last sync date, which if not null, 
	 * limits the query results
	 * @return an array of SalesPerson objects
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getSalesPersons(java.util.Date lastSyncDate) 
		throws RemoteException, AngError;


	/**
	 * get the groups a user belongs to
	 * @param userPK the user primary key to search with
	 * @return an ArrayList of UserGroup objects
	 */
	public ArrayList getGroupsForUser(long userPK)
		throws RemoteException, AngError;

	/**
	 * add a user to a user group
	 * @param userPK the user primary key to add
	 * @param group the user group to add to
	 * @exception AngError thrown when an app error occurs
	 */
	public void addUserToGroup(long userPK, UserGroup group) 
		throws RemoteException, AngError;


	 //THIS NEEDS TO BE OPTIMIZED, TO ONLY RETURN NAMES, NOT THE
	 //FULL SP
	/**
	 * get all users in a group
 	 *
	 * @param groupName the group we are searching with
	 * @return an array of SalesPersons that were found in the group
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getUsersInGroup(String groupName) 
		throws RemoteException, AngError;

	/**
	 * get a UserGroup using a user_group primary key
	 * @param userGroupPK user group primary key value
	 * @return a UserGroup if found, null if not found
	 * @exception AngError thrown when an app error occurs
	 */
	public UserGroup getUserGroup(long pk) 
		throws RemoteException, AngError;

	/**
	 * get all user groups 
	 * @param lastSyncDate a user's last sync date which limits the query
	 * @return an array of UserGroup
	 * @exception AngError thrown when an app error occurs
	 */
	public Object[] getUserGroups(java.util.Date lastSyncDate) 
		throws RemoteException, AngError;

	/**
	 * add a user group 
	 * @param g the user group to add
	 * @exception AngError thrown when an app error occurs
	 */
	public void addUserGroup(UserGroup g) 
		throws RemoteException, AngError;

	/**
	 * load a user group 
	 * @param g the user group to load
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadUserGroup(UserGroup g) 
		throws RemoteException, AngError;

	/**
	 * delete a user group 
	 * @param groupName the user group name we are to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteUserGroup(String groupName) 
		throws RemoteException, AngError;

	/**
	 * delete all user groups
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllUserGroup() 
		throws RemoteException, AngError;

	/**
	 * delete a user in a group 
	 * @param g the user group to search in
	 * @param userPK the user primary key to delete 
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteUserInGroup(UserGroup g, long userPK)
		throws RemoteException, AngError;

	/**
	 * get a Lead
	 * @param pk the Lead primary key
	 * @return the Lead that was found or null if not found
	 */
	public Lead getLead(long pk)
		throws RemoteException, AngError;

	/**
	 * update a lead 
	 * @param campaignPK the containing Campaign primary key
	 * @param l the Lead we are to update
	 * @exception AngError thrown when an app error occurs
	 */
	public void updateLead(long campaignPK, Lead l)
		throws RemoteException, AngError;

	/**
	 * get all campaigns 
	 * @param lastSyncDate a user's last sync date which limits the
	 * query or null if no limit
	 * @return an ArrayList of campaigns
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCampaigns(java.util.Date lastSyncDate)
		throws RemoteException, AngError;
	/**
	 * load campaign 
	 * @param c campaign to load
	 * @exception AngError thrown when an app error occurs
	 */
	public void loadCampaign(Campaign c)
		throws RemoteException, AngError;

	/**
	 * delete all campaigns 
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteAllCampaign()
		throws RemoteException, AngError;

	
	/**
	 * get all leads for a campaign
	 * @param campaignPK the campaign primary key to search with
	 * @return an ArrayList of campaign leads that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCampaignLeads(long campaignPK)
		throws RemoteException, AngError;

	/**
	 * get all leads
	 * @param lastSyncDate a user's last sync date used to limit
	 * the query or null if no limit
	 * @return an ArrayList of campaign leads that were found
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCampaignLeads(java.util.Date lastSyncDate)
		throws RemoteException, AngError;

	/**
	 * delete a CustomerInventory 
	 * @param ci the CustomerInventory we are to delete
	 * @exception AngError thrown when an app error occurs
	 */
	public void deleteCustomerInventory(CustomerInventory ci)
		throws RemoteException, AngError;

	/**
	 * add a CustomerInventory 
	 * @param ci the CustomerInventory we are adding
	 * @return the new CustomerInventory's primary key
	 * @exception AngError thrown when an app error occurs
	 */
	public long addCustomerInventory(CustomerInventory ci)
		throws RemoteException, AngError;

	/**
	 * get CustomerInventory  for a given customer
	 * @param custPK the Customer's PK we are searching with
	 * @return the list of CustomerInventory 
	 * @exception AngError thrown when an app error occurs
	 */
	public ArrayList getCustomerInventory(long custPK)
		throws RemoteException, AngError;


	/**
 	 * upload all the info needed about the objects that
	 * were deleted on the user's local client while
	 * operating in a disconnected mode in order to
	 * sync the remote database with the changes
	 * @param deleteList an ArrayList of deleted object
	 * info
	 */
	public void uploadDeletes(ArrayList deletes)
		throws RemoteException, AngError;


	/**
	 * get all the user group members
	 *
	 * @param lastSyncDate a user's last sync date or null, which
	 * is used to limit the query
	 * @return an ArrayList of UserGroupMembers
	 */
	public ArrayList getGroupMembers(java.util.Date lastSyncDate) 
		throws RemoteException, AngError;	

	/**
	 * get a UserGroupMember row
	 * @param pk the UserGroupMember primary key
	 * @return the UserGroupMember or null if not found
	 */
	public UserGroupMember getGroupMember(long pk)
		throws RemoteException, AngError;

	/**
	 * load a UserGroupMember row
	 * @param row the UserGroupMember row being loaded
	 */
	public void loadGroupMember(UserGroupMember row)
		throws RemoteException, AngError;
}
