/*
 *  This software module is part of the AT&T Self-Service Cloud (SSC) distribution.
 *
 *  Copyright (c) 2010-2013 by AT&T Intellectual Property. All rights reserved.
 *
 *  AT&T and the AT&T logo are trademarks of AT&T Intellectual Property.
 *
 *  Use of this software is permitted only as described by the Research
 *  Collaboration Agreement (RCA) in effect between AT&T, Rutgers, and 
 *  the organization to which this module has been delivered.  
 *  This software module may not be copied or otherwise distributed except 
 *  as described in the (RCA). 
 *
 *  This software is considered Confidential Information as described in section 7 of the RCA.
 *
 *  Information and Software Systems Research
 *  AT&T Labs
 *  Florham Park, NJ
 * 
 * Author: Jeffrey Bickford, Andres Lagar-Cavilla
 *
 * Project: Patagonix, memsharing
 * 
 * Supplies functions to  to map domU kernel memory.  
 *
*/
#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>

#include "xenctrl.h"
#include "xg_save_restore.h"
#include "xg_private.h"
	
/* Address size of the guest */
unsigned int guest_width;

/* Number of xen_pfn_t in a page */
//#define FPP             (PAGE_SIZE/(guest_width))

/* Number of entries in the pfn_to_mfn_frame_list_list */
#define P2M_FLL_ENTRIES (((p2m_size)+(FPP*FPP)-1)/(FPP*FPP))

/* number of pfns this guest has (i.e. number of entries in the P2M) */
static unsigned long p2m_size;

/* Live mapping of the table mapping each PFN to its current MFN. */
static xen_pfn_t *live_p2m = NULL;
static xen_pfn_t *my_p2m = NULL;

/* Double and single indirect references to the live P2M table */
void *live_p2m_frame_list_list = NULL;
void *live_p2m_frame_list = NULL;

/* Copies of the above. */
xen_pfn_t *p2m_frame_list_list = NULL;
xen_pfn_t *p2m_frame_list = NULL;

//#define VGA_IO_START  0xA0000UL
//#define VGA_IO_SIZE   0x20000
#define VGA_IO_START_PG 0xA0
#define VGA_IO_SIZE_PG  0x20
#define GB_TO_PGS(x)    (((unsigned long) (x)) << 18)
#define MMIO_START_PG   GB_TO_PGS(3)
#define MMIO_SIZE_PG    GB_TO_PGS(1)
static inline xen_pfn_t *identity_hvm_p2m(void)
{
    
    /* Uses the global p2m_size */
    xen_pfn_t i, *rc;
    if (!(rc = (xen_pfn_t *) malloc(sizeof(xen_pfn_t) * p2m_size)))
        return rc;
    for (i = 0; i < p2m_size; i++) {
        rc[i] = i;
    }
    for (i = VGA_IO_START_PG; i < (VGA_IO_START_PG + VGA_IO_SIZE_PG); i++)
        rc[i] = 0;
    i = MMIO_START_PG;
    while ((i < p2m_size)&&(i < (MMIO_START_PG + MMIO_SIZE_PG)))
        rc[i++] = 0;
    return rc;
}

/*
** Map the top-level page of MFNs from the guest. The guest might not have
** finished resuming from a previous restore operation, so we wait a while for
** it to update the MFN to a reasonable value.
*/
static void *map_frame_list_list(int xc_handle, uint32_t dom, unsigned int guest_width,
                                 shared_info_any_t *shinfo)
{
    int count = 100;
    void *p;
    uint64_t fll = GET_FIELD(shinfo, arch.pfn_to_mfn_frame_list_list);

    while ( count-- && (fll == 0) )
    {
        usleep(10000);
        fll = GET_FIELD(shinfo, arch.pfn_to_mfn_frame_list_list);
    }

    if ( fll == 0 )
    {
        ERROR("Timed out waiting for frame list updated.");
        return NULL;
    }

    p = xc_map_foreign_range(xc_handle, dom, PAGE_SIZE, PROT_READ, fll);
    if ( p == NULL )
        ERROR("Couldn't map p2m_frame_list_list (errno %d)", errno);

    return p;
}

static xen_pfn_t *map_p2m_table(int xc_handle, uint32_t dom, unsigned long p2m_size, unsigned int guest_width, shared_info_any_t *live_shinfo)
{
	vcpu_guest_context_any_t ctxt;
	
	/* The mapping of the live p2m table itself */
	xen_pfn_t *p2m = NULL;
	
	int i, success = 0;

	live_p2m_frame_list_list = map_frame_list_list(xc_handle, dom, guest_width,
                                                   live_shinfo);
	if ( !live_p2m_frame_list_list )
		goto out;
	
	/* Get a local copy of the live_P2M_frame_list_list */
	if ( !(p2m_frame_list_list = (xen_pfn_t*)malloc(PAGE_SIZE)) )
	{
		ERROR("Couldn't allocate p2m_frame_list_list array");
		goto out;
	}
	memcpy(p2m_frame_list_list, live_p2m_frame_list_list, PAGE_SIZE);
	
	/* Canonicalize guest's unsigned long vs ours */
	if ( guest_width > sizeof(unsigned long) )
		for ( i = 0; i < PAGE_SIZE/sizeof(unsigned long); i++ )
			if ( i < PAGE_SIZE/guest_width )
				p2m_frame_list_list[i] = ((uint64_t *)p2m_frame_list_list)[i];
			else
				p2m_frame_list_list[i] = 0;
	else if ( guest_width < sizeof(unsigned long) )
		for ( i = PAGE_SIZE/sizeof(unsigned long) - 1; i >= 0; i-- )
			p2m_frame_list_list[i] = ((uint32_t *)p2m_frame_list_list)[i];
	
	live_p2m_frame_list =
		xc_map_foreign_batch(xc_handle, dom, PROT_READ,
						p2m_frame_list_list,
						P2M_FLL_ENTRIES);
	if ( !live_p2m_frame_list )
	{
		ERROR("Couldn't map p2m_frame_list");
		goto out;
	}
	
	/* Get a local copy of the live_P2M_frame_list */
	if ( !(p2m_frame_list = (xen_pfn_t*)malloc(P2M_TOOLS_FL_SIZE)) )
	{
		ERROR("Couldn't allocate p2m_frame_list array");
		goto out;
	}
	memset(p2m_frame_list, 0, P2M_TOOLS_FL_SIZE);
	memcpy(p2m_frame_list, live_p2m_frame_list, P2M_GUEST_FL_SIZE);
	
	/* Canonicalize guest's unsigned long vs ours */
	if ( guest_width > sizeof(unsigned long) )
		for ( i = 0; i < P2M_FL_ENTRIES; i++ )
			p2m_frame_list[i] = ((uint64_t *)p2m_frame_list)[i];
	else if ( guest_width < sizeof(unsigned long) )
		for ( i = P2M_FL_ENTRIES - 1; i >= 0; i-- )
			p2m_frame_list[i] = ((uint32_t *)p2m_frame_list)[i];
	
	
	/* Map all the frames of the pfn->mfn table. For migrate to succeed,
		the guest must not change which frames are used for this purpose.
		(its not clear why it would want to change them, and we'll be OK
		from a safety POV anyhow. */
	
	p2m = (xen_pfn_t*)xc_map_foreign_batch(xc_handle, dom, PROT_READ,
							p2m_frame_list,
							P2M_FL_ENTRIES);
	if ( !p2m )
	{
		ERROR("Couldn't map p2m table");
		goto out;
	}
	live_p2m = p2m; /* So that translation macros will work */


	success = 1;

out:

	if ( !success && p2m )
		munmap(p2m, P2M_FLL_ENTRIES * PAGE_SIZE);

	if ( live_p2m_frame_list_list )
		munmap(live_p2m_frame_list_list, PAGE_SIZE);
	
	if ( live_p2m_frame_list )
		munmap(live_p2m_frame_list, P2M_FLL_ENTRIES * PAGE_SIZE);
	
	if ( p2m_frame_list_list ) 
		free(p2m_frame_list_list);
	
	if ( p2m_frame_list ) 
		free(p2m_frame_list);
	
	return success ? p2m : NULL;
}

static inline xen_pfn_t * __get_p2m_table_pv(int xc_handle, uint32_t dom)
{
	xc_dominfo_t info;
	xen_pfn_t *p2m;
	/* Live mapping of shared info structure */
	shared_info_any_t *live_shinfo = NULL;
	/* The new domain's shared-info frame number. */
	unsigned long shared_info_frame;
	/* max mfn of the whole machine */
	static unsigned long max_mfn;
	
	/* virtual starting address of the hypervisor */
	static unsigned long hvirt_start;
	
	/* #levels of page tables used by the current guest */
	static unsigned int pt_levels;

	if ( !get_platform_info(xc_handle, dom,
					&max_mfn, &hvirt_start, &pt_levels, &guest_width) )
	{
		fprintf(stderr,"Unable to get platform info.");
		exit(-1);
	}

	if ( xc_domain_getinfo(xc_handle, dom, 1, &info) != 1 )
	{
		fprintf(stderr,"Could not get domain info");
		exit(-1);
	}

	shared_info_frame = info.shared_info_frame;

	/* Map the shared info frame */
	live_shinfo = (shared_info_any_t *)xc_map_foreign_range(xc_handle, dom, PAGE_SIZE,
								PROT_READ, shared_info_frame);
	if ( !live_shinfo )
	{
		fprintf(stderr,"Couldn't map live_shinfo");
		exit(-1);
	}

	/* Get the size of the P2M table */
	p2m_size = xc_memory_op(xc_handle, XENMEM_maximum_gpfn, &dom) + 1;
	printf("p2m_size: 0x%x\n",p2m_size);

	/* Map the P2M table, and write the list of P2M frames */
	p2m = map_p2m_table(xc_handle, dom, p2m_size, guest_width, live_shinfo);
	if ( p2m == NULL )
	{
		fprintf(stderr,"Failed to map the p2m frame list\n");
		exit(-1);
	}

    munmap(live_shinfo, PAGE_SIZE);
	return p2m;
}

static inline xen_pfn_t * __get_p2m_table_hvm(int xc_handle, uint32_t dom)
{
	p2m_size = xc_memory_op(xc_handle, XENMEM_maximum_gpfn, &dom) + 1;
    return identity_hvm_p2m();
}

xen_pfn_t * get_p2m_table(int xc_handle, uint32_t dom, int hvm)
{
    return ((hvm) ?
            __get_p2m_table_hvm(xc_handle, dom) :
            __get_p2m_table_pv(xc_handle, dom));
}

static inline unsigned char * __build_memory_map_pv(uint32_t domain, 
                            xen_pfn_t **res_p2m, unsigned long *num_pfns)
{
	int xc_handle;
	xc_dominfo_t info;
	uint32_t dom;
	int i;
	unsigned char *memory;
	dom = domain;

	xc_handle = xc_interface_open();
	
	if(xc_handle == -1)
	{
		fprintf(stderr, "Unable to get handle to hypervisor\n");
		exit(-1);
	}

	live_p2m = __get_p2m_table_pv(xc_handle, dom);

	/* Copy live_p2m to my_p2m, pass my_p2m to mmap*/

	my_p2m = (xen_pfn_t *)malloc(p2m_size * sizeof(xen_pfn_t));
	memcpy(my_p2m,live_p2m,p2m_size * sizeof(xen_pfn_t));

 	memory = (unsigned char *)xc_map_foreign_batch(xc_handle, dom, PROT_READ,
                           my_p2m, p2m_size );

	if(memory == NULL)
	{
		fprintf(stderr, "Failed to map memory\n");
		exit(-1);
	}

	/* unmap p2m table */
	if ( live_p2m )
		munmap(live_p2m, P2M_FLL_ENTRIES * PAGE_SIZE);
    if ((!res_p2m) && (my_p2m)) free(my_p2m);
    if ((res_p2m) && (my_p2m)) *res_p2m = my_p2m;
    if (num_pfns) *num_pfns = p2m_size;

	xc_interface_close(xc_handle);
    
	return memory;
}


unsigned char *__build_memory_map_hvm(uint32_t domain, xen_pfn_t **res_p2m,
                                        unsigned long *num_pfns)
{
	int xc_handle;
    xen_pfn_t *p2m;
    char *memory;

	xc_handle = xc_interface_open();
	if(xc_handle == -1)
	{
		fprintf(stderr, "Unable to get handle to hypervisor\n");
		exit(-1);
	}	

    p2m_size = xc_memory_op(xc_handle, XENMEM_maximum_gpfn, &domain) + 1;
    p2m = identity_hvm_p2m();
    if (!p2m) {
		fprintf(stderr, "Unable to get identity p2m\n");
		exit(-1);
    }

 	memory = (unsigned char *)xc_map_foreign_batch(xc_handle, domain, PROT_READ,
                           p2m, p2m_size );
    if ((!res_p2m) && (p2m)) free(p2m);
    if ((res_p2m) && (p2m)) *res_p2m = p2m;
    if (num_pfns) *num_pfns = p2m_size;
	xc_interface_close(xc_handle);
    return memory;
}

unsigned char * build_memory_map(uint32_t domain, int hvm, 
                                xen_pfn_t **p2m, unsigned long *num_pfns)
{
    return ((hvm) ? 
            __build_memory_map_hvm(domain, p2m, num_pfns) :
            __build_memory_map_pv(domain, p2m, num_pfns));
}

