/******************************************************************************
 *  This software module is part of the AT&T Self-Service Cloud (SSC) distribution.
 *
 *  Copyright (c) 2010-2013 by AT&T Intellectual Property. All rights reserved.
 *
 *  AT&T and the AT&T logo are trademarks of AT&T Intellectual Property.
 *
 *  Use of this software is permitted only as described by the Research
 *  Collaboration Agreement (RCA) in effect between AT&T, Rutgers, and 
 *  the organization to which this module has been delivered.  
 *  This software module may not be copied or otherwise distributed except 
 *  as described in the (RCA). 
 *
 *  This software is considered Confidential Information as described in section 7 of the RCA.
 *
 *  Information and Software Systems Research
 *  AT&T Labs
 *  Florham Park, NJ
 * 
 * memsharing.c                                                               *
 *                                                                            *
 * Author: Andres Lagar Cavilla                                               *
 *                                                                            *
 * Purpose: Pauses two domains, and attempts to share as many pages as        *
 * possible based on content fingerprints.                                    *
 *                                                                            *    
 * Usage: ./memsharing <domid 1> <domid 2>                                    *
 ******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/mman.h>
#include <gcrypt.h>
#include <time.h>
#include "xenctrl.h"
#include <errno.h>

#define INVALID_P2M_ENTRY   ((xen_pfn_t)-1)
#define fprintf(...)
unsigned char *build_memory_map(uint32_t domain, int is_hvm, 
                            xen_pfn_t **p2m, unsigned long *num_pfns);
xen_pfn_t *get_p2m_table(int xc_handle, uint32_t dom, int is_hvm);

static xen_pfn_t *live_p2m_1, *live_p2m_2, *res_p2m_1, *res_p2m_2;
unsigned long p2m_size_1, p2m_size_2;	
unsigned char *memory_map_1, *memory_map_2;

int xc_handle, dom1, dom2, dom1_hvm, dom2_hvm;
FILE *logfile;
int hash_len;

struct hash_entry {
    unsigned long pfn;
    unsigned long mfn;
    unsigned char *hash;
    char *hash_str;
    struct hash_entry *next;
    char shared;
};

#define HSTZ 2048
struct hash_entry *hash_table[HSTZ];

#define PAIRS_SZ 2048

/* Get the number of pages of a domain and print it */
int output_nr_pages(int xc_handle, int dom)
{
    xc_dominfo_t info;
    int rc = xc_domain_getinfo(xc_handle, (uint32_t) dom, 1, &info);
    if (rc != 1) return rc;
    fprintf(logfile, "Dom %d has %lu pages\n", dom, info.nr_pages);
    return 0;
}

int is_hvm(int xc_handle, int dom)
{
    xc_dominfo_t info;
    int rc = xc_domain_getinfo(xc_handle, (uint32_t) dom, 1, &info);
    if (rc != 1) return -1;
    return !!info.hvm;
}

/* hash must be of hash_len 
     Length of resulting sha1 hash - gcry_md_get_algo_dlen
     returns digest lenght for an algo 
    int hash_len = gcry_md_get_algo_dlen( GCRY_MD_SHA1 ); */
char *hash_page(unsigned char *data, unsigned char *hash)
{
    /* output sha1 hash - converted to hex representation
     * 2 hex digits for every byte + 1 for trailing \0 */
    char *out = (char *) malloc( sizeof(char) * ((hash_len*2)+1) );
    char *p = out;

    /* calculate the SHA1 digest. This is a bit of a shortcut function
     * most gcrypt operations require the creation of a handle, etc. */
    gcry_md_hash_buffer( GCRY_MD_SHA1, hash, data, PAGE_SIZE );

    /* Convert each byte to its 2 digit ascii
     * hex representation and place in out */
    int i;
    for ( i = 0; i < hash_len; i++, p += 2 ) {
        snprintf ( p, 3, "%02x", hash[i] );
    }

   return out;
}

int add_hash(unsigned long pfn, unsigned long mfn, char *page)
{
    unsigned long idx;
    struct hash_entry *entry = (struct hash_entry *) 
                            malloc(sizeof(struct hash_entry));
    if (!entry) return -ENOMEM;

    entry->pfn = pfn;
    entry->mfn = mfn;
    entry->shared = 0;
    entry->hash = (unsigned char *) malloc(
                    sizeof(unsigned char) * hash_len);
    if (!entry->hash) {
        free(entry);
        return -ENOMEM;
    }
    entry->hash_str = hash_page(page, entry->hash);

    idx = ((unsigned long *) entry->hash)[0] % HSTZ;
    entry->next = hash_table[idx];
    hash_table[idx] = entry;

    return 0;
}

struct hash_entry *find_match(char *page)
{
    unsigned long idx;
    char *hash_str;
    struct hash_entry *entry, *second_chance_entry = NULL;
    unsigned char *hash = (unsigned char *) malloc(
                        sizeof(unsigned char) * hash_len);
    if (!hash) {
        fprintf(logfile, "ENOMEM when finding match\n");
        return NULL;
    }
    hash_str = hash_page(page, hash);
    if (!hash_str) {
        fprintf(logfile, "ENOMEM 2 when finding match\n");
        free(hash);
        return NULL;
    }
    free(hash_str);

    idx = ((unsigned long *) hash)[0] % HSTZ;
    entry = hash_table[idx];

    while (entry) {
        if (!memcmp(hash, entry->hash, sizeof(unsigned char) * hash_len)) {
            /* Found! */
            if (!entry->shared) break;
            /* This entry is shared, remember just in case */
            second_chance_entry = entry;
        }
        entry = entry->next;
    }

    free(hash);
    if ((!entry)&&(second_chance_entry)) entry = second_chance_entry;
    return entry;
}

void my_handler(int s) {
	printf("caught signal %d\n",s);
	fclose(logfile);
	exit(1);
}

#define READY
#undef READY
#ifdef READY
void do_sharing(int xc_handle, int dom1, int dom2, 
                uint32_t count, uint32_t *pairs, int hvm)
{
    int rc;
    uint32_t global;

    if (hvm) {
        /* Do it */
        if ((rc = xc_share_mfns(xc_handle, (uint32_t) dom1, 
                        (uint32_t) dom2, count, pairs, &global))) {
            fprintf(logfile, "Sharing call failed rc %d count %u\n", 
                                rc, count);
            exit(-1);
        };
    }
    fprintf(logfile, "Shared %u mfns, global count now %u\n", 
                count, global);
}
#else
void do_sharing(int xc_handle, int dom1, int dom2, 
                uint32_t count, uint32_t *pairs, int hvm)
{
    fprintf(logfile, "Called sharing on %u mfns\n", count); 
}
#endif


//#define VGA_IO_START  0xA0000UL
//#define VGA_IO_SIZE   0x20000
#define VGA_IO_START_PG 0xA0
#define VGA_IO_SIZE_PG  0x20
#define GB_TO_PGS(x)    (((unsigned long) (x)) << 18)
#define MMIO_START_PG   GB_TO_PGS(3)
#define MMIO_SIZE_PG    GB_TO_PGS(1)
static inline unsigned long get_mfn(unsigned long pfn, xen_pfn_t *live_p2m, 
                                    xen_pfn_t *res_p2m, unsigned long p2m_size)
{
    unsigned long mfn1, mfn2;
    if (pfn >= p2m_size) return 0;
    if ((pfn >= VGA_IO_START_PG) && (pfn < (VGA_IO_START_PG + VGA_IO_SIZE_PG))) 
        return 0; 
    if ((pfn >= MMIO_START_PG) && (pfn < (MMIO_START_PG + MMIO_SIZE_PG))) 
        return 0; 
    mfn1 = live_p2m[pfn];
    mfn2 = res_p2m[pfn];
    if ((mfn1 == INVALID_P2M_ENTRY) || (mfn1 & 0xF0000000) ||
        (mfn2 == INVALID_P2M_ENTRY) || (mfn2 & 0xF0000000)) {
        return 0;
    }
    return mfn1;
}

int main(int argc, char **argv) 
{
	int frc,errno,rc;
    unsigned long i;
    uint32_t pairs[2 * PAIRS_SZ];
    uint32_t count, total = 0;

    if(argc != 4) {
        printf("usage: %s logfile id1 id2\n",argv[0]);
        return 0;
    }

    hash_len = gcry_md_get_algo_dlen( GCRY_MD_SHA1 ); 

	signal(SIGINT,my_handler);

	dom1 = atoi(argv[2]);
    dom2 = atoi(argv[3]);

	live_p2m_1 = NULL;
	live_p2m_2 = NULL;
    memory_map_1 = NULL;
    memory_map_2 = NULL;

    memset(hash_table, 0, sizeof(struct hash_entry *) * HSTZ);

	logfile = fopen(argv[1],"w");
	
	if(logfile == NULL) {
		fprintf(stderr, "Can't open log file.\n");
		exit(1);
	}	
	
	xc_handle = xc_interface_open();

	if(xc_handle == -1) {
		fprintf(stderr, "Unable to get handle to hypervisor\n");
		exit(-1);
	}
	

    /* Not doing this right now as we don't support page breaking atm */
#if 0
	/* Live suspend. Enable log-dirty mode. */
	if ( xc_shadow_control(xc_handle, dom,
							XEN_DOMCTL_SHADOW_OP_ENABLE_LOGDIRTY,
							NULL, 0, NULL, 0, NULL) < 0 )
	{
		/* log-dirty already enabled? There's no test op,
		 * so attempt to disable then reenable it */
		frc = xc_shadow_control(xc_handle, dom, 
					XEN_DOMCTL_SHADOW_OP_OFF,
					NULL, 0, NULL, 0, NULL);
		if ( frc >= 0 )
		{
			frc = xc_shadow_control(xc_handle, dom,
									XEN_DOMCTL_SHADOW_OP_ENABLE_LOGDIRTY,
									NULL, 0, NULL, 0, NULL);
		}

		if ( frc < 0 )
		{
			fprintf(stderr,"Couldn't enable shadow mode (rc %d) (errno %d)", frc, errno );
			exit(-1);
		}
	}
#endif
	
	fprintf(logfile,"start memsharing\n");	
    output_nr_pages(xc_handle, dom1);
    output_nr_pages(xc_handle, dom2);
    dom1_hvm = is_hvm(xc_handle, dom1);
    dom2_hvm = is_hvm(xc_handle, dom2);
// Cuurently only working with PV-VM for Eurosys due to time constraints
/*
    if ((dom1_hvm < 0)||(dom2_hvm < 0)) {
        fprintf(logfile, "Not HVM dom %d %d %d %d\n", dom1, dom1_hvm, dom2, dom2_hvm);
        exit(-1);
    }
    if (dom1_hvm != dom2_hvm) {
        fprintf(logfile, "Can't do haphazard sharing %d %d %d %d\n", dom1, dom1_hvm, dom2, dom2_hvm);
        exit(-1);
    }
*/
    fprintf(logfile, "Dom %d HVM %d\n", dom1, dom1_hvm);
    fprintf(logfile, "Dom %d HVM %d\n", dom2, dom2_hvm);

    xc_domain_pause(xc_handle, dom1);
	memory_map_1 = (unsigned char *)build_memory_map(dom1, dom1_hvm, &res_p2m_1, &p2m_size_1);
	live_p2m_1 = (xen_pfn_t *) get_p2m_table(xc_handle, dom1, dom1_hvm);
    if ((!memory_map_1) || (!live_p2m_1)) {
        fprintf(logfile, "Unable to map dom 1 %d\n", dom1);
        exit(-1);
    }

    xc_domain_pause(xc_handle, dom2);
	memory_map_2 = (unsigned char *)build_memory_map(dom2, dom2_hvm, &res_p2m_2, &p2m_size_2);
	live_p2m_2 = (xen_pfn_t *) get_p2m_table(xc_handle, dom2, dom2_hvm);
    if ((!memory_map_2) || (!live_p2m_2)) {
        fprintf(logfile, "Unable to map dom 2 %d\n", dom2);
        exit(-1);
    }

    /* Scan the whole banana, domain 1 */
    for (i = 0; i < p2m_size_1; i++) {
        char *page = memory_map_1 + (PAGE_SIZE * i);
        unsigned long mfn = get_mfn(i, live_p2m_1, res_p2m_1, p2m_size_1);
        if (mfn == 0) continue;
        if (add_hash(i, mfn, page)) {
            fprintf(logfile, "GAAA %lu %lu\n", i, mfn);
            exit(-1);
        }
    }

    (void)munmap(memory_map_1, p2m_size_1 * PAGE_SIZE);

    /* Scan domain 2, and find matches */
    memset(pairs, 0, sizeof(uint32_t) * 2 * PAIRS_SZ);
    count = 0;
    for (i = 0; i < p2m_size_2; i++) {
        struct hash_entry *entry;
        char *page = memory_map_2 + (PAGE_SIZE * i);
        unsigned long mfn = get_mfn(i, live_p2m_1, res_p2m_1, p2m_size_1);
        if (mfn == 0) continue;
        if ((entry = find_match(page))) {
            fprintf(logfile, "Found match dom 1 %d pfn %lu mfn %lu dom2 %d pfn %lu mfn %lu hash %s count %u\n",
                            dom1, entry->pfn, entry->mfn, dom2, i, mfn, entry->hash_str, count);
            /* The second item in the pair is for dom1, which we have unmapped */
            pairs[count * 2] = (uint32_t) mfn;
            pairs[(count * 2) + 1] = (uint32_t) entry->mfn;
            count++;
            total++;
            entry->shared = 1;
            if (count == PAIRS_SZ) {
                /* Note dom1, which we have unmapped, goes second. Its pages don't 
                 * have extra refs due to the map, and therefore can be freed
                 * during the domctl */
                do_sharing(xc_handle, dom2, dom1, count, pairs, dom1_hvm);
                memset(pairs, 0, sizeof(uint32_t) * 2 * PAIRS_SZ);
                count = 0;
            }
        }
    }

    if (count) {
        /* Last call */
        do_sharing(xc_handle, dom2, dom1, count, pairs, dom1_hvm);
    }

    fprintf(logfile, "Done, total of %u matches found\n", total);
    output_nr_pages(xc_handle, dom1);
    output_nr_pages(xc_handle, dom2);

    return 0;
}

