#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ucontext.h>
#include <signal.h>
#include <sys/time.h>
#include "thread.h"

int first = 1;
int currid = 2;
int count = 0;
int size = 0;
int condid = 1;
int inlibrary = 0;
int hassignaled = 0;
int addtofinished = 0;
char mainstack[SIGSTKSZ];
node *current = NULL;
thread_t main_thread;
thread_list *threads;
thread_list *finished = NULL;
ucontext_t scheduler,main_loop;

/*
 * Performs a context switch from the current context to the
 * next thread in the threads list.
 */
void nextcontext(){
	node *last;

	last = current;
	current = current->next;

	/* if the current thread was finished add it to finished list   */
	/* we do this here so we did not lose the current->next pointer */
	if(addtofinished == 1){
		last->next = finished->head;
		finished->head = last;
		finished->size++;
		addtofinished = 0;
	}
      
	if(current != NULL){

		swapcontext(&last->thread->context,&current->thread->context);	
	}
}

/*
 * Frees all nodes that are in the finished thread_list.  The threads in
 * this list have called thread_exit() and are in the state FINISHED.
 */
int garbage_collect(){
	node *prev;

	while(finished->size > 0){
		prev = finished->head;
		finished->head = finished->head->next;

		prev = NULL;
		finished->size--;
		free(prev);
	}

	finished->head = NULL;
}

/*
 * Function for the sigaction handler.  Every time the timer invokes this
 * function count is incremented.  When the count reaches a certain number
 * we switch to the next context.  If the current context is in a library
 * function we do not switch to the next context but set a flag to 1.
 */
static void timer_func(){

	count++;

	if(count == 3){
		count = 0;

		if(finished->head != NULL)
			garbage_collect();

		if(inlibrary == 0){
			//printf("IN LIBRARY 0 CONTEXT SWITCH\n");
			nextcontext();
		}
		else{
		//	printf("IN LIBRARY 1\n");
			hassignaled = 1;
		}
	}
}

/*
 * Sets flag to 1 so we know the current thread is in our thread library.
 */
void enter_library(){
	//printf("ENTER LIB\n");
	inlibrary = 1;
}

/*
 * Sets flag to 0 so we know the current thread is no longer in the
 * thread library.  If the timer signal went off during the time the thread
 * was in the library a context switch was postponed.  In this case we set
 * the corresponding flag to 0 and switch to the next thread.
 */
void leave_library(){
//	printf("LEAVE LIB\n");
	inlibrary = 0;


	if(hassignaled == 1){
		hassignaled = 0;
	//	printf("HAS SIGNALED CONTEXT SWITCH\n");
		nextcontext();
	}
}

/*
 * Initializes the timer for the preemptive scheduler.
 * The timer is currently set to signal every 100 msec.
 */
int init_timer(){

	struct sigaction sa;
	struct itimerval timer;

	memset(&sa,0,sizeof(sa));  // this removed "Alarm clock" echo problem
	sa.sa_handler = timer_func;

	if (sigaction(SIGALRM, &sa, NULL) == -1){
		printf("SIGALARM ERROR\n");
	}

	/* timer 10 msec */

	timer.it_interval.tv_sec = 0;
	timer.it_interval.tv_usec = 10000;

	timer.it_value.tv_sec = 0;
	timer.it_value.tv_usec = 10000;

	setitimer(ITIMER_REAL,&timer,NULL);

}

/*
 * Takes in a thread structure and a thread_list structure.
 * This function creates a new node for the thread and adds it
 * to the end of the thread_list.
 */
void sched_thread(thread_t *thread, thread_list *queue){
	node *newnode = (node*)malloc(sizeof(node));
	node *prev;
	newnode->thread= thread;
	//printf("IN SCHED THREAD 2\n");
	if(queue->head == NULL){
		//printf("HEAD IS NULL\n");
		queue->head = newnode;
		//printf("NEWNODE NEXT\n");
		queue->tail = newnode;
		//printf("NEWNODE NEXT\n");
		newnode->next = queue->head;
	}else{
		//printf("ELSE CLAUSE\n");
		node *ptr = queue->head;
		while(ptr->next != queue->head){
			prev = ptr;
			ptr = ptr->next;
		}

		ptr->next = newnode;
		queue->tail = newnode;
		newnode->next = queue->head;
	}

	//printf("AFTER\n");
	queue->size++;
}

/*
 * Takes in a thread structure.  This is a stub function which is called by
 * every thread that is made in thread_create().  The stub calls the actual
 * function passed into thread_create().  After the function is finished if
 * thread thread did not properly exit we call thread_exit().
 */
void *stub_function(thread_t *thread){

	void *(*function)(void*) = thread->func;
	function(thread->arg);

	
	if(thread->state != FINISHED)
		thread_exit();
	


}

/*
 * Creates a new thread.  The first time this is called the global thread_list
 * structs are allocated for 20 threads.  We also create a node for the main
 * context and add this to the thread_list that is used for the READY threads.
 * All other times including the first, the function initializes a context for
 * the thread pointer and adds the thread to the ready queue.
 */
int thread_create(thread_t *thread, thread_attr_t *attr, void *(*start_routine)(void*), void *arg){

	enter_library();
	printf("CREATE CALLED\n");
	int threadid;
	
	//printf("CREATE THREAD\n");

	if(first == 1){
	//	printf("ALLOCATE THREADS\n");
		threads = (thread_list *)malloc(sizeof(thread_list)*20);
		finished = (thread_list *)malloc(sizeof(thread_list)*20);
	//	mutex_blocked = (thread_list *)malloc(sizeof(thread_list)*20);
		//cond_blocked = (thread_list *)malloc(sizeof(thread_list)*20);
		
	//	memset(threads,0,sizeof(threads);
		
		//threads->head = NULL;
		//threads = NULL;
	}
	
	//printf("GET CONTEXT\n");
	/* create thread context */
	getcontext(&thread->context);
	thread->context.uc_stack.ss_sp = thread->stack;
	thread->context.uc_stack.ss_size = SIGSTKSZ;



	thread->func = start_routine;
	thread->arg = arg;

	//printf("MAKE CONTEXT\n");
	makecontext(&thread->context,stub_function,1,thread);
	//printf("AFTER MAKE\n");
	thread->tid = currid; // set id of thread
	threadid = currid;
	currid++; 		// increase id counter by 1
	//thread->state = READY;
	//printf("BEFORE SCHED\n");
	sched_thread(thread, threads);
	//printf("AFTER SCHED\n");


	if(first == 1){
		first = 0;
		//printf("IN FIRST\n");

		main_thread.context = main_loop;

		main_thread.context.uc_stack.ss_sp = mainstack;
		main_thread.context.uc_stack.ss_size = SIGSTKSZ;
		main_thread.tid = 1;
		sched_thread(&main_thread, threads);

		current = threads->head;

		init_timer();
		leave_library();
		swapcontext(&main_thread.context,&current->thread->context);
	}

	if(first == 0)
		leave_library();

	return 0;
}

/*
 * Removes the current thread from the thread_list argument.
 */
removenode(thread_list *queue){
	node *ptr;

	ptr = queue->head;

	if(ptr == current){
		if(queue->size> 1){
			queue->head = queue->head->next;
			queue->tail->next = queue->head;
		}
		else if(queue->size == 1){
			current->next = NULL;
		}

		queue->size--;
		return;
	}

	while(ptr->next != current){
		ptr = ptr->next;
	}

	ptr->next = current->next;
	queue->size--;

	if(current->thread->state == FINISHED){

		if(finished->head == NULL){
			finished->head = current;
			finished->size++;
		}
		else
			addtofinished = 1;
	}

}

/*
 * Used for the thread_list structures in the mutex and condition variables.
 * This function removes the head of the thread_list passed in.  If the size
 * is zero after we remove the head, we set the head to NULL.
 */
void removehead(thread_list *queue){

	if(queue->head != NULL){
		queue->tail->next = queue->head->next;
		queue->head = queue->head->next;
	}

	queue->size--;

	if(queue->size == 0){
		queue->head = NULL;
	}

}

/*
 * Thread exit removes the current node from the threads list and calls
 * nextcontext().  This will switch to the next context in the list.
 */
void thread_exit(){
	enter_library();

	current->thread->state = FINISHED;
	
	removenode(threads);

	leave_library();
	nextcontext();
}

/*
 * Calls nextcontext().  This will save the current thread context
 * and switch to the next thread in the scheduler threads list.
 */
void thread_yield(){
	enter_library();
	nextcontext();
	leave_library();
}

int thread_cond_init(thread_cond_t *cond, const thread_condattr_t *attr){
	enter_library();
	cond->id = condid;
	condid++;
	cond->signal = 0;
	cond->cond_blocked = (thread_list *)malloc(sizeof(thread_list)*20);
	leave_library();
	return 0;
}

int thread_cond_broadcast(thread_cond_t *cond){
	enter_library();

	cond->signal = 1;

	/* if there are threads waiting, wake up all threads */
	while(cond->cond_blocked->size > 0){
		sched_thread(cond->cond_blocked->head->thread,threads);
		removehead(cond->cond_blocked);
	}

	leave_library();
	return 0;
}

int thread_cond_signal(thread_cond_t *cond){
	enter_library();
	cond->signal = 1;

	/* if there are threads waiting, wake up one thread */
	if(cond->cond_blocked->size > 0){
		sched_thread(cond->cond_blocked->head->thread,threads);
		removehead(cond->cond_blocked);
	}

	leave_library();
	return 0;
}

int thread_cond_wait(thread_cond_t *cond, thread_mutex_t *mutex){
	enter_library();

	/* release lock */
	if(mutex->lock == 1){
		thread_mutex_unlock(mutex);
	}

	/* block the calling thread */
	removenode(threads);
	current->thread->state=BLOCKED;
	sched_thread(current->thread, cond->cond_blocked);
	//printf("SIZE OF COND BLOCKED: %d\n",cond->cond_blocked);

	while(cond->signal == 0){
		nextcontext();
	}

	thread_mutex_lock(mutex);

	leave_library();
	return 0;
}



/*
 * Since we do not have to worry about attribute objects the only thing
 * we have to do is initialize the lock to unlocked.
 */
int thread_mutex_init(thread_mutex_t *mutex, const thread_mutexattr_t *attr){
	enter_library();
	mutex->lock = 0;
	mutex->mutex_blocked = (thread_list *)malloc(sizeof(thread_list)*20);
	mutex->mutex_blocked->head = NULL;
	leave_library();
	return 0;
}

/*
 * Locks the mutex.  While the lock is held we block the current thread
 * and yield to the next thread in the scheduler.  When the lock is no
 * longer held we change the lock to 1 and the current thread now holds
 * the lock.  Blocked threads are added to the blocked thread_list struct
 * located in the mutex structure;
 */
int thread_mutex_lock(thread_mutex_t *mutex){
	enter_library();

	while(mutex->lock == 1){
		removenode(threads);
		current->thread->state=BLOCKED;
		sched_thread(current->thread, mutex->mutex_blocked);

		nextcontext();
	}

	mutex->lock = 1;
	leave_library();
	return 0;
}

/*
 * Unlock the lock.  If there are blocked threads in the blocked
 * thread list then we remove one and schedule it.
 */
int thread_mutex_unlock(thread_mutex_t *mutex){
	enter_library();
	mutex->lock = 0;

	if(mutex->mutex_blocked->size > 0){
		sched_thread(mutex->mutex_blocked->head->thread,threads);
		removehead(mutex->mutex_blocked);
	}

	leave_library();
	return 0;
}
