/**
 * threadpool.c
 *
 * This file will contain your implementation of a threadpool.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <sched.h>
#include "threadpool.h"

#define USE_MY_PTHREADS 0
// 0 = use standard pthreads, 1 = use my threads package

#if USE_MY_PTHREADS
#define PTHREAD_T               thread_t
#define PTHREAD_CREATE          thread_create
#define PTHREAD_EXIT            thread_exit
#define SCHED_YIELD             thread_yield
#define PTHREAD_COND_INIT       thread_cond_init
#define PTHREAD_COND_BROADCAST  thread_cond_broadcast
#define PTHREAD_COND_SIGNAL     thread_cond_signal
#define PTHREAD_COND_WAIT       thread_cond_wait
#define PTHREAD_MUTEX_INIT      thread_mutex_init
#define PTHREAD_MUTEX_LOCK      thread_mutex_lock
#define PTHREAD_MUTEX_UNLOCK    thread_mutex_unlock
#else
#define PTHREAD_T               pthread_t
#define PTHREAD_CREATE          pthread_create
#define PTHREAD_EXIT            pthread_exit
#define SCHED_YIELD             pthread_yield
#define PTHREAD_COND_INIT       pthread_cond_init
#define PTHREAD_COND_BROADCAST  pthread_cond_broadcast
#define PTHREAD_COND_SIGNAL     pthread_cond_signal
#define PTHREAD_COND_WAIT       pthread_cond_wait
#define PTHREAD_MUTEX_INIT      pthread_mutex_init
#define PTHREAD_MUTEX_LOCK      pthread_mutex_lock
#define PTHREAD_MUTEX_UNLOCK    pthread_mutex_unlock
#endif 
// _threadpool is the internal threadpool structure that is
// cast to type "threadpool" before it given out to callers

int block = 0;

typedef struct work_st {
  dispatch_fn fn;               // the function pointed to
  void* args;                   // the arguments
  struct work_st * next;        // the next piece of work in the internal q
} work;

typedef struct _threadpool_st {
  // you should fill in this structure with whatever you need
  PTHREAD_T * threads;          // the collection of threads
  int size, qsize, killswitch;  // size of pool, size of q, and 
  // flag for threads to die
  work * q;                     // pointer to head of internal work q
  work * qtail;                 // pointer to tail of internal work q  
  pthread_mutex_t qmut;         // internal q mutex
  pthread_cond_t qe, qne;       // conditions for empty, non-empty q
} _threadpool;

static void *do_work(void *vprt_args) {
  // since it's a hassle to initialize a thread, just keep all the original
  // ones and let them loop through work as it comes in.
  work * current; //the piece of work we're operating on
  _threadpool * pool = (_threadpool*) vprt_args; 
  while (1) {                           // loop forever
    if (pool->killswitch) {             // if the killswitch is set,
      //      PTHREAD_EXIT(NULL);               // die.
    }
    PTHREAD_MUTEX_LOCK(&(pool->qmut));  // grab the q lock.
    while (pool->qsize==0) {            // wait for work
      if (pool->killswitch) {           // since this is where a thread
        //        PTHREAD_EXIT(NULL);             // waiting for work will be,
      }                                 // check on wakeup if it should die.
      PTHREAD_MUTEX_UNLOCK(&(pool->qmut));              // release lock
      block++;
     // printf("BLOCKED: %d\n",block);
      PTHREAD_COND_WAIT(&(pool->qne),&(pool->qmut));    // wait until work,
    }                                                   // q are both avail
    current = pool->q;          // operating on top element of q
    pool->qsize--;              // decrement q size since we're popping
    if (pool->qsize==0) {       // if the q is empty, 
      pool->q = NULL;           // wipe the pointers.
      pool->qtail = NULL;       
      PTHREAD_COND_SIGNAL(&(pool->qe)); // let someone know there's no work.
    }
    else {
      pool->q = current->next; // if there's an heir apparent, business as
    }                          // usual.
    PTHREAD_MUTEX_UNLOCK(&(pool->qmut)); // release the q lock
    current->fn(current->args);          // call the function we've been 
    free(current);                       // given, then clean up.
  }
  return NULL; //never reached
}

threadpool create_threadpool(int num_threads_in_pool) {
  _threadpool *pool;

  // sanity check the argument
  if ((num_threads_in_pool <= 0) || (num_threads_in_pool > MAXT_IN_POOL))
    return NULL;

  pool = (_threadpool *) malloc(sizeof(_threadpool)); //allocate the space
  if (pool == NULL) {
    fprintf(stderr, "Out of memory creating a new threadpool!\n");
    return NULL;
  }
  PTHREAD_MUTEX_INIT(&(pool->qmut),NULL); // initialize mutex and conditions
  PTHREAD_COND_INIT(&(pool->qe),NULL);
  PTHREAD_COND_INIT(&(pool->qne),NULL);

  pool->size = num_threads_in_pool; // set pool size
  pool->qsize=pool->killswitch=0;   // q is empty and we're not killed yet
  pool->q=pool->qtail=NULL;         // and we have no work to do.
  pool->threads = (PTHREAD_T  *) malloc(sizeof(PTHREAD_T) * pool->size);
  //allocate space for as many threads as we need
  int i;
  for (i = 0; i < pool->size; i++) { // send each thread to work
    if (PTHREAD_CREATE(&(pool->threads[i]), NULL, do_work, (void *) pool)!= 0) {
      fprintf(stderr, "Error creating new thread");
      return NULL;
    }
  }

  return (threadpool) pool;
}


void dispatch(threadpool from_me, dispatch_fn dispatch_to_here,
    void *arg) {
  // add your code here to dispatch a thread
  _threadpool *pool = (_threadpool *) from_me;
  if (pool->killswitch) { // sorry, we're closed for today. come back later.
    return;
  } else {
    work *w = (work *)malloc(sizeof(work)); // else create a work object
    w->fn = dispatch_to_here;               // fill in the fields
    w->args=arg;                            // pass the args alone
    w->next=NULL;                           // back of the line, buddy.
    PTHREAD_MUTEX_LOCK(&(pool->qmut));      // grab the q mutex
    while (pool->size == pool->qsize) {     // if we don't have space,
      PTHREAD_MUTEX_UNLOCK(&(pool->qmut));  // release the mutex,
      SCHED_YIELD();                        // wait for a thread to finish,
      PTHREAD_MUTEX_LOCK(&(pool->qmut));    // and then try again.
    }
    if(pool->q==NULL) {                     // if the q is empty to start,
      pool->q=pool->qtail=w;                // make sure we know it's there
      PTHREAD_COND_SIGNAL(&(pool->qne));    // then tell everyone else.
    } else {
      pool->qtail->next=w;                  // otherwise, business as usual.
      pool->qtail=w;
    }
    (pool->qsize)++;                        // increment the amount of work.
    PTHREAD_COND_SIGNAL(&(pool->qe));
    PTHREAD_MUTEX_UNLOCK(&(pool->qmut));    // we're done with the q now.
  }
}

void destroy_threadpool(threadpool destroyme) {
  _threadpool *pool = (_threadpool *) destroyme;
  // add your code here to kill a threadpool
  pool->killswitch=1;       // closing time: kill threads; block q
  while (pool->q != NULL) { // while there's still work left,
    SCHED_YIELD();          // let other threads finish since we can't add 
    // to the q
  }
  int i;
  for (i=0; i<pool->size; i++) { // give every thread an exit command.
    dispatch(pool, (void *)(&PTHREAD_EXIT), NULL);
  }
  while (pool->qsize>0) {
    SCHED_YIELD();
  }
  free(pool->threads);                  
  pthread_mutex_destroy(&(pool->qmut)); // then clean up the rest of the 
  pthread_cond_destroy(&(pool->qe));    // objects.
  pthread_cond_destroy(&(pool->qne));
  return;
}
