/******************************************************************************
 *  This software module is part of the AT&T Self-Service Cloud (SSC) distribution.
 *
 *  Copyright (c) 2010-2013 by AT&T Intellectual Property. All rights reserved.
 *
 *  AT&T and the AT&T logo are trademarks of AT&T Intellectual Property.
 *
 *  Use of this software is permitted only as described by the Research
 *  Collaboration Agreement (RCA) in effect between AT&T, Rutgers, and 
 *  the organization to which this module has been delivered.  
 *  This software module may not be copied or otherwise distributed except 
 *  as described in the (RCA). 
 *
 *  This software is considered Confidential Information as described in section 7 of the RCA.
 *
 *  Information and Software Systems Research
 *  AT&T Labs
 *  Florham Park, NJ
 * kernel.c -- program to dump kernel code to a database                      *
 *                                                                            *
 * Author: Jeffrey Bickford                                                   *
 *                                                                            *
 * Purpose: Hashes all kernel code pages and dumps them to a database.        * 
 *                                                                            *    
 * Usage: ./kernel <domid>                                                    *
 ******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/mman.h>
#include <gcrypt.h>
#include <sqlite3.h>
#include <time.h>
extern "C" {
#include <xenctrl.h>
}
using namespace std;


unsigned char *build_memory_map(uint32_t domain);
xen_pfn_t *get_p2m_table(int xc_handle, uint32_t dom);
xen_pfn_t *get_m2p_table(int xc_handle, uint32_t dom);

int xc_handle, dom;
sqlite3 *db, *kernel_db;
static xen_pfn_t *live_p2m;
static xen_pfn_t *live_m2p;
unsigned long p2m_size;	
unsigned char *memory_map;

char *hash_page(unsigned char *data)
{
    //unsigned int PAGE_SIZE = 0x1000;
    /* Length of resulting sha1 hash - gcry_md_get_algo_dlen
     * returns digest lenght for an algo */
    int hash_len = gcry_md_get_algo_dlen( GCRY_MD_SHA1 );

    /* output sha1 hash - this will be binary data */
    unsigned char hash[ hash_len ];

    /* output sha1 hash - converted to hex representation
     * 2 hex digits for every byte + 1 for trailing \0 */
    char *out = (char *) malloc( sizeof(char) * ((hash_len*2)+1) );
    char *p = out;

    /* calculate the SHA1 digest. This is a bit of a shortcut function
     * most gcrypt operations require the creation of a handle, etc. */
    gcry_md_hash_buffer( GCRY_MD_SHA1, hash, data, PAGE_SIZE );

    /* Convert each byte to its 2 digit ascii
     * hex representation and place in out */
    int i;
    for ( i = 0; i < hash_len; i++, p += 2 ) {
        snprintf ( p, 3, "%02x", hash[i] );
    }

   return out;
   //return *hash;
}

int main(int argc, char **argv) 
{

	int frc,errno,rc;
	int count = 0;
	char *zErrMsg = 0;

	dom = atoi(argv[1]);

	live_p2m = NULL;
	live_m2p = NULL;
	
	rc = sqlite3_open("/root/patagonix/database/kernel.db",&kernel_db);

	if(rc) {
		fprintf(stderr, "Can't open database: %s",sqlite3_errmsg(kernel_db));
		sqlite3_close(kernel_db);
		exit(1);
	}

	xc_handle = xc_interface_open();

	if(xc_handle == -1) {
		fprintf(stderr, "Unable to get handle to hypervisor\n");
		exit(-1);
	}
	
	memory_map = (unsigned char *)build_memory_map(dom);
	live_p2m = (xen_pfn_t *) get_p2m_table(xc_handle, dom);
	live_m2p = (xen_pfn_t *) get_m2p_table(xc_handle, dom);

	p2m_size = xc_memory_op(xc_handle, XENMEM_maximum_gpfn, &dom) + 1;

	unsigned long addr;

	for(addr = 0xc0100000; addr < 0xc0373000; addr = addr + 0x1000)
	{
		unsigned long pfn = (addr - 0xc0000000) >> 12;
		char *current_hash = hash_page(memory_map + (pfn * 0x1000));

		char sqlbuf[0x1000];
		memset(sqlbuf,0,0x1000);
		sprintf(sqlbuf,"INSERT INTO hashes VALUES('0x%lx','%s');",addr,current_hash);
		
		rc = sqlite3_exec(kernel_db,sqlbuf, NULL, 0, &zErrMsg);
		if(rc != SQLITE_OK)
		{
			fprintf(stderr, "SQL error: %s\n", zErrMsg);
			sqlite3_free(zErrMsg);
		}

		printf("0x%x: %s\n",addr,current_hash);
		count++;
	}

	
	for(addr = 0xc03bd000; addr < 0xc03d600; addr = addr + 0x1000)
	{
		unsigned long pfn = (addr - 0xc0000000) >> 12;
		char *current_hash = hash_page(memory_map + (pfn * 0x1000));
		
		char sqlbuf[0x1000];
		memset(sqlbuf,0,0x1000);
		sprintf(sqlbuf,"INSERT INTO hashes VALUES('0x%lx','%s');",addr,current_hash);
		
		rc = sqlite3_exec(kernel_db,sqlbuf, NULL, 0, &zErrMsg);
		if(rc != SQLITE_OK)
		{
			fprintf(stderr, "SQL error: %s\n", zErrMsg);
			sqlite3_free(zErrMsg);
		}

		printf("0x%x: %s\n",addr,current_hash);
		count++;
	}

	printf("count: %d\n",count);		
						
}

