/******************************************************************************
 *  This software module is part of the AT&T Self-Service Cloud (SSC) distribution.
 *
 *  Copyright (c) 2010-2013 by AT&T Intellectual Property. All rights reserved.
 *
 *  AT&T and the AT&T logo are trademarks of AT&T Intellectual Property.
 *
 *  Use of this software is permitted only as described by the Research
 *  Collaboration Agreement (RCA) in effect between AT&T, Rutgers, and 
 *  the organization to which this module has been delivered.  
 *  This software module may not be copied or otherwise distributed except 
 *  as described in the (RCA). 
 *
 *  This software is considered Confidential Information as described in section 7 of the RCA.
 *
 *  Information and Software Systems Research
 *  AT&T Labs
 *  Florham Park, NJ
 *                                                                            *
 * Author: Shakeel Butt                                                       *
 *                                                                            *
 * Purpose: Checks code pages of executables against known code in a          *
 *          database.                                                         * 
 *                                                                            *    
 * Usage: ./syscall_monitor <domid>                                           *
 ******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/mman.h>
#include <gcrypt.h>
#include <sqlite3.h>
#include <utility>
#include <map>
#include <time.h>
#include <vector>
extern "C" {
#include "xenctrl.h"
}
using namespace std;


void *syscall_monitor_handle;
int xc_handle, dom;
struct syscall_monitor_info *info;
uint64_t syscall_monitor_info_frame;

/////////////syscall_monitor.h
#define MAX_SYSCALL_MONITOR_INFO_ENTRIES 50
   
struct syscall_monitor_entry {
    uint64_t rax;
    uint64_t rbx;
    uint64_t rcx;
    uint64_t rdx;    
    uint64_t rsi;
    uint64_t rdi;    
};  
    
typedef struct syscall_monitor_entry syscall_monitor_entry_t;
    
struct syscall_monitor_info {
    unsigned long count;
    struct syscall_monitor_entry entries[MAX_SYSCALL_MONITOR_INFO_ENTRIES];
};  

typedef struct syscall_monitor_info syscall_monitor_info_t;
/////////////////////////////



void my_handler(int s) {
	printf("caught signal %d\n",s);
	xc_syscall_monitor_destroy(&syscall_monitor_handle);
	exit(1);
}


int is_domain_paused(int xc_handle,int dom){

	xc_dominfo_t info;

	if(xc_domain_getinfo(xc_handle,dom,1,&info)!=1) {
		printf("Unable to get info\n");
		exit(-1);
	}

	return info.paused;
}

int main(int argc, char **argv) 
{

	int frc,errno,rc;
    char syscall_page_info[4096];

    if(argc != 2) {
        printf("usage: %s dom_id\n",argv[0]);
        return 0;
    }
	signal(SIGINT,my_handler);

	dom = atoi(argv[1]);

	xc_handle = xc_interface_open();

	if(xc_handle == -1) {
		fprintf(stderr, "Unable to get handle to hypervisor\n");
		exit(-1);
	}
	
	if(xc_syscall_monitor_get_shared_page(xc_handle,dom,&syscall_monitor_info_frame) < 0) {
		fprintf(stderr, "Unable to get syscall_monitor shared page\n");
		exit(-1);
	}

	info = (struct syscall_monitor_info *)xc_map_foreign_range(xc_handle, dom,
						0x1000, 
						PROT_READ | PROT_WRITE,
						syscall_monitor_info_frame);


	syscall_monitor_handle = (void *)xc_syscall_monitor_prepare(xc_handle,dom);
	
	if(syscall_monitor_handle == NULL)
		printf("Unable to get syscall_monitor handle\n");

	int start = xc_syscall_monitor_start(syscall_monitor_handle);
	if (start < 0)
		printf("Unable to start syscall_monitor\n");

	fprintf(stdout,"start syscall_monitor\n");	

    while(is_domain_paused(xc_handle,dom));

	while(1)
	{
		xc_syscall_monitor_wait(syscall_monitor_handle);

		xc_domain_pause(xc_handle, dom);
		
		if(info->count){	
			memcpy(syscall_page_info,info, 4096 );	
		}	
		
		//printf("count = %d\n", info->count);
		info->count = 0;

		xc_domain_unpause(xc_handle,dom);
	}	
}

