------------------------------- MODULE SnipSnap -----------------------------
EXTENDS Naturals, Integers, Sequences, FiniteSets, TLC

VARIABLES mem, memInt, ctl, buf, cache, cow, memQ, snapCtl, snapIdx, memt0, memCopied
CONSTANT Proc, Adr, Val, QLen, CacheSets, CacheWays

ASSUME (QLen \in Nat) /\ (QLen > 0)
\* Cache Size must be even and greater than 1
ASSUME (CacheSets \in Nat) /\ (CacheSets >= 1)
ASSUME (CacheWays \in Nat) /\ (CacheWays >= 1) \*CacheWays must be power of 2
-----------------------------------------------------------------------------
(* memory requests *)
MReq == [op : {"Rd"}, adr: Adr]
          \cup [op : {"Wr"}, adr: Adr, val : Val]

NoVal == CHOOSE v : v \notin Val

Init == /\ mem = CHOOSE m \in [Adr->Val]: TRUE
        /\ ctl = [p \in Proc |-> "rdy"]
        /\ buf = [p \in Proc |-> NoVal]
        /\ cache = [row \in (0..CacheSets-1) |-> [a \in Adr |-> NoVal]]
        /\ cow = [row \in (0..CacheSets-1) |-> [a \in Adr |-> NoVal]]
        /\ memQ = << >>
        /\ memInt \in {<<CHOOSE p \in Proc : TRUE, NoVal>>}
        /\ snapCtl = -1
        /\ snapIdx = 0
        /\ memt0 = NoVal
        /\ memCopied = NoVal

TypeInvariant == 
  /\ mem     \in [Adr  -> Val]
  /\ ctl     \in [Proc -> {"rdy", "busy", "waiting", "done"}] 
  /\ buf     \in [Proc -> MReq \cup Val \cup {NoVal}]
  /\ cache   \in [(0..CacheSets-1) -> [Adr -> Val \cup {NoVal}]]
  /\ cow     \in [(0..CacheSets-1) -> [Adr -> Val \cup {NoVal}]]
  /\ memQ    \in Seq(MReq)
  /\ snapCtl \in {-1, 0, 1}
  /\ snapIdx \in Adr \cup {NoVal}
  /\ memt0   \in [Adr->Val] \cup {NoVal}
  /\ memCopied \in [Adr->Val] \cup {NoVal}

-----------------------------------------------------------------------------
\* total cache size
CacheSize == CacheSets * CacheWays

\* cache usage
CacheRowUsage(r) == Cardinality({a \in Adr : cache[r][a] # NoVal})
RecCUsage[r \in Int] == IF r = -1 THEN 0
                                  ELSE CacheRowUsage(r) + RecCUsage[r-1]
CacheUsage == RecCUsage[CacheSets-1]

\* CoW usage
CoWRowUsage(r) == Cardinality({a \in Adr : cow[r][a] # NoVal})
RecCoWUsage[r \in Int] == IF r = -1 THEN 0
                                    ELSE CoWRowUsage(r) + RecCoWUsage[r-1]
CoWUsage == RecCoWUsage[CacheSets-1]
-----------------------------------------------------------------------------
\* safety checks

\* limit the number of valid entries in the cache & CoW area
CacheProp == /\ CacheUsage + CoWUsage <= CacheSize
             /\ \A r \in (0..CacheSets-1): /\ CacheRowUsage(r) <= CacheWays
                                           /\ CoWRowUsage(r) <= CacheWays

SnapshotConsistency ==
    (snapCtl = 1 /\ snapIdx = NoVal /\ memt0 # NoVal) => memt0 = memCopied

-----------------------------------------------------------------------------
\* processor <--> memory system interface
Req(p) == /\ ctl[p] = "rdy"
          \* when snapCtl equals -1, we are not in snapshot mode
          \* when snapCtl equals 0, we are in transition to snapshot mode
          \* when snalCtl equals 1, we are in snapshot mode
          \* we can't receive/process requests while transitioning from
          \* standard mode to snapshot mode.
          /\ snapCtl # 0
          /\ \E req \in  MReq :
               /\ memInt' = <<p, req>> \* Send request to the mem. interface
               /\ buf' = [buf EXCEPT ![p] = req]
               /\ ctl' = [ctl EXCEPT ![p] = "busy"]
          /\ UNCHANGED <<mem, memQ, cache, cow, snapCtl, snapIdx, memt0, memCopied>>

Rsp(p) == /\ ctl[p] = "done"
          /\ memInt' = <<p, buf[p]>> \* Request recv'ed
          /\ ctl' = [ctl EXCEPT ![p]= "rdy"]
          /\ UNCHANGED <<mem, memQ, cache, cow, buf, snapCtl, snapIdx, memt0, memCopied>>
-----------------------------------------------------------------------------
(* read miss - req not found in the cache, send to memory. We don't *)
(* look at the CoW area after a Read req, it may contain stale data *)
RdMiss(p) ==
    LET row == IF buf[p] # NoVal THEN (buf[p].adr % CacheSets) ELSE -1
	\* CSize is the cache size based on the current mode the system is
        CSize == IF snapCtl <= 0 THEN CacheSize ELSE CacheSize \div 2
    IN  /\ (ctl[p] = "busy") /\ (buf[p].op = "Rd")
        /\ CacheUsage < CSize \* Any free CL in the cache?
        /\ CacheRowUsage(row) < CacheWays \* Any free CL in the set?
        /\ cache[row][buf[p].adr] = NoVal
        /\ Len(memQ) < QLen
        /\ memQ' = Append(memQ, buf[p])
        /\ ctl' = [ctl EXCEPT ![p] = "waiting"]
        /\ UNCHANGED <<memInt, mem, cache, cow, buf, snapCtl, snapIdx, memt0, memCopied>>

(* write-allocate write miss - we actually perform a read operation here *)
(* writing the address to memory is done in DoWr(p)                      *)
WrMissRdAlloc(p) ==
    LET row == IF buf[p] # NoVal THEN buf[p].adr % CacheSets ELSE -1
        \* CSize is the cache size based on the current mode the system is
        CSize == IF snapCtl <= 0 THEN CacheSize ELSE CacheSize \div 2
    IN  /\ (ctl[p] = "busy") /\ (buf[p].op = "Wr") 
        /\ CacheUsage < CSize \* Any free CL in the cache?
        /\ CacheRowUsage(row) < CacheWays \* Any free CL in the set?
        /\ cache[row][buf[p].adr] = NoVal
        /\ Len(memQ) < QLen
        /\ memQ' = Append(memQ, [op |-> "Rd", adr |-> buf[p].adr])
        /\ ctl' = [ctl EXCEPT ![p] = "waiting"]
        /\ UNCHANGED <<memInt, mem, cache, cow, buf, snapCtl, snapIdx, memt0, memCopied>>

Evict(adr) ==
    LET row == adr % CacheSets
    IN  \* no outstanding request for address 'adr'
        /\ \A pr \in Proc : (ctl[pr] = "waiting") => (buf[pr].adr # adr)
        /\ cache[row][adr] # NoVal
        \* write back cache[row][adr] to the memory
        /\ Len(memQ) < QLen
        /\ memQ' = Append(memQ, [op |-> "Wr", adr |-> adr, val |-> cache[row][adr]])
        \* clear the entry
        /\ cache' = [cache EXCEPT ![row][adr] = NoVal]
        /\ UNCHANGED <<memInt, mem, buf, ctl, cow, snapCtl, snapIdx, memt0, memCopied>>

(* DoRd requests always get data from the cache *)
(* RdMiss will bring data from the memory to the cache *)
DoRd(p) ==
    LET row == buf[p].adr % CacheSets
        adr == buf[p].adr
    IN  /\ ctl[p] \in {"busy", "waiting"} 
        /\ buf[p].op = "Rd" 
        /\ cache[row][adr] # NoVal
        /\ buf' = [buf EXCEPT ![p] = cache[row][adr]]
        /\ ctl' = [ctl EXCEPT ![p] = "done"]
        /\ UNCHANGED <<memInt, mem, memQ, cache, cow, snapCtl, snapIdx, memt0, memCopied>>

DoWr(p) == 
    LET r == buf[p]
        row == buf[p].adr % CacheSets
        \* CSize is the cache size based on the current mode the system is
        CSize == IF snapCtl <= 0 THEN CacheSize ELSE CacheSize \div 2
    IN  /\ ctl[p] \in {"busy", "waiting"}
        /\ CoWUsage < CSize   \* We can't write if CoW is full
        /\ CoWRowUsage(row) < CacheWays \* We can't write if CoW's set is full
        /\ (r.op = "Wr")
        /\ cache[row][r.adr] # NoVal
        /\ cow' = IF /\ snapCtl = 1
                     /\ snapIdx # NoVal
                     /\ r.adr >= snapIdx
                     /\ cow[row][r.adr] = NoVal
                    THEN [cow EXCEPT ![row][r.adr] = cache[row][r.adr]]
                    ELSE cow
        /\ cache' = IF cache[row][r.adr] # NoVal
                      THEN [cache EXCEPT ![row][r.adr] = r.val]
                      ELSE cache
        /\ buf' = [buf EXCEPT ![p] = NoVal]
        /\ ctl' = [ctl EXCEPT ![p] = "done"]
        /\ UNCHANGED <<memInt, mem, memQ, snapCtl, snapIdx, memt0, memCopied>>

(* memory/cache operation *)
(* vmem is a helper function used in MemQRd *)
vmem  ==
  LET f[i \in 0 .. Len(memQ)] == 
        IF i=0 THEN mem
               ELSE IF memQ[i].op = "Rd"
                      THEN f[i-1]
                      ELSE [f[i-1] EXCEPT ![memQ[i].adr] =
                                              memQ[i].val]
  IN f[Len(memQ)] 

MemQWr == LET r == Head(memQ)
          IN  /\ (memQ # << >>)  /\   (r.op = "Wr") 
              /\ mem' = [mem EXCEPT ![r.adr] = r.val] 
              /\ memQ' = Tail(memQ)
              /\ UNCHANGED <<memInt, buf, ctl, cache, cow, snapCtl, snapIdx, memt0, memCopied>>

MemQRd == 
  LET r == Head(memQ)
      row == r.adr % CacheSets
  IN  /\ (memQ # << >> ) /\ (r.op = "Rd")
      /\ memQ' = Tail(memQ)
      /\ cache' = [cache EXCEPT ![row][r.adr] = vmem[r.adr]]
      /\ UNCHANGED <<memInt, mem, buf, ctl, cow, snapCtl, snapIdx, memt0, memCopied>>

(* trigger the snapshot *)
StartSnapReq == /\ snapCtl = -1
                /\ snapCtl' = 0
                /\ snapIdx' = 0
                /\ UNCHANGED <<memInt, mem, memQ, buf, ctl, cache, cow, memt0, memCopied>>
                
(* wait until all outstanding mem requests are finished          *)
(* so we can flush the cache, and effectively start the snapshot *)
StartSnapAck ==
    /\ snapCtl = 0
    \* no more outstanding memory requests
    /\ \A p \in Proc : ctl[p] = "rdy"
    \* all the cache entries have been written-back
    /\ \A row \in (0..CacheSets-1) :  \A a \in Adr: cache[row][a] = NoVal
    \* flush the cache and cow
    /\ cache' = [row \in (0..CacheSets-1) |-> [a \in Adr |-> NoVal]]
    /\ cow' = [row \in (0..CacheSets-1) |-> [a \in Adr |-> NoVal]]
    \* flush the memory queue
    /\ Len(memQ) = 0
    \* switch to snapshot mode
    /\ snapCtl' = snapCtl+1
    \* make a copy of the memory to verify if our snapshot method
    /\ memt0' = mem
    /\ memCopied' = CHOOSE m \in [Adr->Val]: TRUE
    /\ UNCHANGED <<memInt, mem, memQ, buf, ctl, snapIdx>>

(* taking the snapshot - increament snapIdx as we perform the snapshot  *)
(* we also evict entries in the CoW area that have already been copied  *)
TakingSnap ==
    LET row == snapIdx % CacheSets
    IN  /\ snapCtl > 0
        /\ snapIdx # NoVal
        /\ snapIdx < Cardinality(Adr)  \* snapIdx is zero-based
        /\ memCopied' = CASE cow[row][snapIdx] # NoVal ->
                               [memCopied EXCEPT ![snapIdx] = cow[row][snapIdx]] []
                             /\ cache[row][snapIdx] # NoVal
                             /\ cow[row][snapIdx] = NoVal ->
                               [memCopied EXCEPT ![snapIdx] = cache[row][snapIdx]] []
                        OTHER ->
                               [memCopied EXCEPT ![snapIdx] = mem[snapIdx]]
        /\ snapIdx' = IF snapIdx < Cardinality(Adr)-1
                        THEN snapIdx+1
                        ELSE NoVal
        /\ cow' = [cow EXCEPT ![row][snapIdx] = NoVal]
        /\ UNCHANGED <<memInt, mem, memQ, buf, ctl, cache, snapCtl, memt0>>

Next == \/ \E p\in Proc : \/ Req(p) \/ Rsp(p)
                          \/ RdMiss(p) \/ WrMissRdAlloc(p)
                          \/ DoRd(p) \/ DoWr(p)
        \/ \E a \in Adr : Evict(a)
        \/ MemQWr \/ MemQRd
        \/ StartSnapReq \/ StartSnapAck \/ TakingSnap

Spec == 
  Init /\ [][Next]_<<memInt, mem, memQ, buf, ctl, cache, cow, snapIdx, snapCtl, memt0, memCopied>>
-----------------------------------------------------------------------------
THEOREM Spec => [](TypeInvariant /\ CacheProp /\ SnapshotConsistency)
-----------------------------------------------------------------------------
=============================================================================
